package br.pucrio.tecgraf.soma.logsmonitor.websocket;

import br.pucrio.tecgraf.soma.logsmonitor.appservice.JobLogsAppService;
import br.pucrio.tecgraf.soma.logsmonitor.manager.PublisherManager;
import br.pucrio.tecgraf.soma.logsmonitor.manager.WebSocketSessionManager;
import br.pucrio.tecgraf.soma.logsmonitor.model.Message;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.ErrorType;
import br.pucrio.tecgraf.soma.logsmonitor.model.error.InvalidMessageErrorEvent;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.ConcurrentWebSocketSessionDecorator;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.util.HashMap;
import java.util.Map;

@Component
public class SocketHandler extends TextWebSocketHandler {
  private static final Log logger = LogFactory.getLog(SocketHandler.class);

  private final Integer sendTimeLimit = 10 * 1000;
  private final Integer bufferSizeLimit = 10 * 1024 * 1024;

  @Autowired private WebSocketSessionManager webSocketSessionManager;
  @Autowired private WebSocketNotificatioErrorService webSocketErrorService;
  @Autowired private ObjectMapper objectMapper;
  @Autowired private PublisherManager publisherManager;
  @Autowired private JobLogsAppService appService;

  @Override
  public void afterConnectionEstablished(WebSocketSession session) {
    WebSocketSession concurrentSession =
        new ConcurrentWebSocketSessionDecorator(session, sendTimeLimit, bufferSizeLimit);
    webSocketSessionManager.putSession(session.getId(), concurrentSession);
  }

  @Override
  public void afterConnectionClosed(WebSocketSession session, CloseStatus status) throws Exception {
    webSocketSessionManager.removeSession(session.getId());
    publisherManager.onSessionClosed(session.getId());
    super.afterConnectionClosed(session, status);
  }

  @Override
  public void handleTextMessage(WebSocketSession session, TextMessage message) throws Exception {
    Message topicMessage = null;
    try {
      topicMessage = objectMapper.readValue(message.getPayload(), Message.class);
      WebSocketSession concurrentSession =
          new ConcurrentWebSocketSessionDecorator(session, sendTimeLimit, bufferSizeLimit);
      appService.processMessage(topicMessage, concurrentSession);

    } catch (JsonParseException e) {
      String errorMsg =
          String.format(
              "Error deserializing message from %s: invalid JSON", session.getRemoteAddress());
      String subsId = getSubscriptionId(message.getPayload());
      notifyInvalidMessageError(session.getId(), subsId, errorMsg);

    } catch (JsonMappingException | IllegalArgumentException e) {
      String errorMsg =
          String.format(
              "Error deserializing message from %s: %s: %s",
              session.getRemoteAddress(), e.getClass().getSimpleName(), e.getMessage());
      String subsId = getSubscriptionId(message.getPayload());
      notifyInvalidMessageError(session.getId(), subsId, errorMsg);
    }
  }

  private void notifyInvalidMessageError(String sessionId, String subscriptionId, String errorMsg) {
    logger.error(errorMsg);
    Map<String, Object> map = new HashMap<>();
    map.put(InvalidMessageErrorEvent.JSON_PROPERTY_EXCEPTION_MESSAGE, errorMsg);
    webSocketErrorService.onErrorNotify(
        sessionId, subscriptionId, ErrorType.INVALID_MESSAGE, "Invalid Message", map);
  }

  private String getSubscriptionId(String playload) throws JsonProcessingException {
    final JsonNode jsonNode = objectMapper.readTree(playload);
    return jsonNode.has(Message.JSON_PROPERTY_SUBSCRIPTION_ID)
        ? jsonNode.get(Message.JSON_PROPERTY_SUBSCRIPTION_ID).asText()
        : "";
  }
}
