/* Copyright (c) 2001 - 2007 TOPP - www.openplans.org. All rights reserved.
 * This code is licensed under the GPL 2.0 license, available at the root
 * application directory.
 */
package org.geoserver.security.decorators;

import org.geoserver.catalog.CoverageInfo;
import org.geoserver.catalog.CoverageStoreInfo;
import org.geoserver.catalog.DataStoreInfo;
import org.geoserver.catalog.FeatureTypeInfo;
import org.geoserver.catalog.LayerInfo;
import org.geoserver.platform.ExtensionPriority;
import org.geoserver.security.SecureCatalogImpl.WrapperPolicy;

/**
 * Creates security wrappers for the most common catalog objects
 * 
 * @author Andrea Aime - TOPP
 * 
 */
public class DefaultSecureCatalogFactory implements SecuredObjectFactory {

    public boolean canSecure(Class clazz) {
        return CoverageInfo.class.isAssignableFrom(clazz)
                || CoverageStoreInfo.class.isAssignableFrom(clazz)
                || DataStoreInfo.class.isAssignableFrom(clazz)
                || FeatureTypeInfo.class.isAssignableFrom(clazz)
                || LayerInfo.class.isAssignableFrom(clazz);
    }

    public Object secure(Object object, WrapperPolicy policy) {
        // null safe
        if (object == null)
            return null;

        Class clazz = object.getClass();
        if (CoverageInfo.class.isAssignableFrom(clazz))
            return new SecuredCoverageInfo((CoverageInfo) object, policy);
        else if (CoverageStoreInfo.class.isAssignableFrom(clazz))
            return new SecuredCoverageStoreInfo((CoverageStoreInfo) object, policy);
        else if (DataStoreInfo.class.isAssignableFrom(clazz))
            return new SecuredDataStoreInfo((DataStoreInfo) object, policy);
        else if (FeatureTypeInfo.class.isAssignableFrom(clazz))
            return new SecuredFeatureTypeInfo((FeatureTypeInfo) object, policy);
        else if (LayerInfo.class.isAssignableFrom(clazz))
            return new SecuredLayerInfo((LayerInfo) object, policy);
        else
            throw new IllegalArgumentException("Don't know how to wrap");
    }

    /**
     * Returns {@link ExtensionPriority#LOWEST} since the wrappers generated by
     * this factory
     */
    public int getPriority() {
        return ExtensionPriority.LOWEST;
    }

}
