/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 * 
 *    (C) 2005-2008, Open Source Geospatial Foundation (OSGeo)
 *    
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.renderer.style.customshape;

import java.awt.Graphics2D;
import java.awt.Shape;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.geotools.renderer.style.MarkFactory;
import org.opengis.feature.Feature;
import org.opengis.filter.expression.Expression;

/**
 * This Mark Factory represents a ShapeCreator registry. It allows users to
 * dinamically register new ShapeCreators, allowing user defined shapes to be
 * used in rendering. It defines a base syntax for custom shapes well known
 * names. All starts with the string 'customshape://'. Everything after this
 * prefix is passed as is to each ShapeCreator, and stops when the first one
 * generates a valid shape.
 * 
 * @author alexmc,fmoura
 * 
 */
public class CustomShapeMarkFactory implements MarkFactory
{
    private static final String SHAPE_PREFIX = "customshape://";

    /** The logger for the rendering module. */
    @SuppressWarnings("unused")
    private static final Logger LOGGER = org.geotools.util.logging.Logging.getLogger("org.geotools.rendering");

    private static Map<String, ShapeCreator> shapeCreatorRegistry = new HashMap<String, ShapeCreator>();

    static
    {
        // register the default shape creators;
        ShapeCreator[] defaultShapeCreators = new ShapeCreator[] {
                new VerticalLineShapeCreator(),
                new HorizontalLineShapeCreator(),
                new SlashShapeCreator(),
                new BackslashShapeCreator(),
                new DotShapeCreator(),
                new PlusShapeCreator(),
                new TimesShapeCreator(),
                new HatchShapeCreator(),
                new RectangleShapeCreator(),
                new DiamondShapeCreator() };

        for (ShapeCreator shapeCreator : defaultShapeCreators)
        {
            shapeCreatorRegistry.put(shapeCreator.getShapeName(), shapeCreator);
        }
    }

    /**
     * Registers a new ShapeCreator
     * 
     * @param shapeCreator The shape creator to be registered
     * @throws IllegalArgumentException if shapeCreator is null.
     */
    public static void registerShapeCreator(ShapeCreator shapeCreator)
    {
        if (shapeCreator == null)
            throw new IllegalArgumentException("shape creator can't be null.");

        synchronized (shapeCreatorRegistry)
        {
            shapeCreatorRegistry.put(shapeCreator.getShapeName(), shapeCreator);
        }
    }

    /**
     * Unregisters a ShapeCreator based in its name.
     * 
     * @param name The name of the shape creator to be unregistered
     * @throws IllegalArgumentException if name is null.
     */
    public static void unregisterShapeCreator(String name)
    {
        if (name == null)
            throw new IllegalArgumentException("shape creator name can't be null.");

        synchronized (shapeCreatorRegistry)
        {
            shapeCreatorRegistry.remove(name);
        }
    }

    /**
     * Returns the WellKnownName for the given shapeSpecification.
     * 
     * @param shapeSpecification A ShapeSpecification
     * @return
     */
    public static String buildWellKnownName(String shapeSpecification)
    {
        if (shapeSpecification == null)
            throw new IllegalArgumentException("shape specification can't be null.");
        return SHAPE_PREFIX + shapeSpecification;
    }

    public Shape getShape(Graphics2D graphics, Expression symbolUrl, Feature feature) throws Exception
    {
        // cannot handle a null url
        if (symbolUrl == null)
            return null;

        // see if it's a customshape://
        String wellKnownName = symbolUrl.evaluate(feature, String.class);
        if (!wellKnownName.startsWith(SHAPE_PREFIX))
            return null;

        String shapeSpecification = wellKnownName.substring(SHAPE_PREFIX.length());

        Shape shape = null;
        for (ShapeCreator creator : shapeCreatorRegistry.values())
        {
            shape = creator.createShape(shapeSpecification);
            if (shape != null)
                break;
        }

        return shape;
    }
}
