/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.graal.python.util;

import com.oracle.graal.python.builtins.modules.MathModuleBuiltins;
import com.oracle.graal.python.nodes.ErrorMessages;
import com.oracle.graal.python.nodes.PRaiseNode;
import com.oracle.graal.python.runtime.exception.PythonErrorType;
import com.oracle.graal.python.util.OverflowException;
import com.oracle.graal.python.util.PythonUtils;
import com.oracle.truffle.api.CompilerAsserts;
import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.memory.ByteArraySupport;
import com.oracle.truffle.api.nodes.Node;
import java.math.BigInteger;

public final class NumericSupport {
    private static final long NEG_ZERO_RAWBITS = Double.doubleToRawLongBits(-0.0);
    private static final double EPSILON = 1.0E-17;
    private final ByteArraySupport support;
    private final boolean bigEndian;
    private static final NumericSupport BE_NUM_SUPPORT = new NumericSupport(true);
    private static final NumericSupport LE_NUM_SUPPORT = new NumericSupport(false);

    private NumericSupport(boolean bigEndian) {
        this.support = bigEndian ? ByteArraySupport.bigEndian() : ByteArraySupport.littleEndian();
        this.bigEndian = bigEndian;
    }

    public static NumericSupport bigEndian() {
        return BE_NUM_SUPPORT;
    }

    public static NumericSupport littleEndian() {
        return LE_NUM_SUPPORT;
    }

    private static void reverse(byte[] buffer) {
        NumericSupport.reverse(buffer, 0, buffer.length);
    }

    private static void reverse(byte[] buffer, int offset, int numBytes) {
        CompilerAsserts.partialEvaluationConstant((int)numBytes);
        assert (offset + numBytes <= buffer.length) : "cannot reverse byte array, offset + numBytes exceeds byte array length";
        for (int i = 0; i < numBytes / 2; ++i) {
            int a = i + offset;
            int b = numBytes - i - 1 + offset;
            byte tmp = buffer[a];
            buffer[a] = buffer[b];
            buffer[b] = tmp;
        }
    }

    public static boolean equalsApprox(double a, double b) {
        return Math.abs(a - b) < 1.0E-17;
    }

    @CompilerDirectives.TruffleBoundary
    static short floatToShortBits(double value, Node raisingNode) {
        short bits;
        int e;
        int sign;
        double signum = Math.signum(value);
        if (value == 0.0) {
            sign = Double.doubleToRawLongBits(value) == NEG_ZERO_RAWBITS ? 1 : 0;
            e = 0;
            bits = 0;
        } else if (Double.isInfinite(value)) {
            sign = signum == -1.0 ? 1 : 0;
            e = 31;
            bits = 0;
        } else if (Double.isNaN(value)) {
            sign = signum == -1.0 ? 1 : 0;
            e = 31;
            bits = 512;
        } else {
            sign = value < 0.0 ? 1 : 0;
            double v = sign == 1 ? -value : value;
            double[] fraction = MathModuleBuiltins.FrexpNode.frexp(v);
            double f = fraction[0];
            e = (int)fraction[1];
            if (f < 0.5 || f >= 1.0) {
                throw PRaiseNode.raiseUncached(raisingNode, PythonErrorType.SystemError, ErrorMessages.RES_O_O_RANGE, "frexp()");
            }
            f *= 2.0;
            if (--e >= 16) {
                throw PRaiseNode.raiseUncached(raisingNode, PythonErrorType.OverflowError, ErrorMessages.FLOAT_TO_LARGE_TO_PACK_WITH_S_FMT, "e");
            }
            if (e < -25) {
                f = 0.0;
                e = 0;
            } else if (e < -14) {
                f = Math.scalb(f, 14 + e);
                e = 0;
            } else {
                e += 15;
                f -= 1.0;
            }
            bits = (short)(f *= 1024.0);
            assert (bits < 1024);
            assert (e < 31);
            if ((f - (double)bits > 0.5 || NumericSupport.equalsApprox(f - (double)bits, 0.5) && (bits & 1) != 0) && (bits = (short)(bits + 1)) == 1024) {
                bits = 0;
                if (++e == 31) {
                    throw PRaiseNode.raiseUncached(raisingNode, PythonErrorType.OverflowError, ErrorMessages.FLOAT_TO_LARGE_TO_PACK_WITH_S_FMT, "e");
                }
            }
        }
        bits = (short)(bits | (short)(e << 10 | sign << 15));
        return bits;
    }

    @CompilerDirectives.TruffleBoundary
    static float shortBitsToFloat(short bits) {
        int sign = (bits & 0x8000) >> 15;
        int e = (bits & 0x7C00) >> 10;
        int f = bits & 0x3FF;
        if (e == 31) {
            if (f == 0) {
                return sign == 1 ? Float.NEGATIVE_INFINITY : Float.POSITIVE_INFINITY;
            }
            return sign == 1 ? Float.NaN : Float.NaN;
        }
        float value = (float)f / 1024.0f;
        if (e == 0) {
            e = -14;
        } else {
            value += 1.0f;
            e -= 15;
        }
        value = Math.scalb(value, e);
        if (sign == 1) {
            value = -value;
        }
        return value;
    }

    public byte getByte(byte[] buffer, int index) throws IndexOutOfBoundsException {
        return this.support.getByte(buffer, index);
    }

    public void putByte(byte[] buffer, int index, byte value) throws IndexOutOfBoundsException {
        this.support.putByte(buffer, index, value);
    }

    public short getShort(byte[] buffer, int index) throws IndexOutOfBoundsException {
        return this.support.getShort(buffer, index);
    }

    public void putShort(byte[] buffer, int index, short value) throws IndexOutOfBoundsException {
        this.support.putShort(buffer, index, value);
    }

    public int getInt(byte[] buffer, int index) throws IndexOutOfBoundsException {
        return this.support.getInt(buffer, index);
    }

    public void putInt(byte[] buffer, int index, int value) throws IndexOutOfBoundsException {
        this.support.putInt(buffer, index, value);
    }

    public long getLong(byte[] buffer, int index) throws IndexOutOfBoundsException {
        return this.support.getLong(buffer, index);
    }

    public void putLong(byte[] buffer, int index, long value) throws IndexOutOfBoundsException {
        this.support.putLong(buffer, index, value);
    }

    public long getLong(byte[] buffer, int index, int numBytes) throws IndexOutOfBoundsException {
        switch (numBytes) {
            case 1: {
                return this.getByte(buffer, index);
            }
            case 2: {
                return this.getShort(buffer, index);
            }
            case 4: {
                return this.getInt(buffer, index);
            }
            case 8: {
                return this.getLong(buffer, index);
            }
        }
        throw CompilerDirectives.shouldNotReachHere((String)"number of bytes must be 1,2,4 or 8");
    }

    public long getLongUnsigned(byte[] buffer, int index, int numBytes) throws IndexOutOfBoundsException {
        switch (numBytes) {
            case 1: {
                return (long)this.getByte(buffer, index) & 0xFFL;
            }
            case 2: {
                return (long)this.getShort(buffer, index) & 0xFFFFL;
            }
            case 4: {
                return (long)this.getInt(buffer, index) & 0xFFFFFFFFL;
            }
            case 8: {
                return this.getLong(buffer, index);
            }
        }
        throw CompilerDirectives.shouldNotReachHere((String)"number of bytes must be 1,2,4 or 8");
    }

    public void putLong(byte[] buffer, int index, long value, int numBytes) throws IndexOutOfBoundsException {
        switch (numBytes) {
            case 1: {
                this.putByte(buffer, index, (byte)value);
                break;
            }
            case 2: {
                this.putShort(buffer, index, (short)value);
                break;
            }
            case 4: {
                this.putInt(buffer, index, (int)value);
                break;
            }
            case 8: {
                this.putLong(buffer, index, value);
                break;
            }
            default: {
                throw CompilerDirectives.shouldNotReachHere((String)"number of bytes must be 1,2,4 or 8");
            }
        }
    }

    public BigInteger getBigInteger(byte[] buffer, boolean signed) throws OverflowException {
        return this.getBigInteger(buffer, 0, buffer.length, signed);
    }

    @CompilerDirectives.TruffleBoundary
    public BigInteger getBigInteger(byte[] buffer, int offset, int length, boolean signed) throws OverflowException {
        assert (length <= buffer.length - offset);
        if (length == 0) {
            return BigInteger.ZERO;
        }
        byte[] bytes = buffer;
        if (!this.bigEndian) {
            bytes = new byte[length];
            for (int i = 0; i < length; ++i) {
                bytes[length - i - 1] = buffer[offset + i];
            }
            offset = 0;
        }
        try {
            if (signed) {
                return new BigInteger(bytes, offset, length);
            }
            return new BigInteger(1, bytes, offset, length);
        }
        catch (ArithmeticException e) {
            throw OverflowException.INSTANCE;
        }
    }

    @CompilerDirectives.TruffleBoundary
    public void putBigInteger(byte[] buffer, int index, BigInteger value, int numBytes) throws IndexOutOfBoundsException, OverflowException {
        assert (numBytes <= buffer.length - index);
        byte[] src = value.toByteArray();
        int srcIndex = 0;
        int srcBytes = src.length;
        if (srcBytes > numBytes) {
            for (i = 0; i < src.length; ++i) {
                srcIndex = i;
                if (src[i] != 0) break;
            }
            if ((srcBytes -= srcIndex) > numBytes) {
                throw OverflowException.INSTANCE;
            }
        } else if (srcBytes < numBytes && value.signum() < 0) {
            for (i = index; i < index + numBytes - srcBytes; ++i) {
                buffer[i] = -1;
            }
        }
        int dstIndex = index + (numBytes - srcBytes);
        PythonUtils.arraycopy(src, srcIndex, buffer, dstIndex, srcBytes);
        if (!this.bigEndian) {
            NumericSupport.reverse(buffer, index, numBytes);
        }
    }

    public float getFloat(byte[] buffer, int index) throws IndexOutOfBoundsException {
        return this.support.getFloat(buffer, index);
    }

    public void putFloat(byte[] buffer, int index, float value) throws IndexOutOfBoundsException {
        this.support.putFloat(buffer, index, value);
    }

    public float getHalfFloat(byte[] buffer, int index) throws IndexOutOfBoundsException {
        short bits = this.support.getShort(buffer, index);
        return NumericSupport.shortBitsToFloat(bits);
    }

    public void putHalfFloat(byte[] buffer, int index, double value, Node raisingNode) throws IndexOutOfBoundsException {
        short bits = NumericSupport.floatToShortBits(value, raisingNode);
        this.support.putShort(buffer, index, bits);
    }

    public double getDouble(byte[] buffer, int index) throws IndexOutOfBoundsException {
        return this.support.getDouble(buffer, index);
    }

    public void putDouble(byte[] buffer, int index, double value) throws IndexOutOfBoundsException {
        this.support.putDouble(buffer, index, value);
    }

    public double getDouble(byte[] buffer, int index, int numBytes) throws IndexOutOfBoundsException {
        switch (numBytes) {
            case 2: {
                return this.getHalfFloat(buffer, index);
            }
            case 4: {
                return this.getFloat(buffer, index);
            }
            case 8: {
                return this.getDouble(buffer, index);
            }
        }
        throw CompilerDirectives.shouldNotReachHere((String)"number of bytes must be 2,4 or 8");
    }

    public void putDouble(Node inliningTarget, byte[] buffer, int index, double value, int numBytes, PRaiseNode.Lazy raiseNode) throws IndexOutOfBoundsException {
        switch (numBytes) {
            case 2: {
                this.putHalfFloat(buffer, index, value, inliningTarget);
                break;
            }
            case 4: {
                float fValue = (float)value;
                if (Float.isInfinite(fValue) && Double.isFinite(value)) {
                    throw raiseNode.get(inliningTarget).raise(PythonErrorType.OverflowError, ErrorMessages.FLOAT_TO_LARGE_TO_PACK_WITH_S_FMT, "f");
                }
                this.putFloat(buffer, index, fValue);
                break;
            }
            case 8: {
                this.putDouble(buffer, index, value);
                break;
            }
            default: {
                throw CompilerDirectives.shouldNotReachHere((String)"number of bytes must be 2,4 or 8");
            }
        }
    }

    public static short asUnsigned(byte value) {
        return (short)(value & 0xFF);
    }

    public static int asUnsigned(short value) {
        return value & 0xFFFF;
    }

    public static long asUnsigned(int value) {
        return (long)value & 0xFFFFFFFFL;
    }

    @CompilerDirectives.TruffleBoundary
    public static BigInteger asUnsigned(long value) {
        if (value >= 0L) {
            return BigInteger.valueOf(value);
        }
        int upper = (int)(value >>> 32);
        int lower = (int)value;
        return BigInteger.valueOf(Integer.toUnsignedLong(upper)).shiftLeft(32).add(BigInteger.valueOf(Integer.toUnsignedLong(lower)));
    }
}

