/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.binder.internal;

import org.hibernate.AnnotationException;
import org.hibernate.annotations.Comment;
import org.hibernate.annotations.Comments;
import org.hibernate.binder.AttributeBinder;
import org.hibernate.binder.TypeBinder;
import org.hibernate.boot.spi.MetadataBuildingContext;
import org.hibernate.mapping.Component;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;

import java.util.HashSet;
import java.util.Set;

/**
 * Handles the {@link Comments} annotation.
 *
 * @author Gavin King
 */
public class CommentsBinder implements TypeBinder<Comments>, AttributeBinder<Comments> {
	@Override
	public void bind(Comments comments, MetadataBuildingContext context, PersistentClass entity, Property property) {
		final CommentBinder commentBinder = new CommentBinder();
		final Set<String> ons = new HashSet<>( comments.value().length );
		for ( Comment comment : comments.value() ) {
			if ( !ons.add( comment.on() ) ) {
				throw new AnnotationException( "Multiple '@Comment' annotations of '"
						+ property.getName() + "' had the same 'on'" );
			}
			commentBinder.bind( comment, context, entity, property );
		}
	}

	@Override
	public void bind(Comments comments, MetadataBuildingContext context, PersistentClass entity) {
		final CommentBinder commentBinder = new CommentBinder();
		final Set<String> ons = new HashSet<>( comments.value().length );
		for ( Comment comment : comments.value() ) {
			if ( !ons.add( comment.on() ) ) {
				throw new AnnotationException( "Multiple '@Comment' annotations of entity '"
						+ entity.getEntityName() + "' had the same 'on'" );
			}
			commentBinder.bind( comment, context, entity );
		}
	}

	@Override
	public void bind(Comments comments, MetadataBuildingContext context, Component embeddable) {
		throw new AnnotationException( "Embeddable class '" + embeddable.getComponentClassName()
				+ "' was annotated '@Comment' (annotate its attributes instead)" );
	}
}
