/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableDoubleBag;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.list.primitive.MutableDoubleList;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.impl.bag.mutable.primitive.DoubleHashBag;
import com.gs.collections.impl.block.factory.primitive.DoublePredicates;
import com.gs.collections.impl.factory.primitive.DoubleSets;
import com.gs.collections.impl.list.mutable.primitive.DoubleArrayList;
import com.gs.collections.impl.set.mutable.primitive.DoubleHashSet;
import com.gs.collections.impl.utility.internal.primitive.DoubleIterableIterate;
import com.gs.collections.impl.utility.primitive.LazyDoubleIterate;

import java.util.NoSuchElementException;

/**
 * This file was automatically generated from template file abstractLazyPrimitiveIterable.stg.
 *
 * @since 5.0
 */
public abstract class AbstractLazyDoubleIterable implements LazyDoubleIterable
{
    public int size()
    {
        return this.count(DoublePredicates.alwaysTrue());
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public boolean isEmpty()
    {
        return DoubleIterableIterate.isEmpty(this);
    }

    public boolean notEmpty()
    {
        return DoubleIterableIterate.notEmpty(this);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        DoubleIterableIterate.appendString(this, appendable, start, separator, end);
    }

    public boolean contains(double value)
    {
        return this.anySatisfy(DoublePredicates.equal(value));
    }

    public boolean containsAll(double... source)
    {
        return this.containsAll(DoubleSets.immutable.of(source));
    }

    public boolean containsAll(DoubleIterable source)
    {
        return source.allSatisfy(new DoublePredicate()
        {
            public boolean accept(double value)
            {
                return AbstractLazyDoubleIterable.this.contains(value);
            }
        });
    }

    public LazyDoubleIterable select(DoublePredicate predicate)
    {
        return LazyDoubleIterate.select(this, predicate);
    }

    public LazyDoubleIterable reject(DoublePredicate predicate)
    {
        return LazyDoubleIterate.select(this, DoublePredicates.not(predicate));
    }

    public <V> LazyIterable<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        return LazyDoubleIterate.collect(this, function);
    }

    public double detectIfNone(DoublePredicate predicate, double ifNone)
    {
        return DoubleIterableIterate.detectIfNone(this, predicate, ifNone);
    }

    public int count(DoublePredicate predicate)
    {
        return DoubleIterableIterate.count(this, predicate);
    }

    public boolean anySatisfy(DoublePredicate predicate)
    {
        return DoubleIterableIterate.anySatisfy(this, predicate);
    }

    public boolean allSatisfy(DoublePredicate predicate)
    {
        return DoubleIterableIterate.allSatisfy(this, predicate);
    }

    public boolean noneSatisfy(DoublePredicate predicate)
    {
        return DoubleIterableIterate.noneSatisfy(this, predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        return DoubleIterableIterate.injectInto(this, injectedValue, function);
    }

    public double[] toArray()
    {
        return this.toList().toArray();
    }

    public MutableDoubleList toList()
    {
        final MutableDoubleList list = new DoubleArrayList();
        this.forEach(new DoubleProcedure()
        {
            public void value(double each)
            {
                list.add(each);
            }
        });
        return list;
    }

    public MutableDoubleSet toSet()
    {
        final MutableDoubleSet set = new DoubleHashSet();
        this.forEach(new DoubleProcedure()
        {
            public void value(double each)
            {
                set.add(each);
            }
        });
        return set;
    }

    public MutableDoubleBag toBag()
    {
        final MutableDoubleBag bag = new DoubleHashBag();
        this.forEach(new DoubleProcedure()
        {
            public void value(double each)
            {
                bag.add(each);
            }
        });
        return bag;
    }

    public double sum()
    {
        DoubleSumProcedure procedure = new DoubleSumProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public double max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        DoubleMaxProcedure procedure = new DoubleMaxProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public double maxIfEmpty(double ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.max();
    }

    public double min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        DoubleMinProcedure procedure = new DoubleMinProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public double minIfEmpty(double ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        double[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            double first = sortedArray[middleIndex];
            double second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public double[] toSortedArray()
    {
        return this.toSortedList().toArray();
    }

    public MutableDoubleList toSortedList()
    {
        return DoubleArrayList.newList(this).sortThis();
    }

    private static final class DoubleMaxProcedure implements DoubleProcedure
    {
        private boolean visitedOnce;
        private double max;

        public void value(double each)
        {
            if (this.visitedOnce)
            {
                if (Double.compare(this.max, each) < 0)
                {
                    this.max = each;
                }
            }
            else
            {
                this.max = each;
                this.visitedOnce = true;
            }
        }

        public double getValue()
        {
            return this.max;
        }
    }

    private static final class DoubleMinProcedure implements DoubleProcedure
    {
        private boolean visitedOnce;
        private double min;

        public void value(double each)
        {
            if (this.visitedOnce)
            {
                if (Double.compare(each, this.min) < 0)
                {
                    this.min = each;
                }
            }
            else
            {
                this.min = each;
                this.visitedOnce = true;
            }
        }

        public double getValue()
        {
            return this.min;
        }
    }

    private static final class DoubleSumProcedure implements DoubleProcedure
    {
        private double sum = 0;

        public void value(double each)
        {
            this.sum += each;
        }

        public double getValue()
        {
            return this.sum;
        }
    }

    public LazyDoubleIterable asLazy()
    {
        return this;
    }
}
