/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.impl.factory.primitive.IntLists;
import com.gs.collections.impl.lazy.primitive.CollectIntToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectIntIterable;

/**
 * LazyIntIterate is a factory class which creates "deferred" int iterables around the specified int iterables. A "deferred"
 * int iterable performs some operation, such as filtering or transforming, when the result int iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyIntIterate
{
    private static final LazyIntIterable EMPTY_ITERABLE = IntLists.immutable.of().asLazy();

    private LazyIntIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred int iterable for the specified int iterable.
     */
    public static LazyIntIterable adapt(IntIterable iterable)
    {
        return new LazyIntIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering int iterable for the specified int iterable.
     */
    public static LazyIntIterable select(IntIterable iterable, IntPredicate predicate)
    {
        return new SelectIntIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming int iterable for the specified int iterable.
     */
    public static <V> LazyIterable<V> collect(
            IntIterable iterable,
            IntToObjectFunction<? extends V> function)
    {
        return new CollectIntToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming int iterable for the specified int iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            IntIterable iterable,
            IntPredicate predicate,
            IntToObjectFunction<? extends V> function)
    {
        return LazyIntIterate.select(iterable, predicate).collect(function);
    }

    public static LazyIntIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
