package csbase.client.algorithms.commands.view;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommandInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;

/**
 * Cria uma visualizao de um {@link CommandInfo}
 * 
 * @author Tecgraf / PUC-Rio
 */
public class AlgorithmCommandViewFactory {

  /**
   * Template
   */
  private static final String VIEW_INDEX_TEMPLATE = "|%s|";

  /**
   * Template
   */
  private static final String NODE_INDEX_TEMPLATE = VIEW_INDEX_TEMPLATE
    + "|%s|";

  /**
   * Cria uma viso do comando.
   * 
   * @param owner A janela pai da view a ser criada (No aceita {@code null}.)
   * @param command O comando (No aceita {@code null}).
   * @param preferredTab Qual a tab gostaria que estivesse selecionada quando
   *        esta janela for criada. No  garantido que esta janela abra com a
   *        tab escolhida selecionada, pois aquela tab pode nem existir caso
   *        seja a tab de log.
   * 
   * @throws ClientException falha ao carregar informaes do comando.
   * @throws IllegalArgumentException se a janela (window) for de tipo errado ou
   *         se o comando for {@code null}.
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  public static void showView(final DesktopComponentFrame owner,
    final CommandInfo command, final TabType preferredTab)
    throws ClientException {

    if (null == command) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }

    RemoteTask<AlgorithmConfigurator> task =
      new RemoteTask<AlgorithmConfigurator>() {
        @Override
        public void performTask() throws Exception {
          AlgorithmConfigurator configurator = command.getConfigurator();
          setResult(configurator);
        }
      };
    String msg = LNG.get("AlgorithmCommandViewFactory.loading.configurator");
    if (!task.execute(owner, msg, msg)) {
      return;
    }
    AlgorithmConfigurator configurator = task.getResult();

    String index = String.format(VIEW_INDEX_TEMPLATE, command.getId());

    CommandViewType viewType;
    if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
      viewType = CommandViewType.FLOW;
    }
    else {
      viewType = CommandViewType.SIMPLE;
    }

    showView(index, owner, command, configurator, preferredTab, viewType, null);
  }

  /**
   * Cria uma viso do comando.
   * 
   * @param owner A janela pai da view a ser criada (No aceita {@code null}.)
   * @param command um comando de fluxo (No aceita {@code null}).
   * @param node n que deve ter seus parmetros visualizados.
   * @param preferredTab Qual a tab gostaria que estivesse selecionada quando
   *        esta janela for criada. No  garantido que esta janela abra com a
   *        tab escolhida selecionada, pois aquela tab pode nem existir caso
   *        seja a tab de log.
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  public static void showView(final DesktopComponentFrame owner,
    final CommandInfo command, final GraphNode node, final TabType preferredTab)
    throws ClientException {

    if (null == command) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }
    if (null == node) {
      throw new IllegalArgumentException("O parmetro node est nulo.");
    }

    AlgorithmConfigurator configurator =
      node.getAlgorithmConfiguratorView().getConfigurator();

    String index =
      String.format(NODE_INDEX_TEMPLATE, owner.getIndex(), node.getId());

    showView(index, owner, command, configurator, preferredTab,
      CommandViewType.NODE, node.getId());
  }

  /**
   * Cria um dilogo.
   * 
   * @param index identificador da view. Usado para garantir a unicidade.
   * @param owner A janela pai da view a ser criada (No aceita {@code null}.)
   * @param command O comando (No aceita {@code null}).
   * @param configurator o configurador.
   * @param preferredTab Qual a tab gostaria que estivesse selecionada quando
   *        esta janela for criada. No  garantido que esta janela abra com a
   *        tab escolhida selecionada, pois aquela tab pode nem existir caso
   *        seja a tab de log.
   * @param logViewType Tipo de log a ser mostrado nessa viso do comando (No
   *        aceita {@code null}).
   * @param nodeId Identificador do algoritmo, caso seja um n de um fluxo.
   * 
   * @throws ClientException falha ao carrega informaes do comando
   * @throws IllegalArgumentException se a janela (window) for de tipo errado ou
   *         se o comando for {@code null}.
   */
  private static void showView(final Object index,
    final DesktopComponentFrame owner, final CommandInfo command,
    final AlgorithmConfigurator configurator, final TabType preferredTab,
    final CommandViewType logViewType, Integer nodeId) throws ClientException {

    if (null == command) {
      throw new IllegalArgumentException(
        "No foi passado um comando para ser visualizado.");
    }

    AbstractAlgorithmCommandView view =
      (AbstractAlgorithmCommandView) DesktopComponentFrame
        .getDesktopComponentFrame(index);

    if (null == view) {
      view =
        createView(index, owner, command, configurator, logViewType, nodeId);
    }

    if (null == view) {
      StandardErrorDialogs.showErrorDialog(owner,
        LNG.get("AlgorithmCommandViewFactory.error.view.unknown"));
    }
    else {
      view.show(preferredTab);
    }
  }

  /**
   * Criao da viso.
   * 
   * @param index o ndice
   * @param owner a janela-me.
   * @param command o comando
   * @param configurator o configurador
   * @param viewType Tipo de log a ser mostrado nessa viso do comando.
   * @param nodeId Identificador do algoritmo, caso seja um n de um fluxo.
   * 
   * @return a viso.
   * 
   * @throws ClientException se houver falha na construo da viso.
   */
  private static AbstractAlgorithmCommandView createView(final Object index,
    final DesktopComponentFrame owner, final CommandInfo command,
    final AlgorithmConfigurator configurator, CommandViewType viewType,
    Integer nodeId) throws ClientException {

    if (null == configurator) {
      return null;
    }

    switch (viewType) {
      case FLOW:
        return new FlowAlgorithmCommandView(index, owner, command, configurator);
      case SIMPLE:
        return new SimpleAlgorithmCommandView(index, owner, command,
          configurator, CommandViewType.SIMPLE, null);
      case NODE:
        return new SimpleAlgorithmCommandView(index, owner, command,
          configurator, CommandViewType.NODE, nodeId);
      default:
        return null;
    }
  }

}
