package csbase.client.algorithms.view;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.rmi.RemoteException;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.algorithms.parameters.ParameterView.Mode;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.kernel.ClientException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * Painel representando a viso dos parmetros de um configurador.
 * 
 * @param <T> tipo de configurador que este painel representa.
 * 
 * @author Tecgraf / PUC-Rio
 */
public abstract class AbstractAlgorithmConfiguratorPanel<T extends AlgorithmConfigurator>
  extends JPanel implements ViewValidator {

  /**
   * Enumerao usada para decidir quais os botes internos so mostrados.
   */
  private enum ShowInnerButtons {
    /** Mostra todos os botes internos */
    ALL,
    /** Mostra o boto restaurar padro */
    RESTORE_DEFAULT,
    /** No exibe os botes internos */
    NONE
  }

  /**
   * Indica que botes internos devem ser exibidos. Atualmente s existe o boto
   * de restaurar os valores padro.
   */
  private static ShowInnerButtons showInnerButtons;

  /**
   * Janela que contm o painel.
   */
  private final Window owner;

  /**
   * Configurador do algoritmo representado no painel.
   */
  private final T configurator;

  /**
   * Painel de parmetros.
   */
  private AbstractParametersPanel parametersPanel;

  static {
    /**
     * Carrega a propriedade que indica que botes internos devem ser exibidos.
     */
    try {
      Configuration configuration =
        ConfigurationManager.getInstance().getConfiguration(
          AlgorithmConfiguratorView.class);
      showInnerButtons =
        ShowInnerButtons.valueOf(configuration.getOptionalProperty(
          "show.inner.buttons", ShowInnerButtons.ALL.name()));
    }
    catch (ConfigurationManagerException e) {
      showInnerButtons = ShowInnerButtons.ALL;
    }
    catch (IllegalArgumentException e) {
      // Se caiu aqui  porque o valor da propriedade no arquivo est invlido
      showInnerButtons = ShowInnerButtons.ALL;
    }
  }

  /**
   * Construtor.
   * 
   * @param owner A janela que criou esta (No aceita {@code null}).
   * @param configurator o configurador de algortimo que ser representado por
   *        este painel.
   * @param mode Modo de visualizao. Pode ser {@link Mode#CONFIGURATION} ou
   *        {@link Mode#REPORT}.
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  protected AbstractAlgorithmConfiguratorPanel(final Window owner,
    final T configurator, final Mode mode) throws ClientException {
    super();
    this.owner = owner;
    this.configurator = configurator;
    initialize(mode);
  }

  /**
   * @return the owner
   */
  public Window getOwner() {
    return owner;
  }

  /**
   * Obtm o configurador de algortimo representado por este painel.
   * 
   * @return o configurador de algortimo representado por este painel.
   */
  public T getConfigurator() {
    return configurator;
  }

  /**
   * @return a janela que detm este componente.
   */
  public Window getWindow() {
    return SwingUtilities.windowForComponent(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean highlightValidationResult(ViewValidationResult result) {
    return parametersPanel.highlightValidationResult(result);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ViewValidationResult validate(ValidationMode validationMode)
    throws RemoteException {
    return parametersPanel.validate(validationMode);
  }

  /**
   * Reseta os valores do configurador.<br>
   * Utilizado somente no modo {@link Mode#CONFIGURATION}
   */
  public void resetValues() {
    getConfigurator().resetValues();
  }

  /**
   * Cria o painel em modo {@link Mode#REPORT}.
   * 
   * @return O painel em modo {@link Mode#REPORT}.
   * @throws ClientException em caso de falha.
   */
  protected abstract AbstractParametersPanel createParametersReportPanel()
    throws ClientException;

  /**
   * Cria o painel em modo {@link Mode#CONFIGURATION}.
   * 
   * @return O painel em modo {@link Mode#CONFIGURATION}.
   * @throws ClientException em caso de falha.
   */
  protected abstract AbstractParametersPanel createParametersConfigurationPanel()
    throws ClientException;

  /**
   * Inicializa o painel.
   * 
   * @param mode Modo de visualizao. Pode ser {@link Mode#CONFIGURATION} ou
   *        {@link Mode#REPORT}.
   * @throws ClientException em caso de falha.
   */
  private void initialize(final Mode mode) throws ClientException {
    switch (mode) {
      case REPORT: {
        initializeInReportMode();
        break;
      }
      case CONFIGURATION: {
        initializeInConfigurationMode();
        break;
      }
      default: {
        throw new IllegalArgumentException("Modo de viso desconhecido.");
      }
    }
  }

  /**
   * Inicializa o painel em modo {@link Mode#REPORT}.
   * 
   * @throws ClientException em caso de falha.
   */
  private void initializeInReportMode() throws ClientException {
    parametersPanel = createParametersReportPanel();
    setLayout(new GridLayout());
    add(parametersPanel, new GBC(0, 0).both());
  }

  /**
   * Inicializa o painel em modo {@link Mode#CONFIGURATION}.
   * 
   * @throws ClientException em caso de falha.
   */
  private void initializeInConfigurationMode() throws ClientException {
    parametersPanel = createParametersConfigurationPanel();

    if (ShowInnerButtons.NONE == showInnerButtons) {
      setLayout(new GridLayout());
      add(parametersPanel, new GBC(0, 0).both());
    }
    else {
      setLayout(new BorderLayout());

      add(parametersPanel, BorderLayout.CENTER);

      JPanel buttonsPanel = createButtonPanel();
      add(buttonsPanel, BorderLayout.SOUTH);
    }
  }

  /**
   * Cria um painel com os botes.
   * 
   * @return O painel com os botes.
   */
  private JPanel createButtonPanel() {
    JPanel buttonsPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));

    JButton restoreButton = new JButton(getString("resetDefaultValues"));
    restoreButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        resetValues();
      }

    });
    buttonsPanel.add(restoreButton);

    return buttonsPanel;
  }

  /**
   * Obtm um texto no mecanismo de internacionalizao.
   * 
   * @param key A chave para o texto (Ser prfixado o nome deste classe seguido
   *        de '.'; no aceita {@code null}).
   * 
   * @return O texto.
   */
  private String getString(String key) {
    return LNG.get(AlgorithmConfiguratorView.class.getName() + "." + key);
  }

  /**
   * Painel com os parmetros.
   */
  protected abstract class AbstractParametersPanel extends JPanel implements
    ViewValidator {
    // Classe de marcao.
  }
}
