/*
 * $Id: AbstractSimpleApplicationAction.java 109131 2010-08-16 22:05:21Z clinio
 * $
 */

package csbase.client.applications;

import java.awt.event.ActionEvent;
import java.net.URL;
import java.text.MessageFormat;

import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.KeyStroke;

import csbase.client.ClientKeyStrokeUtil;
import csbase.client.util.StandardErrorDialogs;

/**
 * <p>
 * Implementao abstrata de {@link ApplicationAction} que visa facilitar a
 * criao de aes dentro de uma aplicao.
 * </p>
 * <p>
 * Esta classe ir buscar na fraseologia da ao pelo nome, mnemnico e dica da
 * ao. As chaves que referenciam estes valores so, respectivamente, "
 * {@code getClass().getSimpleName()}.{@value #TAG_NAME}", "
 * {@code getClass().getSimpleName()}.{@value #TAG_MNEMONIC}", "
 * {@code getClass().getSimpleName()}.{@value #TAG_TOOLTIP}".
 * {@code getClass().getSimpleName()}.{@value #TAG_ACCELERATOR}".
 * </p>
 * <p>
 * Alm disso, essa classe tambm disponibiliza, para seus descendentes, os
 * mtodos {@link #getApplication()} e
 * {@link #getClassString(String, Object...)}. Este ltimo possibilita o acesso
 * a fraseologia da aplicao.
 * </p>
 * 
 * @param <T> tipo da aplicao.
 * 
 * @author Tecgraf / PUC-Rio
 */
public abstract class AbstractSimpleApplicationAction<T extends Application>
  extends ApplicationAction<T> {
  /**
   * Tag usado para buscar o nome da ao.
   */
  final static protected String TAG_NAME = "name";

  /**
   * Tag usado para buscar o mnemnico da ao.
   */
  final static protected String TAG_MNEMONIC = "mnemonic";

  /**
   * Tag usado para buscar o tooltip da ao.
   */
  final static protected String TAG_TOOLTIP = "tooltip";

  /**
   * Tag usado para buscar o acelerador da ao.
   */
  final static protected String TAG_ACCELERATOR = "accelerator";

  /**
   * Construtor.
   * 
   * @param application aplicao na qual esta ao est sendo executada.
   * @param icon a imagem da ao.
   */
  protected AbstractSimpleApplicationAction(final T application,
    final ImageIcon icon) {
    this(application);
    putValue(Action.SMALL_ICON, icon);
  }

  /**
   * Construtor.
   * 
   * @param application aplicao na qual esta ao est sendo executada.
   */
  protected AbstractSimpleApplicationAction(final T application) {
    super(application);

    final String name = getName();
    if (name != null) {
      putValue(Action.NAME, name);
    }

    final Integer mnemonic = getMnemonic();
    if (mnemonic != null) {
      putValue(Action.MNEMONIC_KEY, mnemonic);
    }

    final String tooltip = getToolTip();
    if (tooltip != null) {
      putValue(Action.SHORT_DESCRIPTION, tooltip);
    }

    final KeyStroke keyStroke = getAccelerator();
    if (keyStroke != null) {
      putValue(Action.ACCELERATOR_KEY, keyStroke);
    }

    putDefaultIcon(application);
  }

  /**
   * Coloca cone no padro se houver.
   * 
   * @param application aplicao
   */
  private void putDefaultIcon(final T application) {
    final ImageIcon defaultIcon = getDefaultIcon();
    if (defaultIcon != null) {
      putValue(Action.SMALL_ICON, defaultIcon);
      return;
    }

    final ImageIcon internalIcon = loadInternalIcon();
    if (internalIcon != null) {
      putValue(Action.SMALL_ICON, internalIcon);
    }
  }

  /**
   * Coloca cone padronizado, caso no seja definido algum. Normalmente, esse
   * mtodo  interessante para as subclasses redefinirem retornando alguma
   * imagem padronizada, mesmo usando o construtor que no define a imagem.
   * Neste caso temos a possibilidade dos protocolos push ou pull para o ajuste
   * de imagens.
   * 
   * @return a imagem.
   */
  protected ImageIcon getDefaultIcon() {
    return null;
  }

  /**
   * Coloca cone no padro se houver.
   * 
   * @return a imagem.
   */
  final private ImageIcon loadInternalIcon() {
    final T application = getApplication();
    final String sep = "/";
    final Class<? extends Application> appClass = application.getClass();
    final Class<?> actionClass = this.getClass();
    final String actionClassName = actionClass.getSimpleName();
    final String path = "actions" + sep + actionClassName + ".16.gif";
    final String resPath = "resources" + sep + "images" + sep + path;
    final URL url = appClass.getResource(resPath);
    if (url == null) {
      return null;
    }
    final ImageIcon imageIcon = new ImageIcon(url);
    return imageIcon;
  }

  /**
   * Repassa a chamada ao mtodo {@link #handleActionPerformed(ActionEvent)} e
   * trata quaisquer excees que ele venha a lanar.
   * 
   * @param ae evento que gerou a ao.
   */
  @Override
  public final void actionPerformed(ActionEvent ae) {
    try {
      handleActionPerformed(ae);
    }
    catch (Exception e) {
      final T application = getApplication();
      final ApplicationFrame frame = application.getApplicationFrame();
      final String title = frame.getTitle();
      StandardErrorDialogs.showErrorDialog(frame, title, e);
    }
  }

  /**
   * Mtodo que implementa a lgica da ao.
   * 
   * @param ae evento que gerou a ao.
   * 
   * @throws Exception se houver falha na ao.
   */
  protected abstract void handleActionPerformed(ActionEvent ae)
    throws Exception;

  /**
   * Retorna um texto de internacionalizao, usando uma chave sem o prefixo
   * padro que  o nome da classe. Obtm dinamicamente o nome da classe para
   * usar como prefixo da busca.
   * 
   * @param key_sufix o sufixo da chave. A chave ser: \"
   *        {@code getClass().getSimpleName()}.{@code key_sufix}\"
   * 
   * @param args argumentos do texto. O texto ser formatado utilizando-se da
   *        classe {@link MessageFormat}.
   * 
   * @return o texto internacionalizado.
   */
  protected final String getClassString(String key_sufix, Object... args) {
    final String className = getClass().getSimpleName();
    final String formattedKey = String.format("%s.%s", className, key_sufix);
    final T app = getApplication();
    if (app.hasString(formattedKey)) {
      return app.getString(formattedKey, args);
    }
    return null;
  }

  /**
   * Retorna um texto de internacionalizao.
   * 
   * @param key_sufix o sufixo da chave.
   * 
   * @param args argumentos do texto. O texto ser formatado utilizando-se da
   *        classe {@link MessageFormat}.
   * 
   * @return o texto internacionalizado.
   */
  protected final String getString(String key_sufix, Object... args) {
    final T app = getApplication();
    if (app.hasString(key_sufix)) {
      return app.getString(key_sufix, args);
    }
    return null;
  }

  /**
   * Obtm o nome da ao.<br>
   * O nome ser procurado na fraseologia da aplicao a partir da chave \"
   * {@code getClass().getSimpleName()}.{@value #TAG_NAME}\".
   * 
   * @return o nome da ao ou null caso no seja encontrado na fraseologia da
   *         aplicao.
   */
  private final String getName() {
    final String name = getClassString(TAG_NAME);
    if (name == null) {
      final String className = getClass().getSimpleName();
      return "<<" + className + "." + TAG_NAME + ">>";
    }
    return name;
  }

  /**
   * Obtm a dica da ao.<br>
   * O nome ser procurado na fraseologia da aplicao a partir da chave \"
   * {@code getClass().getSimpleName()}.{@value #TAG_TOOLTIP}\".
   * 
   * @return a dica da ao ou null caso no seja encontrado na fraseologia da
   *         aplicao.
   */
  private final String getToolTip() {
    return getClassString(TAG_TOOLTIP);
  }

  /**
   * Obtm o mnemonico da ao.<br>
   * O nome ser procurado na fraseologia da aplicao a partir da chave \"
   * {@code getClass().getSimpleName()}.{@value #TAG_MNEMONIC}\".
   * 
   * @return o mnemonico da ao ou null caso no seja encontrado na fraseologia
   *         da aplicao.
   */
  private final Integer getMnemonic() {
    final String mn = getClassString(TAG_MNEMONIC);
    if (mn == null) {
      return null;
    }
    if (mn.trim().isEmpty()) {
      return null;
    }
    return Integer.valueOf(mn.charAt(0));
  }

  /**
   * Obtm o mnemonico da ao.<br>
   * O nome ser procurado na fraseologia da aplicao a partir da chave \"
   * {@code getClass().getSimpleName()}.{@value #TAG_MNEMONIC}\".
   * 
   * @return o mnemonico da ao ou null caso no seja encontrado na fraseologia
   *         da aplicao.
   */
  private final KeyStroke getAccelerator() {
    final String ac = getClassString(TAG_ACCELERATOR);
    if (ac == null) {
      return null;
    }
    if (ac.trim().isEmpty()) {
      return null;
    }

    return getStrokeFromText(ac);
  }

  /**
   * Busca o stroke a ser usado com base no texto.
   * 
   * @param text texto.
   * @return stroke
   */
  protected KeyStroke getStrokeFromText(final String text) {
    final KeyStroke keyStroke = ClientKeyStrokeUtil.getCtrlKeyStroke(text);
    return keyStroke;
  }

}
