package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.Iterator;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.algorithmsmanager.DataPropertiesListener;
import csbase.client.applications.algorithmsmanager.models.CategoryNodeInterface;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;

/**
 * Painel com as informaes bsicas da categoria selecionada para edio.
 * 
 */
public class CategoryBasicInfoPanel extends CommonInfoEditPanel {

  /** Campo texto (read-only) com o nome da categoria pai */
  private JTextField tf_categoryParentName;

  /** Campo texto com o nome da categoria */
  private JTextField tf_categoryName;

  /** Indica que est sendo feita uma edio e os dados foram alterados */
  private boolean isEditing;

  /**
   * Painel de associao de algoritmos a categorias
   */
  private AlgorithmsAndCategoriesBindPanel bindPanel;

  /**
   * Classe que representa o dado de algoritmo a ser gerenciado no painel de
   * associaes de algoritmos e categorias.
   * 
   */
  private class AlgorithmData implements DataInterface,
    Comparable<AlgorithmData> {
    /** Informaes do algoritmo */
    private AlgorithmInfo algoInfo;

    /**
     * Construtor.
     * 
     * @param algoInfo informaes do algoritmo
     */
    AlgorithmData(AlgorithmInfo algoInfo) {
      this.algoInfo = algoInfo;
    }

    @Override
    public String getName() {
      return algoInfo.getName();
    }

    @Override
    public String getId() {
      return algoInfo.getId();
    }

    @Override
    public Icon getIcon() {
      return ApplicationImages.ICON_ALGORITHM_16;
    }

    @Override
    public String getFullName() {
      return getName();
    }

    @Override
    public int compareTo(AlgorithmData algoData) {
      return algoInfo.getId().compareToIgnoreCase(algoData.getId());
    }
  }

  /**
   * Constri o painel para alterao de uma categoria selecionada
   * 
   * @param categoryPane painel com os dados de atualizao de uma categoria
   */
  public CategoryBasicInfoPanel(CategoryEditDataPanel categoryPane) {
    super(categoryPane);
  }

  /**
   * Obtm o painel com os campos para alterao de uma categoria.
   * 
   * @return o painel criado
   */
  private JPanel getUpdatePanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    JLabel categoryNameLabel =
      new JLabel(getString("CategoryBasicInfoPanel.label.name"));
    JLabel parentCategoryNameLabel =
      new JLabel(getString("CategoryBasicInfoPanel.label.parent.name"));
    tf_categoryParentName = new JTextField();
    tf_categoryParentName.setEditable(false);
    tf_categoryName = new JTextField(20);
    tf_categoryName.setEditable(false);

    panel.add(parentCategoryNameLabel, new GBC(0, 0).none().west().insets(5, 5,
      5, 5));
    panel.add(tf_categoryParentName, new GBC(1, 0).horizontal().west().insets(
      5, 5, 5, 5));
    panel
      .add(categoryNameLabel, new GBC(0, 1).none().west().insets(5, 5, 5, 5));
    panel.add(tf_categoryName, new GBC(1, 1).horizontal().west().insets(5, 5,
      5, 5));
    panel.add(getDataBindingPanel(), new GBC(0, 2).gridwidth(2).both().west()
      .insets(5, 5, 5, 5));
    return panel;
  }

  /**
   * Obtm o nome da categoria especificado pelo usurio.
   * 
   * @return o nome da categoria
   */
  protected String getFieldCategoryName() {
    return tf_categoryName.getText();
  }

  @Override
  public void initializeData() {
    if (getSelectedCategory() == null) {
      return;
    }
    if (!isEditing) { // && !wasModified()) {
      tf_categoryParentName.setText(getSelectedCategoryParentName());
      tf_categoryName.setText(getSelectedCategoryName());
      tf_categoryName.requestFocus();
      bindPanel.initializeData(getSelectedCategory(), getAvailableAlgorithms(),
        getCategoryAlgorithms());
      revalidate();
    }
    changeOperationsState(isEditing);
  }

  /**
   * Obtm os filhos previamente associados com os ns selecionados. Deve
   * incluir todos os filhos que j estejam associados.
   * 
   * @return os filhos previamente associados com os ns
   */
  private SortedSet<DataInterface> getCategoryAlgorithms() {
    SortedSet<DataInterface> selectedData = new TreeSet<DataInterface>();
    if (getSelectedCategory() != null) {
      CategoryNodeInterface categoryData = getSelectedCategory();
      if (!categoryData.isRootNode()) {
        Set<AlgorithmInfo> algorithms = categoryData.getNode().getAlgorithms();
        selectedData = getAlgorithmDataInterfaceSet(algorithms);
      }
    }
    return selectedData;
  }

  /**
   * Constri, a partir das informaes dos algorimtos, um conjunto de dados com
   * informao comum que o painel de associaes requer.
   * 
   * @param algorithms conjunto de algoritmos
   * @return um conjunto de dados com as informaes dos algoritmos
   */
  private SortedSet<DataInterface> getAlgorithmDataInterfaceSet(
    Set<AlgorithmInfo> algorithms) {
    SortedSet<DataInterface> algorithmsDataSet = new TreeSet<DataInterface>();
    for (AlgorithmInfo algoInfo : algorithms) {
      algorithmsDataSet.add(new AlgorithmData(algoInfo));
    }
    return algorithmsDataSet;
  }

  /**
   * Obtm os filhos disponveis para associao com os ns selecionados. Deve
   * incluir todos os filhos existentes que ainda no estejam associados.
   * 
   * @return os filhos disponveis para a associao
   */
  private SortedSet<DataInterface> getAvailableAlgorithms() {
    SortedSet<DataInterface> availableData = new TreeSet<DataInterface>();
    availableData = getAllAgorithms();

    if (getSelectedCategory() == null) {
      return availableData;
    }
    // ** Exclui da lista de disponveis os filhos que j pertecem ao pai
    // selecionado

    // Por enquanto, tratar o caso de um nico n pai selecionado
    CategoryNodeInterface categoryData = getSelectedCategory();
    if (!categoryData.isRootNode()) {
      Set<AlgorithmInfo> dataChildren = categoryData.getNode().getAlgorithms();
      Iterator<DataInterface> availableIterator = availableData.iterator();

      while (availableIterator.hasNext()) {
        DataInterface node = availableIterator.next();
        if (dataChildren.contains(node)) {
          availableIterator.remove();
        }
      }
    }
    return availableData;
  }

  /**
   * Obter todas os algoritmos disponveis no servidor.
   * 
   * @return as algoritmos disponveis no servidor
   */
  private SortedSet<DataInterface> getAllAgorithms() {
    SortedSet<AlgorithmInfo> allAgorithms = getApplication().getAllAgorithms();
    return getAlgorithmDataInterfaceSet(allAgorithms);
  }

  @Override
  protected void apply() {
    isEditing = false;
    saveInfo();
  }

  @Override
  protected JPanel buildMainInfoPanel() {
    return getUpdatePanel();
  }

  @Override
  protected void cancel() {
    if (confirmCancelling()) {
      initializeData();
    }
  }

  /**
   * Confirma com o usurio se a operao de cancelamento deve ser efetivada, j
   * que perder todas as alteraes feitas sobre o dado.
   * 
   * @return retorna true, se a alterao do dado deve ser cancelada, caso
   *         contrrio, retorna false
   */
  private boolean confirmCancelling() {
    int confirm =
      getApplication().showOptionDialog(
        getString("CategoryBasicInfoPanel.msg.cancel.confirm"),
        new String[] {
            getString("CategoryBasicInfoPanel.msg.cancel.confirm.yes"),
            getString("CategoryBasicInfoPanel.msg.cancel.confirm.no") });
    if (confirm == 0) {
      isEditing = false;
      return true;
    }
    return false;
  }

  /**
   * Obtm o painel que realiza a associao dos dados.
   * 
   * @return o painel
   */
  private JPanel getDataBindingPanel() {
    if (bindPanel == null) {
      bindPanel =
        new AlgorithmsAndCategoriesBindPanel(getApplication(),
          getSelectedCategory(), getAvailableAlgorithms(),
          getCategoryAlgorithms(),
          AlgorithmsAndCategoriesBindPanel.ItemType.ALGORITHM_ITEM);
      bindPanel.addBindListener(new DataPropertiesListener() {

        @Override
        public void propertiesChanged(boolean wasModified) {
          isEditing = wasModified;
          changeOperationsState(wasModified);
        }
      });
    }
    return bindPanel;
  }

  @Override
  public void setDataChanged() {
    isEditing = false;
    changeOperationsState(isEditing);
  }

  /**
   * Obtm a categoria selecionada para edio.
   * 
   * @return retorna a categoria a ser editada
   */
  private String getSelectedCategoryName() {
    if (getSelectedCategory() != null) {
      return getSelectedCategory().getName();
    }
    return null;
  }

  /**
   * Obtm a categoria pai da categoria selecionada para edio.
   * 
   * @return retorna a categoria pai da categoria em edio, caso contrrio,
   *         retorna null
   */
  public String getSelectedCategoryParentName() {
    if (getSelectedCategory() != null) {
      Category parentCategory = null;
      parentCategory = getSelectedCategory().getNode().getParentCategory();
      if (parentCategory != null) {
        return parentCategory.getFullName();
      }
      return CategoryTreeView.ROOT_NAME;
    }
    return null;
  }

  /**
   * Salva efetivamente as informaes no servidor.
   * 
   * @return retorna true se as informaes foram salvas com sucesso, caso
   *         contrrio, retorna false
   */
  protected boolean saveInfo() {
    bindPanel.bindData();
    return true;
  }

  /**
   * Obtm a categoria selecionada para edio.
   * 
   * @return a categoria selecionada para edio
   */
  private CategoryNodeInterface getSelectedCategory() {
    return getCategoryEditDataPanel().getSelectedCategory();
  }

  /**
   * Obtm o painel global de edio de dados de categoria.
   * 
   * @return o painel global de edio de dados de categoria
   */
  private CategoryEditDataPanel getCategoryEditDataPanel() {
    return (CategoryEditDataPanel) getEditPanel();
  }

  @Override
  public boolean wasModified() {
    if (bindPanel.wasModified()) {
      return true;
    }
    return false;
  }

  @Override
  protected String getTitle() {
    return getApplication().getString("CommonEditTabbedPanel.tab.info.basic");
  }
}
