package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.util.ClientUtilities;

/**
 * Classe abstrata que define um painel comum para funcionalidades de edio de
 * determinadas informaes do dado, como as informaes bsicas, informaes de
 * verso, de relacionamentos, etc.
 * 
 * Esse painel precisa ser definido tanto na funcionalidade de gerenciamento de
 * categorias quanto na funcionalidade de gerenciamento de algoritmos, na
 * aplicao Gerenciador de Algoritmos.
 * 
 * Para customizar esse painel, redefinir o mtodo
 * <code> buildMainInfoPanel </code>, que contm as informaes a serem
 * manipuladas.
 * 
 * Na base do painel, so exibidos os botes Aplicar e Cancelar, comuns a
 * qualquer painel para edio de informaes do dado selecionado.
 */
public abstract class CommonInfoEditPanel extends
  AbstractSimpleApplicationPanel<AlgorithmsManager> {
  /** Texto que representa uma alterao nos dados do painel */
  public final static String STR_UPDATE = "*";

  /** painel de edio global do dado, que contm esse painel */
  private CommonEditTabbedPanel dataPane;

  /** Boto para aplicar a operao sobre as informaes */
  private JButton applyButton;

  /** Boto para cancelar a operao sobre as informaes */
  private JButton cancelButton;

  /**
   * Constri o painel para edio das informaes do dado.
   * 
   * @param dataPane painel de edio global do dado, que contm esse painel
   */
  public CommonInfoEditPanel(CommonEditTabbedPanel dataPane) {
    super(dataPane.getApplication());
    this.dataPane = dataPane;
    buildPanel();
  }

  /**
   * Obtm o painel de edio global do dado, que contm esse painel.
   * 
   * @return o painel de edio global do dado, que contm esse painel
   * 
   */
  public CommonEditTabbedPanel getEditPanel() {
    return dataPane;
  }

  /**
   * Constri o painel principal com as informaes especficas da operao de
   * edio a ser realizada sobre o dado.
   * 
   * @return o painel principal criado
   * 
   */
  protected abstract JPanel buildMainInfoPanel();

  /**
   * Aplica a operao realizada sobre as informaes.
   */
  protected abstract void apply();

  /**
   * Cancela a operao realizada sobre as informaes.
   */
  protected abstract void cancel();

  /**
   * Obtm o ttulo do painel que deve ser exibido na aba correspondente.
   * 
   * @return o ttulo do painel
   */
  protected abstract String getTitle();

  /**
   * Inicializa os campos com as informaes do dado, e realiza todas as outras
   * aes necessrias quando uma outra seleo de dado  feita.
   */
  public abstract void initializeData();

  /**
   * Indica que o dado selecionado mudou aps um cancelamento.
   * 
   * Por exemplo, se os dados da categoria corrente estiverem em modo de edio,
   * ento o usurio deve confirmar se deseja cancelar ou no, antes de mudar
   * para a nova categoria.
   * 
   */
  public abstract void setDataChanged();

  /**
   * Verifica se houve alterao nas nas informaes do dado editado.
   * 
   * @return true se houver alterao das informaes, ou false caso nenhuma
   *         informao tenha sido modificada
   */
  public abstract boolean wasModified();

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.AbstractSimpleApplicationPanel#buildPanel()
   */
  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());
    JPanel mainPanel = buildMainInfoPanel();
    JPanel buttonsPanel = getButtonPanel();

    add(mainPanel, new GBC(0, 0).both().insets(5, 5, 5, 5));
    add(buttonsPanel, new GBC(0, 2).none().insets(5, 5, 5, 10));
  }

  /**
   * Cria o painel com os botes referentes a aplicao ou cancelamento da
   * operao realizada sobre as informaes.
   * 
   * @return o painel criado
   */
  private JPanel getButtonPanel() {
    JPanel buttonPanel = new JPanel(new GridBagLayout());
    applyButton =
      new JButton(getApplication()
        .getString("CommonInfoEditPanel.button.apply"));
    applyButton.setEnabled(false);
    applyButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        apply();
      }

    });
    cancelButton =
      new JButton(getApplication().getString(
        "CommonInfoEditPanel.button.cancel"));
    cancelButton.setEnabled(false);
    cancelButton.addActionListener(new ActionListener() {

      @Override
      public void actionPerformed(ActionEvent e) {
        cancel();
      }
    });
    buttonPanel.add(applyButton, new GBC(0, 0).none().south()
      .insets(5, 5, 5, 5));
    buttonPanel.add(cancelButton, new GBC(1, 0).south().insets(5, 5, 5, 5));
    ClientUtilities.adjustEqualSizes(applyButton, cancelButton);

    return buttonPanel;
  }

  /**
   * Exibe ou oculta os botes de operaes comuns ao painel de informaes de
   * um dado a ser manipulado
   * 
   * @param toshow se true, exibe os botes, caso contrrio, oculta os botes
   */
  protected void showOperationButtons(boolean toshow) {
    applyButton.setVisible(toshow);
    cancelButton.setVisible(toshow);
  }

  /**
   * Atribui um estado  ao de aplicar a operao sobre as informaes
   * correntes.
   * 
   * @param changed true, para habilitar a ao de aplicar, caso contrrio,
   *        false
   */
  protected void setApplyActionState(boolean changed) {
    applyButton.setEnabled(changed);
    String title = (!changed) ? getTitle() : getTitle() + STR_UPDATE;
    dataPane.setInfoPanelTitle(this, title);
  }

  /**
   * Atribui um estado  ao de cancelar a operao sobre as informaes
   * correntes.
   * 
   * @param state true, para habilitar a ao de cancelar, caso contrrio, false
   */
  protected void setCancelActionState(boolean state) {
    cancelButton.setEnabled(state);
  }

  /**
   * Modifica o estado das operaes de aplicar e cancelar.
   * 
   * @param changed true, para habilitar a ao de cancelar, caso contrrio,
   *        false
   */
  protected void changeOperationsState(boolean changed) {
    setApplyActionState(changed);
    setCancelActionState(changed);
  }

  /**
   * Constri um painel vazio. Pode ser necessrio para o caso de querer ocupar
   * a rea restante do painel principal.
   * 
   * @return o painel vazio
   * 
   */
  protected JPanel buildEmptyPanel() {
    return new JPanel();
  }
}
