package csbase.client.applications.jobmonitor;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observable;
import java.util.Observer;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.text.JTextComponent;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationComponentDialog;
import csbase.client.applications.jobmonitor.rowmodel.JobInfoRow;

/**
 * Dilogo que detalha as informaes de um {@link JobInfoRow}.
 * 
 * @author Tecgraf
 */
public class JobInfoDialog extends ApplicationComponentDialog<JobMonitor>
  implements Observer {

  /**
   * Mapeia o nome da coluna para o componente swing de texto que exibe o valor.
   */
  private Map<String, JTextComponent> rowAttributes;

  /**
   * Construtor padro.
   * 
   * @param app - referncia para a aplicao.
   * @param jobInfoRow - objeto que encapsula toda informao a ser exibida
   *        neste dialogo.
   */
  public JobInfoDialog(JobMonitor app, JobInfoRow jobInfoRow) {
    super(app);
    setTitle(app.getString("details.title"));

    rowAttributes = new LinkedHashMap<String, JTextComponent>();

    buildInterface(jobInfoRow);
  }

  /**
   * Constri a interface do dilogo.
   * 
   * @param jobInfoRow - objeto que encapsula toda informao a ser exibida
   *        neste dialogo.
   */
  private void buildInterface(JobInfoRow jobInfoRow) {
    Container container = getContentPane();
    container.setLayout(new GridBagLayout());

    Application app = getApplication();

    JButton close = new JButton(app.getString("button.close"));
    close.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        JobInfoDialog.this.setVisible(false);
      }
    });

    container.add(createPropertyPanel(jobInfoRow), new GBC(0, 0).both().insets(
      10));
    container.add(close, new GBC(0, 1).none().bottom(10).right(10).east());

    setMinimumSize(new Dimension(700, 500));
  }

  /**
   * Cria o painel que exibe as propriedades de uma linha como chave-valor.
   * 
   * @param selectedRow - linha selecionada.
   * @return painel.
   */
  private JScrollPane createPropertyPanel(JobInfoRow selectedRow) {

    JPanel panel = new JPanel(new GridBagLayout());

    int verticalIndex = 0;

    for (Entry<String, String> entry : selectedRow.entrySet()) {
      String columnName = entry.getKey();
      String columnValue = entry.getValue();

      JLabel property = new JLabel(columnName);

      JTextComponent value;
      String valueStr = columnValue.toString();
      if (valueStr != null && valueStr.length() > 40) {
        value = new JTextArea(valueStr, 1, 40);
        ((JTextArea) value).setLineWrap(true);
        value.setOpaque(false);
        value.setBorder(UIManager.getBorder("TextField.border"));
      }
      else {
        value = new JTextField(valueStr);
      }
      value.setEditable(false);

      rowAttributes.put(entry.getKey(), value);

      panel.add(property, new GBC(0, verticalIndex).insets(3).east());
      panel.add(value, new GBC(1, verticalIndex++).horizontal().insets(3));
    }

    // adicionando label vazio para alinhar corretamente
    // os componentes de interface quando o dilogo  maximizado.
    panel.add(new JLabel(), new GBC(0, verticalIndex).both());

    return new JScrollPane(panel);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void update(Observable o, Object arg) {
    JobMonitor app = (JobMonitor) o;
    final JobInfoRow row = app.getSelectedObject();

    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {

        if (row != null) {
          for (Entry<String, String> entry : row.entrySet()) {
            String columnName = entry.getKey();
            String value = entry.getValue();

            JTextComponent field = rowAttributes.get(columnName);
            if (field != null) {
              field.setText(value.toString());
            }
          }
        }

      }
    });

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean visible) {
    if (!visible) {
      getApplication().deleteObserver(this);
    }
    super.setVisible(visible);
  }

}
