package csbase.client.applications.projectsmanager.dialogs;

import java.awt.Color;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.labels.StandardPieSectionLabelGenerator;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.title.TextTitle;
import org.jfree.chart.title.Title;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.util.Rotation;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectGeneralAreaOccupationData;
import csbase.client.applications.projectsmanager.proxy.GetGeneralSpaceUsageTask;
import csbase.logic.Utilities;

/**
 * Dilogo que exibe informaes de ocupao da rea de projetos.
 *
 * @author jnlopes
 *
 */
public class ProjectAreaOccupationDialog extends ProjectsManagerDialog {

  /**
   * O grfico JFreeChart
   */
  final private JFreeChart graph;

  /**
   * Campo que indica o espao livre na rea de projetos.
   */
  final private JTextField freeDiskSpaceField = new JTextField();

  /**
   * Campo que indica o espao ocupado na rea de projetos.
   */
  final private JTextField occupiedDiskSpaceField = new JTextField();

  /**
   * Boto para obter o espao atual ocupado na rea de projetos.
   */
  final private JButton refreshButton = new JButton();

  /**
   * Boto para fechar o dilogo.
   */
  final private JButton closeButton = new JButton();

  /**
   * Label de espao ocupado na rea de projetos.
   */
  final private JLabel occupiedDiskSpaceLabel = new JLabel(ProjectsManagerUI
    .getString("ProjectAreaOccupationDialog.occupied.space.label"));

  /**
   * Monta o painel com componentes especficos para visualizao de espao
   * ocupado por usurios.
   */
  private void buildMainPanel() {
    final ChartPanel chartPanel = new ChartPanel(graph);
    chartPanel.setPopupMenu(null);
    BorderUtil.setLowBorder(chartPanel);

    final JLabel freeDiskSpaceLabel =
      new JLabel(ProjectsManagerUI
        .getString("ProjectAreaOccupationDialog.free.space.label"));

    final JComponent[][] textComps =
    { { freeDiskSpaceLabel, freeDiskSpaceField },
      { occupiedDiskSpaceLabel, occupiedDiskSpaceField }, };
    final JPanel textPanel = GUIUtils.createBasicGridPanel(textComps);

    BorderUtil.setLinedBorder(textPanel);

    final JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(refreshButton);
    buttonsPanel.add(closeButton);

    setLayout(new GridBagLayout());
    add(textPanel, new GBC(0, 0).horizontal());
    add(chartPanel, new GBC(0, 1).both());
    add(buttonsPanel, new GBC(0, 2).horizontal());
  }

  /**
   * Inicia componentes.
   */
  private void initGeneralComponents() {
    freeDiskSpaceField.setEditable(false);
    occupiedDiskSpaceField.setEditable(false);

    freeDiskSpaceField.setHorizontalAlignment(SwingConstants.RIGHT);
    occupiedDiskSpaceField.setHorizontalAlignment(SwingConstants.RIGHT);

    refreshButton.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        refreshData();
      }
    });
    refreshButton.setText(ProjectsManagerUI
      .getString("ProjectAreaOccupationDialog.refresh.button"));

    final ProjectAreaOccupationDialog pod = this;
    closeButton.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        pod.dispose();
      }
    });
    closeButton.setText(ProjectsManagerUI
      .getString("ProjectAreaOccupationDialog.close.button"));

    final JComponent[] buttons =
      new JComponent[] { refreshButton, closeButton };
    GUIUtils.matchPreferredSizes(buttons);
  }

  /**
   * Atualiza campos com informaes de ocupao da rea de projetos.
   */
  final private void refreshData() {

    final ProjectsManager projectsManager = getProjectsManager();
    final GetGeneralSpaceUsageTask ggsut =
      new GetGeneralSpaceUsageTask(projectsManager);

    final ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    final String msg = getString("ProjectAreaOccupationDialog.task.message");
    ggsut.execute(appFrame, projectsManager.getName(), msg);

    if (ggsut.wasCancelled()) {
      final String err =
        getString("ProjectAreaOccupationDialog.task.cancelled.message");
      ggsut.showError(err);
      return;
    }

    if (ggsut.getStatus() != true) {
      final Exception exception = ggsut.getError();
      projectsManager.showError(this, exception.getMessage());
      return;
    }

    final ProjectGeneralAreaOccupationData paod = ggsut.getResult();
    updateFields(paod);
    updateChart(paod);
  }

  /**
   * Atualizao do grfico
   *
   * @param paod o dado vindo do servidor.
   */
  private void updateChart(final ProjectGeneralAreaOccupationData paod) {
    final PiePlot pieplot = (PiePlot) graph.getPlot();
    double free;
    double used;
    if (paod == null) {
      used = 0;
      free = 0;
    }
    else {
      free = paod.getFreeSpace();
      used = paod.getOccupiedSpace();
    }

    if ((free < 0) || (used < 0)) {
      used = 0;
      free = 0;
    }

    graph.setTitle(getString("ProjectAreaOccupationDialog.chart.title"));
    final List<Title> subtitles = new ArrayList<Title>();
    final String timeText = Utilities.getFormattedDate((new Date()).getTime());
    subtitles.add(new TextTitle(timeText));
    graph.setSubtitles(subtitles);

    pieplot
    .setNoDataMessage(getString("ProjectAreaOccupationDialog.failed.error"));
    final String freeStr = getString("ProjectAreaOccupationDialog.free.legend");
    final String usedStr = getString("ProjectAreaOccupationDialog.used.legend");

    final DefaultPieDataset dataset = new DefaultPieDataset();
    dataset.setValue(freeStr, free);
    dataset.setValue(usedStr, used);
    pieplot
    .setLabelGenerator(new StandardPieSectionLabelGenerator("{0} : {2}"));

    pieplot.setSectionPaint(freeStr, new Color(150, 255, 150));
    pieplot.setSectionPaint(usedStr, new Color(255, 150, 150));
    pieplot.setDirection(Rotation.CLOCKWISE);
    pieplot.setBackgroundPaint(null);
    pieplot.setOutlinePaint(null);
    pieplot.setCircular(true);
    pieplot.setDataset(dataset);
  }

  /**
   * Atualizao dos campos de texto na interface.
   *
   * @param paod o dado trazido do servidor.
   */
  final private void updateFields(final ProjectGeneralAreaOccupationData paod) {
    if (paod == null) {
      final String invText =
        getString("ProjectAreaOccupationDialog.invalid.text");
      freeDiskSpaceField.setText(invText);
      occupiedDiskSpaceField.setText(invText);
    }
    else {
      final double freeSpace = paod.getFreeSpace();
      final String str1 =
        ProjectsManagerUI.generateSizeStringFromDouble(freeSpace);
      freeDiskSpaceField.setText(str1);

      final double occupiedSpace = paod.getOccupiedSpace();
      final String str2 =
        ProjectsManagerUI.generateSizeStringFromDouble(occupiedSpace);
      occupiedDiskSpaceField.setText(str2);
    }
  }

  /**
   * Construtor
   *
   * @param projectsManager A aplicao.
   */
  public ProjectAreaOccupationDialog(final ProjectsManager projectsManager) {
    super(projectsManager);

    final String title = projectsManager.getName();
    graph = ChartFactory.createPieChart(title, null, true, false, false);

    initGeneralComponents();
    buildMainPanel();
    refreshData();
  }

}
