/*
 * $Id: AbstractProjectDateFilter.java 146656 2013-11-07 17:13:39Z mjulia $
 */

package csbase.client.applications.projectsmanager.panels.filters;

import java.awt.Color;
import java.awt.GridBagLayout;
import java.awt.event.KeyListener;
import java.util.Date;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.calendar.CalendarDateField;
import tecgraf.javautils.gui.calendar.CalendarEvent;
import tecgraf.javautils.gui.calendar.CalendarListener;
import tecgraf.javautils.gui.calendar.CalendarPanel;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;

/**
 * Classe abstrata para filtros de faixas de datas.
 * 
 * @author Tecgraf
 */
abstract class AbstractProjectDateFilter extends AbstractProjectFilter {

  /**
   * Data inicial.
   */
  private CalendarDateField cdfInitial;
  /**
   * Data final.
   */
  private CalendarDateField cdfFinal;
  /**
   * Label de data inicial.
   */
  private JLabel fromDateLabel;
  /**
   * Label de data final.
   */
  private JLabel untilDateLabel;
  /**
   * Painel para label de data inicial.
   */
  private JPanel fromPanel;
  /**
   * Painel para label de data fim.
   */
  private JPanel untilPanel;
  /**
   * Painel para data de criao.
   */
  private JLabel mainLabel;
  /**
   * Data mnima (timestamp)
   */
  static final private long EARLIEST_DATE = 0;
  /**
   * Data mxima (timestamp)
   */
  static final private long LATEST_DATE = Long.MAX_VALUE;

  /**
   * Construtor.
   * 
   * @param projectsManager a aplicao de gerenciamento de projetos
   */
  public AbstractProjectDateFilter(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * Obtm o valor correspondente a uma data para ser usado na comparao feita
   * pelo filtro.
   * 
   * @param date data
   * @param defaultValue valor default
   * @return <code>long</code> correspondente  data para ser usado na
   *         comparao feita pelo filtro
   * 
   * @see Date#getTime()
   */
  private long prepareDateForComparison(final Date date, final long defaultValue) {
    return date != null ? date.getTime() : defaultValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void runSpecificFilter(final List<ProjectsManagerData> allProjects,
    final List<ProjectsManagerData> filteredProjects) {
    final long initialDate =
      prepareDateForComparison(cdfInitial.getDate(), EARLIEST_DATE);
    final long finalDate =
      prepareDateForComparison(cdfFinal.getDate(), LATEST_DATE);

    if (initialDate < 0 || finalDate < 0 || initialDate > finalDate) {
      return;
    }

    for (ProjectsManagerData pmd : allProjects) {
      if (projectMatchesFilter(initialDate, finalDate, pmd)) {
        filteredProjects.add(pmd);
      }
    }
  }

  /**
   * Verifica se a data de criao do projeto est na faixa especificada por
   * duas datas.
   * 
   * @param initialDate data de incio
   * @param finalDate data de fim
   * @param prj projeto
   * @return <code>true</code> se a data de criao do projeto est na faixa
   *         especificada
   */
  protected abstract boolean projectMatchesFilter(final long initialDate,
    final long finalDate, final ProjectsManagerData prj);

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean projectMatchesFilter(final ProjectsManagerData prj) {
    final long initialDate =
      prepareDateForComparison(cdfInitial.getDate(), EARLIEST_DATE);
    final long finalDate =
      prepareDateForComparison(cdfFinal.getDate(), LATEST_DATE);
    if (initialDate < 0 || finalDate < 0 || initialDate > finalDate) {
      return false;
    }

    return projectMatchesFilter(initialDate, finalDate, prj);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addListener(KeyListener kl) {

    final JTextField initialField = cdfInitial.getField();
    initialField.addKeyListener(kl);
    final JTextField finalField = cdfFinal.getField();
    finalField.addKeyListener(kl);

    final ProjectsManager projectsManager = getProjectsManager();

    final CalendarListener initialCalendarListener = new CalendarListener() {
      @Override
      public void dateSelected(CalendarEvent arg0) {
        cdfInitial.setDate(new Date(arg0.getDate()));
        projectsManager.runFilters(true);
      }
    };

    final CalendarListener finalCalendarListener = new CalendarListener() {
      @Override
      public void dateSelected(CalendarEvent arg0) {
        cdfFinal.setDate(new Date(arg0.getDate()));
        projectsManager.runFilters(true);
      }
    };

    final CalendarPanel initialPanel = cdfInitial.getCalendar();
    initialPanel.addCalendarListener(initialCalendarListener);
    final CalendarPanel finalPanel = cdfFinal.getCalendar();
    finalPanel.addCalendarListener(finalCalendarListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void emptyFilterData() {
    final JTextField initialField = cdfInitial.getField();
    initialField.setText("");
    final JTextField finalField = cdfFinal.getField();
    finalField.setText("");
  }

  /**
   * Retorna o painel de data de criao inicial.
   * 
   * @return O painel de data de criao inicial.
   */
  public JPanel getFromPanel() {
    return fromPanel;
  }

  /**
   * Retorna o painel de data de criao final.
   * 
   * @return O painel de data de criao final.
   */
  public JPanel getUntilPanel() {
    return untilPanel;
  }

  /**
   * Monta o painel.
   */
  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());

    fromPanel = new JPanel(new GridBagLayout());

    fromPanel.add(fromDateLabel, new GBC(0, 0));
    fromPanel.add(cdfInitial, new GBC(1, 0).horizontal().insets(0, 10, 0, 10));

    untilPanel = new JPanel(new GridBagLayout());

    untilPanel.add(untilDateLabel, new GBC(1, 0));
    untilPanel.add(cdfFinal, new GBC(2, 0).horizontal().insets(0, 10, 0, 0));

    add(fromPanel, new GBC(0, 0).horizontal());
    add(untilPanel, new GBC(1, 0).horizontal());
  }

  /**
   * Obtm o label principal do painel (aquele que indica seu propsito).
   * 
   * @return label principal do painel
   */
  public JLabel getMainLabel() {
    return mainLabel;
  }

  /**
   * Inicia os componentes.
   */
  @Override
  protected void initComponents() {
    cdfInitial = new CalendarDateField();
    cdfFinal = new CalendarDateField();
    fromDateLabel = new JLabel(getClassString("initial.date.label"));
    untilDateLabel = new JLabel(getClassString("final.date.label"));
    mainLabel = createMainLabel();
    fromDateLabel.setForeground(Color.GRAY);
    untilDateLabel.setForeground(Color.GRAY);
  }

  /**
   * Mtodo para definio do label principal do painel.
   * 
   * @return label principal do painel
   */
  protected abstract JLabel createMainLabel();
}