package csbase.client.applications.projectsmanager.panels.filters;

import java.awt.event.KeyListener;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.swing.JTextField;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.panels.ProjectsManagerPanel;

/**
 * Classe abstrata que define os mtodos necessrios aos painis responsveis
 * pela filtragem da tabela do {@link ProjectsManager}.
 * 
 * @author Tecgraf
 */
public abstract class AbstractProjectFilter extends ProjectsManagerPanel {

  /**
   * Construtor.
   * 
   * @param projectsManager aplicao de gerenciamento de projetos
   */
  protected AbstractProjectFilter(final ProjectsManager projectsManager) {
    super(projectsManager);
    initComponents();
    buildPanel();
  }

  /**
   * Inicializa os componentes.
   */
  protected abstract void initComponents();

  /**
   * Monta o painel.
   */
  protected abstract void buildPanel();

  /**
   * Verifica se um projeto  aceito por este filtro.
   * 
   * @param prj projeto
   * @return <code>true</code> se o projeto  aceito por este filtro
   */
  public abstract boolean projectMatchesFilter(ProjectsManagerData prj);

  /**
   * Executa filtragem baseado nas informacoes obtidas neste painel.
   * 
   * @param allProjects projetos a serem filtrados.
   * @param refresh indica se deve ser feito refresh na tabela
   * @return lista com os projetos filtrados.
   */
  public List<ProjectsManagerData> runFilter(
    final List<ProjectsManagerData> allProjects, final boolean refresh) {
    final List<ProjectsManagerData> filtered =
      new ArrayList<ProjectsManagerData>();
    runSpecificFilter(allProjects, filtered);
    final ProjectsManager projectsManager = getProjectsManager();
    projectsManager.setFilteredProjects(filtered);
    if (refresh) {
      projectsManager.refreshProjectsTable();
    }
    return filtered;
  }

  /**
   * Filtragem propriamente dita, implementada pelas subclasses.
   * 
   * @param allProjects lista de todos os projetos
   * @param filteredProjects lista de projetos que passaram pelo filtro (a ser
   *        preenchida pelo mtodo)
   */
  protected abstract void runSpecificFilter(
    List<ProjectsManagerData> allProjects,
    List<ProjectsManagerData> filteredProjects);

  /**
   * Apaga informaes dos campos dos filtros.
   */
  public abstract void emptyFilterData();

  /**
   * Adiciona listener padro aos elementos deste painel que devem disparar uma
   * nova filtragem sempre que forem modificados.
   * 
   * @param kl Listener
   */
  public abstract void addListener(KeyListener kl);

  /**
   * Converte o texto de um campo em um pattern, considerando que o texto  o
   * prefixo do que deve ser encontrado (i.e. "aaa" == "aaa.*")
   * 
   * @param field texto a ser convertido
   * @return {@link Pattern} compilado a partir do texto do campo, ou null caso
   *         no seja possvel gerar um padro a partir do texto
   */
  protected Pattern convertToPattern(final JTextField field) {
    String text = field.getText().trim();
    Pattern patt = null;

    if (text.isEmpty()) {
      text = ".*";
    }
    else {
      text = text.replace("*", ".*") + ".*";
    }

    try {
      patt = Pattern.compile(text);
    }
    catch (PatternSyntaxException e) {
      StandardDialogs.showErrorDialog(getParent(),
        getString("AbstractProjectFilter.error.title"), "<<"
          + getString("AbstractProjectFilter.error.message") + "\"" + text
          + "\">>");
    }
    return patt;
  }
}