/*
 * $Id:$
 */
package csbase.client.desktop.projectoccupation;

import java.awt.Color;
import java.awt.Font;
import java.rmi.RemoteException;

import javax.swing.JProgressBar;
import javax.swing.plaf.basic.BasicProgressBarUI;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.util.MonitoringTheme;
import csbase.logic.CommonClientProject;
import csbase.logic.diskusageservice.DiskOccupation;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.DiskUsageServiceInterface;

/**
 * Painel de ocupao de projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectOccupationProgressBar extends JProgressBar {

  /**
   * A tree associada.
   */
  final private ProjectTree projectTree;
  /**
   * Thread.
   */
  final private Thread thread;

  /**
   * Indicativo de funcionamento.
   */
  protected boolean alive = false;

  /**
   * Intervalo de consulta.
   */
  private int intervalSec;

  /**
   * Valor para indicar alerta.
   */
  private int alertValue = 90;

  /**
   * Valor para indicar ateno.
   */
  private int warningValue = 75;

  /**
   * Cor de ateno.
   */
  private Color warningColor = MonitoringTheme.getRangeWarningColor();

  /**
   * Cor de estado normal.
   */
  private Color normalColor = MonitoringTheme.getRangeDefaultColor();

  /**
   * Cor de estado de alerta.
   */
  private Color alertColor = MonitoringTheme.getRangeAlertColor();

  /**
   * Tempo do ltimo refresh automtico.
   */
  private long lastRefresh;

  /**
   * Adiciona listeners na project tree.
   */
  private void addProjectTreeListeners() {
    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(final CommonClientProject proj) {
        refresh(true);
      }
    });
  }

  /**
   * Refresh automtico
   */
  private void autoRefresh() {
    while (ProjectOccupationProgressBar.this.alive) {
      final long now = System.currentTimeMillis();
      final long delta = now - lastRefresh;
      long deltaSec = Math.round(delta / 1000.);
      if (deltaSec >= intervalSec) {
        refresh(false);
        deltaSec = intervalSec;
      }
      try {
        //        System.out.println("Z: " + " " + deltaSec + "s " + (new Date()));
        Thread.sleep(deltaSec * 1000);
      }
      catch (final InterruptedException e) {
        //        System.out.println("I: " + (new Date()));
      }
    }
  }

  /**
   * Cria a thread de modo automtico.
   * 
   * @return a thread
   */
  private Thread createAutoThread() {
    final Thread th = new Thread() {
      @Override
      public void run() {
        autoRefresh();
      }
    };

    final String className = this.getClass().getSimpleName();
    final long tag = System.currentTimeMillis();
    final String threadName = className + "::" + tag;
    th.setName(threadName);
    return th;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finalize() throws Throwable {
    shutdown();
    super.finalize();
  }

  /**
   * Retorna o atributo: cor de alerta; (ver {@link #alertColor})
   * 
   * @return o atributo.
   */
  public final Color getAlertColor() {
    return alertColor;
  }

  /**
   * Retorna o atributo: valor de alerta; (ver {@link #alertValue})
   * 
   * @return o atributo.
   */
  public final int getAlertValue() {
    return alertValue;
  }

  /**
   * Retorna intervalo de consulta.
   * 
   * @return intervalSec (ver {@link #intervalSec})
   */
  public final int getIntervalSec() {
    return intervalSec;
  }

  /**
   * Retorna o atributo: cor normal; (ver {@link #normalColor})
   * 
   * @return o atributo.
   */
  public final Color getNormalColor() {
    return normalColor;
  }

  /**
   * Monta texto de internacionalizao.
   * 
   * @param tag tag
   * @param args argumentos
   * @return texto
   */
  private String getString(final String tag, final Object... args) {
    final String prefix = getClass().getSimpleName();
    final String key = prefix + "." + tag;
    final String fmt = LNG.get(key);
    final String text = String.format(fmt, args);
    return text;
  }

  /**
   * Retorna o atributo: cor de aviso; (ver {@link #warningColor})
   * 
   * @return o atributo.
   */
  public final Color getWarningColor() {
    return warningColor;
  }

  /**
   * Retorna o atributo: valor de aviso; (ver {@link #warningValue})
   * 
   * @return o atributo.
   */
  public final int getWarningValue() {
    return warningValue;
  }

  /**
   * Atualiza a barra de progresso.
   * 
   * @param forced fora refresh independentemente da ltima atualizao.
   */
  final public void refresh(final boolean forced) {
    //    System.out.println("R: " + (new Date()));
    if (forced) {
      lastRefresh = 0;
    }
    final long now = System.currentTimeMillis();
    final long delta = now - lastRefresh;
    final int minIntervalMiliSeconds = 1000;
    if (delta < minIntervalMiliSeconds) {
      return;
    }
    //    System.out.println("R!: " + (new Date()));
    lastRefresh = now;

    if (projectTree == null || projectTree.getProject() == null) {
      setData(false, "no-prj", "no-prj", Color.white, 100);
      return;
    }

    final DiskOccupation occ = getCurrentProjectOccupation();
    if (!occ.isValid()) {
      final String text = getString("invalid.text");
      final String tooltip = getString("invalid.tooltip");
      setData(true, text, tooltip, Color.white, 100);
      return;
    }

    final double percUsed = occ.getUsedSpacePerc();
    final double gbFree = occ.getFreeSpaceMb() / 1024.;
    final double gbUsed = occ.getUsedSpaceMb() / 1024.;
    final double gbTotal = occ.getTotalSpaceMb() / 1024.;
    final int value = (int) Math.round(percUsed);
    Color color = normalColor;
    if (value > warningValue) {
      color = warningColor;
    }
    if (value > alertValue) {
      color = alertColor;
    }

    final String txtTag = "normal.text";
    final String text = getString(txtTag, percUsed, gbUsed, gbFree, gbTotal);
    final String tipTag = "normal.tooltip";
    final String tooltip = getString(tipTag, percUsed, gbUsed, gbFree, gbTotal);
    setData(true, text, tooltip, color, value);
  }

  /**
   * Retorna ocupao.
   * 
   * @return ocupao
   */
  private DiskOccupation getCurrentProjectOccupation() {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    if (project == null) {
      return new DiskOccupation();
    }

    final DiskOccupation occ;
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      occ = getOccupationByTask(project);
    }
    else {
      occ = getOccupationDirect(project);
    }
    return occ;
  }

  /**
   * Busca ocupao diretamente do servio.
   * 
   * @param project projeto
   * @return ocupao
   */
  private DiskOccupation getOccupationDirect(final CommonClientProject project) {
    final DiskUsageServiceInterface diskUsageService =
      ClientRemoteLocator.diskUsageService;
    final Object owner = project.getUserId();
    final String projectName = project.getName();
    try {
      final DiskOccupation serverOccupation =
        diskUsageService.getSingleProjectOccupation(owner, projectName);
      return serverOccupation;
    }
    catch (RemoteException e) {
      return new DiskOccupation();
    }
  }

  /**
   * Busca ocupao diretamente do servio via task.
   * 
   * @param project projeto
   * @return ocupao
   */
  private DiskOccupation getOccupationByTask(final CommonClientProject project) {
    RemoteTask<DiskOccupation> task = new RemoteTask<DiskOccupation>() {
      @Override
      public void performTask() throws Exception {
        final DiskUsageServiceInterface diskUsageService =
          ClientRemoteLocator.diskUsageService;
        final Object owner = project.getUserId();
        final String projectName = project.getName();
        final DiskOccupation serverOccupation =
          diskUsageService.getSingleProjectOccupation(owner, projectName);
        setResult(serverOccupation);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      protected void handleError(Exception error) {
        setResult(null);
      }
    };

    task.setProgressDialogDelay(60);
    task.execute(DesktopFrame.getInstance().getDesktopFrame(), "", "Disk: "
      + project.getName());
    final DiskOccupation occ = task.getResult();
    return occ;
  }

  /**
   * Ajustes dos dados da progress bar com invokeLater, se necessrio.
   * 
   * @param visible indicativo de visibilidade
   * @param text texto
   * @param tooltip tooltip
   * @param foreground cor
   * @param value valor
   */
  private void setData(final boolean visible, final String text,
    final String tooltip, final Color foreground, final int value) {
    if (!SwingThreadDispatcher.isEventDispatchThread()) {
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          setDataInEDT(visible, text, tooltip, foreground, value);
        }
      });
    }
    else {
      setDataInEDT(visible, text, tooltip, foreground, value);
    }
  }

  /**
   * Ajustes dos dados da progress bar.
   * 
   * @param visible indicativo de visibilidade
   * @param text texto
   * @param tooltip tooltip
   * @param foreground cor
   * @param value valor
   */
  private void setDataInEDT(final boolean visible, final String text,
    final String tooltip, final Color foreground, final int value) {
    setVisible(visible);
    setToolTipText(tooltip);
    setString(text);
    setForeground(foreground);
    setValue(value);
  }

  /**
   * Ajusta o atributo cor de alerta; (ver {@link #alertColor})
   * 
   * @param alertColor o novo valor
   */
  public final void setAlertColor(final Color alertColor) {
    this.alertColor = alertColor;
  }

  /**
   * Ajusta o atributo valor de alerta; (ver {@link #alertValue})
   * 
   * @param alertValue o novo valor
   */
  public final void setAlertValue(final int alertValue) {
    this.alertValue = alertValue;
  }

  /**
   * Ajusta intervalo de consulta.
   * 
   * @param intervalSec intervalo (ver {@link #intervalSec})
   */
  public final void setIntervalSec(final int intervalSec) {
    this.intervalSec = intervalSec;
    refresh(false);
  }

  /**
   * Ajusta o atributo cor normal; (ver {@link #normalColor})
   * 
   * @param normalColor o novo valor
   */
  public final void setNormalColor(final Color normalColor) {
    this.normalColor = normalColor;
  }

  /**
   * Ajusta o atributo cor de aviso; (ver {@link #warningColor})
   * 
   * @param warningColor o novo valor
   */
  public final void setWarningColor(final Color warningColor) {
    this.warningColor = warningColor;
  }

  /**
   * Ajusta o atributo valor de aviso; (ver {@link #warningValue})
   * 
   * @param warningValue o novo valor
   */
  public final void setWarningValue(final int warningValue) {
    this.warningValue = warningValue;
  }

  /**
   * Provoca o trmino (se existir) da thread de atualizao.
   */
  final public void shutdown() {
    alive = false;
    if (thread != null) {
      thread.interrupt();
    }
  }

  /**
   * Construtor
   * 
   * @param projectTree tree associada.
   * @param automatic indicativo que a atualizao ser feita de modo automtico
   *        e temporizado.
   */
  public ProjectOccupationProgressBar(final ProjectTree projectTree,
    final boolean automatic) {
    this.projectTree = projectTree;
    this.intervalSec = 60;
    this.setFont(new Font(Font.SANS_SERIF, Font.PLAIN, 10));

    addProjectTreeListeners();

    setBorderPainted(true);
    setStringPainted(true);

    /*
     * Usando o "look-and-feel" bsico para evitar que a barra aparea animada
     * em alguns sistemas operacionais (problema observado no MacOS).
     */
    setUI(new BasicProgressBarUI() {
      @Override
      protected Color getSelectionBackground() {
        return Color.black;
      }

      @Override
      protected Color getSelectionForeground() {
        return Color.black;
      }
    });
    if (automatic) {
      lastRefresh = System.currentTimeMillis();
      this.thread = createAutoThread();
      this.alive = true;
      this.thread.start();
    }
    else {
      this.alive = false;
      this.thread = null;
    }
    refresh(true);
  }
}
