package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.UserProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.Permission;
import csbase.logic.User;

/**
 * Janela que associa usurios a uma permisso.
 * 
 * @author Tecgraf
 */
public final class PermissionUserDefinitionDialog {

  private JFrame owner;
  private String dialogTitle;
  private DesktopComponentDialog mainDialog;
  private Container cp;
  private boolean wasChanged;

  private ContainerSelection<User> usersSelection;
  private JButton actionButton;
  private Permission permission;

  /**
   * Construtor.
   * 
   * @param owner Janela pai.
   */
  public PermissionUserDefinitionDialog(JFrame owner) {

    this.owner = owner;
    dialogTitle = LNG.get("IAS_PERMISSION_USERS_TITLE");
    mainDialog = new DesktopComponentDialog(this.owner, this.dialogTitle);
    cp = mainDialog.getContentPane();

    this.wasChanged = false;
  }

  public void showDialog(Permission permission) {

    this.permission = permission;

    Vector<User> users = getUsers();
    Vector<User> selectedUsers = getSelectedUsers(users);

    JPanel selectionPanel = createSelectionPanel(users, selectedUsers);
    cp.add(selectionPanel, BorderLayout.CENTER);
    this.wasChanged = false;

    createActionButton();
    JButton closeButton = createCloseButton();
    ClientUtilities.adjustEqualSizes(actionButton, closeButton);

    JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(actionButton);
    buttonsPanel.add(closeButton);

    cp.add(buttonsPanel, BorderLayout.SOUTH);

    mainDialog.pack();
    mainDialog.center(owner);
    mainDialog.setVisible(true);
  }

  private JButton createCloseButton() {
    JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (wasChanged) {
          Object[] options =
            { LNG.get("IAS_PERMISSION_UPDATE"), LNG.get("IAS_CANCEL") };
          int selected =
            JOptionPane.showOptionDialog(mainDialog,
              LNG.get("IAS_PERMISSION_UPDATE_QUESTION"), dialogTitle,
              JOptionPane.DEFAULT_OPTION, JOptionPane.INFORMATION_MESSAGE,
              null, options, options[0]);
          if (options[selected] == LNG.get("IAS_PERMISSION_UPDATE")) {
            updatePermissionsUserList();
          }
        }
        mainDialog.close();
      }
    });
    return closeButton;
  }

  private void createActionButton() {
    actionButton = new JButton(LNG.get("IAS_PERMISSION_UPDATE"));
    actionButton.setEnabled(false);
    actionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        updatePermissionsUserList();
        mainDialog.close();
      }
    });
  }

  private Vector<User> getUsers() {
    Vector<User> users = null;
    users =
      UserProxy.getAllUsers(mainDialog, dialogTitle, LNG.get("ERRO") + " - "
        + dialogTitle);

    Iterator<User> userIterator = users.iterator();
    while (userIterator.hasNext()) {
      User user = userIterator.next();
      if (user.getId().equals(User.getAdminId())) {
        users.remove(user);
        break;
      }
    }

    return users;
  }

  private Vector<User> getSelectedUsers(Vector<User> users) {
    Vector<User> selectedUsers = new Vector<User>();
    Iterator<User> userIterator = users.iterator();
    while (userIterator.hasNext()) {
      User user = userIterator.next();
      if (user.getPermission(this.permission.getId()) != null) {
        selectedUsers.add(user);
      }
    }
    return selectedUsers;
  }

  private JPanel createSelectionPanel(Vector<User> selectableUsers,
    Vector<User> selectedUsers) {
    ObjectTableProvider selectableFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        User user = (User) item;
        return (user == null) ? null : new String[] { user.getId().toString(),
            user.getName() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_USERS_KEY"),
            LNG.get("IAS_AVAILABLE_USERS") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class, String.class };
      }
    };
    ObjectTableProvider selectedFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        User user = (User) item;
        return (user == null) ? null : new String[] { user.getId().toString(),
            user.getName() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_USERS_KEY"),
            LNG.get("IAS_SELECTED_USERS") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class, String.class };
      }
    };
    this.usersSelection =
      new ContainerSelection<User>(selectableFormat, selectedFormat, true,
        false);
    this.usersSelection.loadItems(selectableUsers, selectedUsers);
    this.usersSelection.adjustTableColumns();
    this.usersSelection
      .addContainerSelectionListener(new ContainerSelectionListener() {
        @Override
        public void containerChanged() {
          wasChanged = true;
          actionButton.setEnabled(true);
        }
      });
    return usersSelection.getPanel();
  }

  private void updatePermissionsUserList() {

    boolean success = true;

    try {
      List<User> usersAvailable = usersSelection.getAvailableItems();
      for (User user : usersAvailable) {
        user.removePermission(permission);
      }
      UserProxy.modifyUsers(owner, dialogTitle,
        LNG.get("IAS_WAITING_USER_MODIFICATION"), usersAvailable);

      List<User> usersSelected = usersSelection.getSelectedItems();
      for (User user : usersSelected) {
        user.addPermission(permission);
      }
      UserProxy.modifyUsers(owner, dialogTitle,
        LNG.get("IAS_WAITING_USER_MODIFICATION"), usersSelected);

      String msg =
        LNG.get(success ? "IAS_PERMISSION_MODIFIED_WITH_SUCCESS"
          : "IAS_PERMISSION_MODIFIED_WITH_ERROR");
      msg = MessageFormat.format(msg, new Object[] { permission.getName() });
      StandardDialogs.showInfoDialog(owner, dialogTitle, msg);

      wasChanged = false;
    }
    catch (Exception e) {
      success = false;
      StandardErrorDialogs.showErrorDialog(owner, LNG.get("ERRO") + " - "
        + dialogTitle, e);
    }
  }
}
