/*
 * ProjectTreePath.java
 * 
 * $Id: ProjectTreePath.java 169707 2015-11-12 10:52:44Z lmoreira $
 */
package csbase.client.project;

import csbase.logic.ClientProjectFile;

/**
 * A classe <code>ProjectTreePath</code> modela um caminho de um arquivo na
 * rvore de projetos. Um objeto dessa classe pode representar tanto um arquivo
 * (ou diretrio) existente, ou um arquivo inexistente, ou seja, que ainda no
 * foi criado. Nesse ltimo caso, o mtodo <code>exists</code> retorna
 * verdadeiro.
 * 
 * @author Julia
 * @version $Revision: 169707 $
 */
public class ProjectTreePath {
  /** O arquivo que esse caminho representa */
  private ClientProjectFile file;
  /** Um caminho na rvore de projeto, no caso do arquivo no existir. */
  private String[] path;
  /**
   * O diretrio ao qual o arquivo pertence (ou vai pertencer, no caso dele no
   * existir.
   */
  private ClientProjectFile parent;

  /**
   * Se o caminho representa um arquivo existe.
   * 
   * @return True, se o arquivo existe ou falso, caso contrrio.
   */
  public boolean exists() {
    return (file != null);
  }

  /**
   * Devolve um array com os caminho completo do nome do arquivo representado,
   * incluindo o nome de todos os sub-diretrios onde ele est.
   * 
   * @return Um array com o caminho do arquivo na rvore de projeto.
   */
  public String[] getPath() {
    return (file == null) ? path : file.getPath();
  }

  /**
   * Devolve o arquivo representado por esse caminho, no caso dele existir. Se o
   * arquivo no existe, devolve null.
   * 
   * @return O arquivo ou null se ele no existe.
   */
  public ClientProjectFile getFile() {
    return file;
  }

  /**
   * Devolve o diretrio ao qual pertence o arquivo aqui representado.
   * 
   * @return O diretrio ao qual esse arquivo pertence ou vai pertencer, no caso
   *         do arquivo ainda no existir.
   */
  public ClientProjectFile getParent() {
    return parent;
  }

  /**
   * Devolve o nome do arquivo.
   * 
   * @return O nome do arquivo.
   */
  public String getName() {
    return path[path.length - 1];
  }
  
  /**
   * Montagem de um caminho de diretrios, usando um separador especificado como
   * parmetro.
   * 
   * @param sep separador de diretrios
   * @param path o caminho do diretrio.
   * 
   * @return uma string com o path completo.
   */
  public static String makePath(String sep, String[] path) {
    if ((sep == null) || (path == null) || (path.length <= 0)) {
      return "";
    }
    String dir = "";
    for (int i = 0; i < (path.length - 1); i++) {
      dir = dir + path[i] + sep;
    }
    dir = dir + path[path.length - 1];
    return dir;
  }

  /**
   * Montagem de um caminho de diretrios, usando um projeto e um separador
   * especificados como parmetros.
   * 
   * @param project nome do projeto
   * @param sep separador de diretrios
   * @param path o caminho do diretrio.
   * 
   * @return uma string com o path completo.
   */
  public static String makePath(String project, String sep, String[] path) {
    if ((sep == null) || (project == null)) {
      return "";
    }
    String dir = "[" + project + "] : " + makePath("/", path);
    return dir;
  }

  /**
   * Compara se dois ProjectTreePath so iguais, atravs do path total. Retorna
   * true se todos os elementos do path so iguais, caso contrrio retorn false.
   * 
   * @param o .
   * 
   * @return .
   */
  @Override
  public boolean equals(Object o) {
    if (!(o instanceof ProjectTreePath)) {
      return false;
    }
    String[] path = ((ProjectTreePath) o).getPath();
    String[] thisPath = this.getPath();
    if (thisPath.length != path.length) {
      return false;
    }
    for (int i = 0; i < thisPath.length; i++) {
      if (!thisPath[i].equals(path[i])) {
        return false;
      }
    }
    return true;
  }

  /**
   * Calcula o cdigo hash do objeto.
   * 
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    String[] thisPath = this.getPath();

    if (thisPath == null || thisPath.length == 0 || thisPath[0] == null) {
      return "path attribute null or zero".hashCode();
    }

    return thisPath[0].hashCode();
  }

  @Override
  public String toString() {
    String toString = "";
    String separator = "";
    String[] path = getPath();
    for (int i = 0; i < path.length; i++) {
      toString += separator;
      toString += path[i];
      separator = "/";
    }
    return toString;
  }

  /**
   * Contri um <code>ProjectTreePath</code> com um arquivo existente.
   * 
   * @param file O arquivo sobre o qual esse objeto  construdo.
   */
  public ProjectTreePath(ClientProjectFile file) {
    this.file = file;
    this.parent = file.getParent();
  }

  /**
   * Contri um <code>ProjectTreePath</code> com um arquivo inexistente.
   * 
   * @param path O caminho do arquivo que ainda no existe.
   * @param dir O diretrio no qual esse arquivo vai ser criado.
   */
  public ProjectTreePath(String[] path, ClientProjectFile dir) {
    this.path = path;
    this.parent = dir;
  }
}
