/*
 * $Id: CommonFileUploadAction.java 169707 2015-11-12 10:52:44Z lmoreira $
 */
package csbase.client.project.action;

import java.awt.event.ActionEvent;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.text.MessageFormat;

import javax.swing.JFileChooser;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.RemoteTask;
import csbase.client.externalresources.ExternalResources;
import csbase.client.project.ProjectFileContainer;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.client.remote.srvproxies.HttpProxy;
import csbase.client.remote.srvproxies.NotificationProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.User;

/**
 * A ao de importao quando executada a partir de um applet, redireciona para
 * um browser. Quando executada como aplicao, abre um FileChooser.
 */
public class CommonFileUploadAction extends CommonProjectAction {
  /**
   * Tamanho do bloco a ser transferido, no caso de aplicao.
   */
  private static final int BLOCK_SIZE = 100 * 1024; // 100 Kb

  /**
   * Diretrio selecionado
   */
  private ClientProjectFile dir;

  /**
   * Indica se o arquivo a ser publicado foi originado da rvore do projeto ou
   * no
   */
  private boolean isFileFromTree;

  /**
   * Diretrio para o qual o arquivo, no originado da rvore do projeto, vai
   * ser importado
   */
  private ClientProjectFile importDir;

  /**
   * Construtor.
   * 
   * @param container o container
   */
  public CommonFileUploadAction(ProjectFileContainer container) {
    super(container);
    this.isFileFromTree = true;
  }

  /**
   * Constri a ao a partir de um diretrio que no est selecionado na rvore
   * do projeto.
   * 
   * @param container rvore do projeto.
   * @param file Diretrio sobre o qual a ao ser realizada.
   */
  public CommonFileUploadAction(ProjectFileContainer container,
    ClientProjectFile file) {
    super(container);
    this.isFileFromTree = false;
    this.importDir = file;
  }

  /**
   * Inicia o processo de upload de um arquivo. Quando o cliente  executado
   * como applet, redireciona para o browser. Quando  executado como aplicao
   * abre o FileChooser.
   * 
   * @param e .
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    this.dir =
      (isFileFromTree) ? (ClientProjectFile) getSelectedFile() : this.importDir;
    if (ExternalResources.getInstance().isEnabled()) {
      jnlpOpenBrowser();
    }
    else {
      transferFile();
    }
  }

  /**
   * Obtm a url para upload do arquivo e redireciona para um browser usando a
   * API JNLP.
   */
  private void jnlpOpenBrowser() {
    try {
      CommonClientProject project = getProject();
      String addr =
        HttpProxy.getUploadURL(project.getId(), dir.getPath(),
          ProjectFileType.UNKNOWN);
      URL url = new URL(addr);
      ExternalResources.getInstance().showDocument(url);
    }
    catch (Exception e) {
      showError(LNG.get("PRJ_PROJECT_FILE_UPLOAD_URL_ERROR"), e);
    }
  }

  /**
   * Copia um arquivo local para a area de projetos quando o sistema est sendo
   * executado como aplicao.
   */
  private void transferFile() {
    JFileChooser chooser = new JFileChooser();
    int returnVal = chooser.showOpenDialog(getWindow());
    if (returnVal == JFileChooser.CANCEL_OPTION) {
      return;
    }
    File fileChoosed = chooser.getSelectedFile();
    String fileName = fileChoosed.getName();
    if (!ClientUtilities.isValidFileName(fileName)) {
      showError(LNG.get("PRJ_PROJECT_FILE_NAME_CHARACTERES_ERROR"));
      return;
    }

    // Verifica se j existe algum arquivo com o nome dado
    final ClientProjectFile child = GetChildFromNameTask.runTask(dir, fileName);
    if (child != null) {
      showError(LNG.get("PRJ_PROJECT_FILE_UPLOAD_FILE_ERROR"));
      return;
    }
    ClientProjectFile newFile = createFile(fileName);
    if (newFile == null) {
      showError(LNG.get("PRJ_PROJECT_FILE_UPLOAD_FILE_ERROR"));
      return;
    }

    //Inicia a transferncia e informa ao usurio que o upload comeou
    new Uploader(fileChoosed, newFile).start();
    String msg =
      MessageFormat.format(LNG.get("PRJ_PROJECT_FILE_UPLOAD_START"),
        new Object[] { fileName });
    StandardDialogs.showInfoDialog(getWindow(),
      LNG.get("PRJ_PROJECT_FILE_UPLOAD_TITLE"), msg);
  }

  /**
   * Cria um arquivo com o nome passado como parmetro. Retorna null caso ocorra
   * algum erro na criao.
   * 
   * @param fileName nome do arquivo a ser criado
   * 
   * @return arquivo criado ou null caso haja alguma falha na criao
   */
  private ClientProjectFile createFile(final String fileName) {
    final CommonClientProject project = getProject();
    RemoteTask<ClientProjectFile> task = new RemoteTask<ClientProjectFile>() {
      @Override
      public void performTask() throws Exception {
        String fileType = ProjectFileType.UNKNOWN;
        ClientProjectFile createdFile =
          project.createFile(dir, fileName, fileType);
        setResult(createdFile);
      }
    };
    String message =
      MessageFormat.format(LNG.get("PRJ_WAITING_UPLOAD_FILE"),
        new Object[] { fileName });
    if ((task.execute(getWindow(), getTitle(), message))
      && (task.getResult() != null)) {
      return task.getResult();
    }
    return null;
  }

  /**
   * Mtodo de aviso de erros de sistema.
   * 
   * @param msg mensagem a ser exibida.
   */
  @Override
  protected void showError(String msg) {
    showError(LNG.get("PRJ_PROJECT_FILE_UPLOAD_TITLE"), msg);
  }

  /**
   * Mtodo de aviso de erros de usurio. A mensagem do erro  exibida junto com
   * a pilha.
   * 
   * @param msg mensagem a ser exibida.
   * @param ex exceo
   */
  private void showError(String msg, Exception ex) {
    StandardErrorDialogs.showErrorDialog(getWindow(),
      LNG.get("PRJ_PROJECT_FILE_UPLOAD_TITLE"), msg, ex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return LNG.get("PRJ_UPLOAD_FILE");
  }

  /**
   * A classe <code>Uploader</code>  responsvel por fazer a transferncia do
   * arquivo quando o cliente  executado com aplicao.
   */
  private class Uploader extends Thread {
    /**
     * Alvo
     */
    private ClientProjectFile targetFile;

    /**
     * Origam
     */
    private File sourceFile;

    /**
     * Construtor
     * 
     * @param sourceFile origem
     * @param targetFile destino.
     */
    public Uploader(File sourceFile, ClientProjectFile targetFile) {
      this.targetFile = targetFile;
      this.sourceFile = sourceFile;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      try {
        InputStream in =
          new BufferedInputStream(new FileInputStream(sourceFile));
        targetFile.upload(in, BLOCK_SIZE, null);
        String msg =
          MessageFormat.format(LNG.get("PRJ_PROJECT_FILE_UPLOAD_SUCCESS"),
            new Object[] { targetFile.getName() });
        Object[] ids = new Object[] { User.getLoggedUser().getId() };
        NotificationProxy.notifyTo(ids, msg, false, false);
      }
      catch (Exception ex) {
        showError(LNG.get("PRJ_PROJECT_FILE_UPLOAD_IO_ERROR"), ex);
        return;
      }
    }
  }
}
