/**
 * $Id: CommandObserver.java 149483 2014-02-10 12:55:12Z pietroguedes $
 */

package csbase.client.remote;

import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.LinkedList;
import java.util.Observable;

import csbase.client.algorithms.commands.CommandListener;
import csbase.logic.CommandEvent;
import csbase.remote.RemoteEvent;
import csbase.remote.RemoteObserver;
import csbase.remote.SGAServiceInterface;

/**
 * Permite aos clientes observar mudanas nos comandos em execuo
 * 
 * @author Tecgraf/PUC-Rio
 * @deprecated O estados dos comandos so enviados diretamente dos servios para
 *             as aplicaes clientes via MessageService. Este observador no
 *             tem mais utilizada e dever ser removido.
 */
@Deprecated
public class CommandObserver extends Observable implements RemoteObserver {
  /**
   * A instncia do observador de comandos.
   */
  private static CommandObserver instance;

  /**
   * Os observadores locais dos comandos.
   */
  private LinkedList<CommandListener> listeners;

  /**
   * Cria o observador de comandos.
   */
  private CommandObserver() {
    listeners = new LinkedList<CommandListener>();
  }

  /**
   * <p>
   * Obtm o singleton.
   * </p>
   * 
   * <p>
   * Ele no cria o singleton. Para cri-lo, utilize o mtodo {@link #start()}.
   * </p>
   * 
   * @return O singleton.
   */
  public static CommandObserver getInstance() {
    if (instance == null) {
      throw new IllegalStateException(
        "A instncia do observador no foi criada.\n"
          + "Chame o mtodo start primeiro.\n");
    }
    return instance;
  }

  /**
   * Cria um singleton desta classe.
   * 
   * @throws RemoteException Em caso de erro de RMI.
   */
  public static void start() throws RemoteException {
    if (instance != null) {
      return;
    }
    instance = new CommandObserver();
    instance.install();
  }

  /**
   * Desinstala o observador do registro do RMI.
   * 
   * @throws RemoteException Em caso de erro de RMI.
   */
  public static void stop() throws RemoteException {
    if (instance == null) {
      return;
    }
    instance.uninstall();
  }

  /**
   * Adiciona o observador no servidor para o identificador do comando dado.
   * 
   * @param commandId Identificador do comando.
   */
  public void addCommandObserver(Object commandId) {
    ClientRemoteMonitor.getInstance().addObserver(
      SGAServiceInterface.SERVICE_NAME, this, commandId);
  }

  /**
   * Adiciona um observador local a este observador remoto.
   * 
   * @param listener O observador local (No aceita {@code null}).
   */
  public void addListener(CommandListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Remove o observador no servidor para o identificador do comando dado.
   * 
   * @param commandId Identificador do comando.
   */
  public void deleteCommandObserver(Object commandId) {
  }

  /**
   * Registra no servidor um observador.
   */
  @Override
  public void install() throws RemoteException {
    UnicastRemoteObject.exportObject(this, 0);
  }

  /**
   * Remove um observador local deste observador remoto.
   * 
   * @param listener O observador local (No aceita {@code null}).
   * 
   * @return {@code false} se o observador a ser removido no estava cadastrado
   *         neste observador remoto.
   */
  public boolean removeListener(CommandListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }

  /**
   * Remove do servidor o registro desse observador.
   */
  @Override
  public void uninstall() throws RemoteException {
    UnicastRemoteObject.unexportObject(this, true);
  }

  /**
   * Trata os eventos que notificam uma mudana no comando em execuo. Eventos
   * e informaes adicionais : - comando terminado . tempo gasto (elapsed, user
   * e cpu) - comando abortado . nenhuma informao adicional
   * 
   * @param event Descrio do evento
   */
  @Override
  public void update(RemoteEvent event) {
    CommandEvent action = (CommandEvent) event;
    String commandId = (String) action.getCommandId();
    switch (action.getType()) {
      case COMPLETED:
        fireCommandWasCompleted(commandId);
        break;
      case SUCCESS:
        fireCommandExecutionSucceded(commandId);
        break;
      case ERROR:
        fireCommandExecutionFailure(commandId);
        break;
      case INIT_FAILURE:
        fireCommandInitFailure(commandId);
        break;
      case KILLED:
        fireCommandWasKilled(commandId);
        break;
      case LOST:
        fireCommandWasLost(commandId);
        break;
      case NO_CODE:
        fireCommandWithNoExitCode(commandId);
    }
    setChanged();
    notifyObservers(action);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void update(RemoteEvent[] events) throws RemoteException {
    for (int i = 0; i < events.length; i++) {
      update(events[i]);
    }
  }

  /**
   * Dispara o evento {@link CommandListener#initFailure(String)}.
   * 
   * @param commandId O identificador do comando (No aceita {@code null}).
   */
  private void fireCommandInitFailure(String commandId) {
    for (CommandListener listener : listeners) {
      listener.initFailure(commandId);
    }
  }

  /**
   * Dispara o evento {@link CommandListener#executionFailure(String)}.
   * 
   * @param commandId id do comando.
   */
  private void fireCommandExecutionFailure(String commandId) {
    for (CommandListener listener : listeners) {
      listener.executionFailure(commandId);
    }
  }

  /**
   * Dispara o evento {@link CommandListener#executionSucceded(String)}.
   * 
   * @param commandId id do comando.
   */
  private void fireCommandExecutionSucceded(String commandId) {
    for (CommandListener listener : listeners) {
      listener.executionSucceded(commandId);
    }
  }

  /**
   * Dispara o evento {@link CommandListener#wasCompleted(String)}.
   * 
   * @param commandId O identificador do comando (No aceita {@code null}).
   */
  private void fireCommandWasCompleted(String commandId) {
    for (CommandListener listener : listeners) {
      listener.wasCompleted(commandId);
    }
  }

  /**
   * Dispara o evento {@link CommandListener#wasKilled(String)}.
   * 
   * @param commandId O identificador do comando (No aceita {@code null}).
   */
  private void fireCommandWasKilled(String commandId) {
    for (CommandListener listener : listeners) {
      listener.wasKilled(commandId);
    }
  }

  /**
   * Dispara o evento {@link CommandListener#wasLost(String)}.
   * 
   * @param commandId O identificador do comando (No aceita {@code null}).
   */
  private void fireCommandWasLost(String commandId) {
    for (CommandListener listener : listeners) {
      listener.wasLost(commandId);
    }
  }

  /**
   * Dispara o evento {@link CommandListener#noExitCode(String)}.
   * 
   * @param commandId O identificador do comando (No aceita {@code null}).
   */
  private void fireCommandWithNoExitCode(String commandId) {
    for (CommandListener listener : listeners) {
      listener.noExitCode(commandId);
    }
  }
}
