/*
 * $Id: UsersChooser.java 150777 2014-03-19 14:16:56Z oikawa $
 */

package csbase.client.util;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import csbase.client.desktop.RemoteTask;
import csbase.logic.User;
import csbase.logic.UserOutline;

/**
 * A classe <code>UsersChooser</code> define a interface para escolha de um
 * grupo de usurios. O construtor recebe os usurios j selecionados para
 * mostrar no lado direito do painel. A lista do lado direito exibe todos os
 * usurio menos aqueles j selecionados. Para usar o <code>UsersChooser</code>,
 * devemos instanci-lo. Em seguida, chamar o mtodo <code>start()</code>. Para
 * obter o resultado, verificamos se a seleo foi confirmada chamando o mtodo
 * <code>wasConfirmed()</code>. o array de usurios selecionados  obtido pelo
 * mtodo <code>getSelectedUsers</code>.
 */
public class UsersChooser extends JDialog {

  /**
   * Janela onde o painel ser colocado
   */
  private JFrame owner;

  /**
   * Usurios selecionados
   */
  private Vector<UserOutline> selectedUsers;

  /**
   * Componente de seleo dos usurios
   */
  private UserContainerSelection containerSelection = null;

  /**
   * Indicativo de confirmao
   */
  private boolean wasConfirmed;

  /**
   * Boto
   */
  protected JButton confirmButton;

  /**
   * Construtor
   * 
   * @param owner janela
   * @param title ttulo
   * @param selectedUsers usurios selecionados.
   */
  public UsersChooser(JFrame owner, String title,
    Vector<UserOutline> selectedUsers) {
    super(owner, title, true);
    this.owner = owner;
    this.selectedUsers = selectedUsers;
    this.wasConfirmed = false;
  }

  /**
   * Inicia o dilogo.
   */
  public void start() {
    JPanel mainPanel = createGUI();

    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        cancel();
      }
    });
    setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);

    getContentPane().setLayout(new BorderLayout(0, 0));
    getContentPane().add(mainPanel);
    pack();
    ClientUtilities.centerWindow(this, owner);

    postInitialization();
    setVisible(true);
  }

  /**
   * Cria o painel de seleo dos usurios.
   * 
   * @return o painel.
   */
  public JPanel createGUI() {
    JPanel mainPanel = new JPanel(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.anchor = GridBagConstraints.WEST;

    constraints.gridy = 0;

    /* Algumas interfaces de edio tem um painel superior, acima da tabela */
    JPanel topPanel = createOptionalTopPanel();
    if (topPanel != null) {
      constraints.gridx = 0;
      constraints.gridy++;
      constraints.weightx = 0;
      constraints.weighty = 0;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.insets = new Insets(5, 10, 0, 10);
      mainPanel.add(topPanel, constraints);
    }

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.weightx = 1.0;
    constraints.weighty = 0;
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.anchor = GridBagConstraints.CENTER;
    constraints.insets = new Insets(5, 10, 0, 10);
    mainPanel.add(createSelectablePanel(), constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    constraints.weightx = 0;
    constraints.weighty = 0;
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.CENTER;
    constraints.insets = new Insets(0, 0, 0, 0);
    mainPanel.add(createButtonsPanel(), constraints);
    return mainPanel;
  }

  /**
   * Cria o painel superior opcional, localizado acima da tabela de elementos. A
   * implementao default indica que no h painel superior. Deve ser
   * sobrescrita por especializaes que contenham esse painel.
   * 
   * @return o painel superior, ou null
   */
  protected JPanel createOptionalTopPanel() {
    return null;
  }

  /**
   * Cria o painel com o componente que permite selecionar os usurios Usa um
   * componente que mostra dois conjuntos, disponveis e escolhidos, nos quais o
   * usurio pode movimentar elementos entre eles.
   * 
   * @return o painel com o componente que permite selecionar os usurios
   */
  private JPanel createSelectablePanel() {
    List<UserOutline> availableUsers = retrieveAllUsers();
    //Remove o usurio logado da lista
    for (int i = 0; i < availableUsers.size(); i++) {
      Object userId = availableUsers.get(i).getId();
      if (userId.equals(User.getLoggedUser().getId())) {
        availableUsers.remove(i);
        break;
      }
    }
    containerSelection =
      new UserContainerSelection(true, new Vector<UserOutline>(availableUsers),
        selectedUsers);
    return containerSelection.getPanel();
  }

  /**
   * Cria o painel com os botes de confirmar e cancelar.
   * 
   * @return o painel com os botes de confirmar e cancelar.
   */
  private JPanel createButtonsPanel() {
    JPanel panel = new JPanel();

    confirmButton = new JButton(LNG.get("UsersChooser.confirm"));
    confirmButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        confirm();
      }
    });
    //Adiciona um listener ao boto de confirmar para saber se a seleo mudou
    containerSelection
      .addContainerSelectionListener(new ContainerSelectionListener() {
        @Override
        public void containerChanged() {
          confirmButton.setEnabled(true);
        }
      });
    confirmButton.setEnabled(false);
    panel.add(confirmButton);

    JButton cancelButton = new JButton(LNG.get("UsersChooser.cancel"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        cancel();
      }
    });
    panel.add(cancelButton);
    return panel;
  }

  /**
   * Ps Inicializao da interface. A implementao default  vazia.
   */
  protected void postInitialization() {
  }

  /**
   * Recupera no servidor todos os usurios disponveis.
   * 
   * @return O vetor com todos os usurios disponveis, ou null caso ocorra
   *         alguma falha
   */
  private List<UserOutline> retrieveAllUsers() {
    RemoteTask<List<UserOutline>> task = new RemoteTask<List<UserOutline>>() {
      @Override
      public void performTask() throws Exception {
        setResult(User.getAllOutlines());
      }
    };
    boolean execute =
      task.execute(owner, getTitle(), LNG.get("UsersChooser.info.get.users"));
    if (execute) {
      return task.getResult();
    }
    return null;
  }

  /**
   * Indica se a operao foi confirmada.
   * 
   * @return indicativo
   */
  public boolean wasConfirmed() {
    return wasConfirmed;
  }

  /**
   * Atuao de conformao.
   */
  private void confirm() {
    wasConfirmed = true;
    setVisible(false);
  }

  /**
   * Consulta  lista de usurios selecionados.
   * 
   * @return lista
   */
  public Object[] getSelectedUsers() {
    return containerSelection.getSelectedItems();
  }

  /**
   * Ao de cancelamento
   */
  private void cancel() {
    wasConfirmed = false;
    setVisible(false);
  }

  /**
   * Ajuste de habilitao de seleo no painel.
   * 
   * @param enable indicativo
   */
  public void setEnabledSelectablePanel(boolean enable) {
    containerSelection.getContainerSelection().setEnabled(enable);
  }

}
