package csbase.client.util.csvpanel;

import java.awt.Window;
import java.nio.charset.Charset;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.parsers.csv.CsvParser;
import csbase.client.Client;
import csbase.client.desktop.LocalTask;
import csbase.client.util.iostring.TextWriter;
import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.logic.ClientFile;

/**
 * Classe utilitria para leitura/escrita de arquivos CSV.
 */
public class CSVFileUtils {

  /**
   * Escreve o arquivo CSV com o contedo indicado na tabela de dados.
   * 
   * @param owner janela dona da operao.
   * @param contents contedo do arquivo em formato tabela.
   * @param file o arquivo CSV.
   * @throws OperationFailureException em caso de erro na escrita do arquivo.
   */
  public static void writeCSVFile(Window owner, String[][] contents,
    ClientFile file) throws OperationFailureException {
    try {
      String content = generateCSVFileContent(contents);
      Client client = Client.getInstance();
      Charset charset = client.getSystemDefaultCharset();
      TextWriter.writeAllWithTask(owner, file, content, charset);
    }
    catch (Exception e) {
      throw new OperationFailureException(e);
    }
  }

  /**
   * Gera o contedo do arquivo CSV a partir da tabela de dados.
   * 
   * @param contents contedo do arquivo em formato tabela.
   * @return o contedo do arquivo de configurao em formato String.
   */
  public static String generateCSVFileContent(String[][] contents) {
    StringBuilder content = new StringBuilder();
    final String lineSeparator = "\n";
    final String valueSeparator = ",";
    final String quote = "\"";
    for (String[] values : contents) {
      for (String value : values) {
        if (value != null) {
          content.append(quote);
          content.append(value);
          content.append(quote);
        }
        content.append(valueSeparator);
      }
      content.append(lineSeparator);
    }
    return content.toString();
  }

  /**
   * L o contedo do arquivo CSV para uma tabela de strings.
   * 
   * @param owner janela dona da operao.
   * @param file o arquivo CSV.
   * @return o mapeamento dos parmetros resultante da configurao.
   * 
   * @throws OperationFailureException em caso de erro na leitura do arquivo.
   */
  public static String[][] readCSVFile(Window owner, final ClientFile file)
    throws OperationFailureException {
    Client client = Client.getInstance();
    final String lngPrefix = CSVFileUtils.class.getName();
    final Charset charset = client.getSystemDefaultCharset();
    LocalTask<String[][]> task = new LocalTask<String[][]>() {
      @Override
      protected void performTask() throws Exception {
        CsvParser parser = new CsvParser();
        String[][] lines = parser.parse(file.getInputStream(), charset);
        if (lines.length > 0) {
          int numParams = lines[0].length;
          for (int i = 1; i < lines.length; i++) {
            if (lines[i].length != numParams) {
              /*
               * Todas as linhas do arquivo precisam ter o mesmo nmero de itens
               * para serem vlidas.
               */
              throw new ParseException(LNG.get(lngPrefix
                + ".read.task.invalidLine.error", new Object[] { i + 1,
                  lines[i].length, numParams }));
            }
          }
        }
        setResult(lines);
      }
    };
    if (!task.execute(owner, LNG.get(lngPrefix + ".read.task.title"),
      LNG.get(lngPrefix + ".read.task.message"))) {
      throw new OperationFailureException(task.getError());
    }
    return task.getResult();
  }
}
