package csbase.client.util.sga;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.algorithms.tasks.ConfiguratorValidationTask;
import csbase.client.algorithms.validation.ValidationTranslator;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.remote.srvproxies.SchedulerProxy;
import csbase.client.util.ClientUtilities;
import csbase.logic.CommandInfo;
import csbase.logic.CommandSubmission;
import csbase.logic.CommonClientProject;
import csbase.logic.Priority;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * O Dilogo de Execuo de Comando  um dilogo que permite ao usurio escolher
 * em qual servidor sga executar o seu algoritmo e executar o comando.
 *
 * @author valeria
 */
public class CommandExecutionDialog extends DesktopComponentDialog {
  /**
   * Nome da propriedade que habilita a escolha de prioridade.  usada apenas
   * nos sistemas que possuem fila de prioridades.
   */
  private static final String ENABLE_PRIORITY = "enable.priority";

  /**
   * Painel seletor de SGAs.
   */
  private SGASelectionPanel sgaPanel;

  /**
   * Boto Executar.
   */
  private JButton executeButton;

  /**
   * Boto Fechar.
   */
  private JButton closeButton;

  /**
   * Rtulo Descrio do Comando.
   */
  private JLabel commandDescriptionLabel;

  /**
   * Campo de Texto Descrio do Comando.
   */
  private JTextField commandDescriptionTextField;

  /**
   * Combo com as descries escolhidas em execues anteriores.
   */
  private JComboBox previousCommandDescriptionComboBox;

  /**
   * CheckBox Notificar Trmino por E-mail.
   */
  private JCheckBox emailCheckBox;

  /**
   * Painel de botes.
   */
  private JPanel buttonPanel;

  /**
   * Rtulo Status dos parmetros do configurador.
   */
  private JLabel parameterStatusLabel;

  /**
   * Rtulo Indicativo de SGA corretamento escolhido/selecionado.
   */
  private JLabel sgaStatusLabel;

  /**
   * Rtulo Prioridade.
   */
  private JLabel priorityLabel;

  /**
   * ComboBox Prioridade.
   */
  private JComboBox priorityComboBox;

  /**
   * Projeto.
   */
  private final CommonClientProject project;

  /**
   * O configurador de algoritmos selecionado.
   */
  private final AlgorithmConfigurator configurator;

  /**
   * Flag que indica se a opo de escolha de prioridade de um comando deve ser
   * exibida.
   */
  private boolean enablePriority;

  /**
   * Listeners interessados nos comandos solicitados com esse dilogo
   */
  private final List<CommandRequestedListener> listeners;

  /**
   * Histrico de descries dos comandos executados.
   */
  private CommandDescriptionHistory commandDescriptionHistory;

  /**
   * Resultado da validao dos parmetros do configurador.
   */
  protected Validation validationResult;

  /**
   * Cria um Dilogo de Execuo de Comandos.
   *
   * @param parentFrame A janela que est requisitando este configurador (No
   *        pode ser nulo).
   * @param configurator O configurador do algoritmo a ser executado.
   * @param project O projeto.
   */
  public CommandExecutionDialog(JFrame parentFrame,
    AlgorithmConfigurator configurator, CommonClientProject project) {
    super(parentFrame);
    listeners = new LinkedList<CommandRequestedListener>();
    this.project = project;
    this.configurator = configurator;
    this.commandDescriptionHistory = new CommandDescriptionHistory();
    Configuration configuration;
    try {
      configuration =
        ConfigurationManager.getInstance().getConfiguration(this.getClass());
      this.enablePriority =
        configuration.getOptionalBooleanProperty(ENABLE_PRIORITY, false);
    }
    catch (ConfigurationManagerException e) {
      this.enablePriority = false;
    }
  }

  /**
   * Inicializa a janela.
   */
  private void initUI() {
    this.validationResult = validateCommand();
    createGui();
    createSGASelectionPanel();
    updateParameterStatus();
    updateCanExecute();
    populatePanel();
    String commandDescription = configurator.getCommandDescription();
    if (commandDescription != null) {
      setCommandDescription(commandDescription);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean visible) {
    if (!isVisible()) {
      initUI();
    }
    super.setVisible(visible);
  }

  /**
   * Valida os parmetros do configurador.
   *
   * @return o resutlado da validao.
   */
  protected Validation validateCommand() {
    Validation result =
      ConfiguratorValidationTask.runTask(this, configurator,
        ValidationMode.FULL);
    return result;
  }

  /**
   * Adiciona um observador ao dilogo.
   *
   * @param listener o observador.
   */
  public void addCommandRequestedListener(CommandRequestedListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Remove um observador do dilogo.
   *
   * @param listener o observador.
   * @return true se aquele listener estava cadastrado.
   */
  public boolean removeCommandRequestedListener(
    CommandRequestedListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }

  /**
   * Cria a interface grfica do configurador de seleo.
   */
  private void createGui() {
    this.executeButton =
      new JButton(LNG.get("CommandExecutionDialog.button.execute"));
    this.executeButton.setEnabled(false);
    this.executeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        execute();
        close();
      }
    });
    this.closeButton =
      new JButton(LNG.get("CommandExecutionDialog.button.close"));
    this.closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        close();
      }
    });
    this.buttonPanel = new JPanel(new FlowLayout());
    this.buttonPanel.add(this.executeButton);
    this.buttonPanel.add(this.closeButton);
    ClientUtilities.adjustEqualSizes(executeButton, closeButton);
    this.commandDescriptionLabel =
      new JLabel(LNG.get("CommandExecutionDialog.label.command_description"));
    this.commandDescriptionTextField = new JTextField();
    this.addWindowFocusListener(new WindowAdapter() {
      @Override
      public void windowGainedFocus(WindowEvent e) {
        commandDescriptionTextField.requestFocusInWindow();
        commandDescriptionTextField.selectAll();
      }
    });
    String[] previousDescriptions = loadPreviousDescriptions();
    this.previousCommandDescriptionComboBox =
      new JComboBox(previousDescriptions);
    this.previousCommandDescriptionComboBox.insertItemAt(LNG
      .get("CommandExecutionDialog.combobox.previous_command_description"), 0);
    this.previousCommandDescriptionComboBox.setSelectedIndex(0);
    this.previousCommandDescriptionComboBox
    .addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (previousCommandDescriptionComboBox.getSelectedIndex() != 0) {
          setCommandDescription(previousCommandDescriptionComboBox
            .getSelectedItem().toString());
        }
      }
    });
    if (previousDescriptions.length > 0) {
      setCommandDescription(previousDescriptions[0]);
    }
    else {
      previousCommandDescriptionComboBox.setEnabled(false);
    }

    if (this.enablePriority) {
      this.priorityLabel =
        new JLabel(LNG.get("CommandExecutionDialog.label.priority"));
      String[] priorityArray = new String[Priority.values().length];
      for (Priority p : Priority.values()) {
        priorityArray[p.ordinal()] =
          LNG.get("CommandExecutionDialog.combobox." + p.name());
      }
      this.priorityComboBox = new JComboBox(priorityArray);
      this.priorityComboBox.setSelectedIndex(Priority.getDefault().ordinal());
    }
    this.emailCheckBox =
      new JCheckBox(LNG.get("CommandExecutionDialog.checkbox.email"));
    this.parameterStatusLabel = new JLabel();

    this.sgaStatusLabel = new JLabel();

    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });
    setModal(false);
  }

  /**
   * Atualiza o status da configurao dos parmetros.
   */
  protected void updateParameterStatus() {
    if (!validationResult.isWellSucceded()) {
      String message = ValidationTranslator.translateMessage(validationResult);
      setParameterStatus(message, ApplicationImages.ICON_ERROR_16);
    }
    else if (this.configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
      FlowAlgorithmConfigurator flowConfigurator =
        (FlowAlgorithmConfigurator) this.configurator;
      if (flowConfigurator.hasBypassedNodes()) {
        if (!flowConfigurator.canBeExecuted()) {
          String text =
            LNG.get("CommandExecutionDialog.label.bypass_not_possible_status");
          setParameterStatus(text, ApplicationImages.ICON_ERROR_16);
        }
        else {
          String text =
            LNG.get("CommandExecutionDialog.label.bypassed_nodes_status");
          setParameterStatus(text, ApplicationImages.ICON_INFORMATION_16);
        }
      }
      else {
        setParameterStatus(null, null);
      }
    }
    else if (this.configurator.isSetDefaultValues()) {
      String text =
        LNG.get("CommandExecutionDialog.label.default_parameter_status");
      setParameterStatus(text, ApplicationImages.ICON_INFORMATION_16);
    }
  }

  /**
   * Obtm as descries utilizadas em execues anteriores.
   *
   * @return a lista de descries utilizadas anteriormente.
   */
  private String[] loadPreviousDescriptions() {
    Task<String[]> task = new RemoteTask<String[]>() {
      @Override
      protected void performTask() throws Exception {
        setResult(commandDescriptionHistory.loadPreviousDescriptions());
      }
    };
    String message =
      LNG
      .get("CommandExecutionDialog.task.message.loading_previous_descriptions");
    boolean success = task.execute(getOwner(), getTitle(), message);
    if (success) {
      return task.getResult();
    }
    return new String[0];
  }

  /**
   * Salva a descrio do comando para poder reutilizar futuramente.
   *
   * @param description a descrio a ser salva.
   */
  private void saveDescription(final String description) {
    if (description == null) {
      return;
    }
    Task<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        commandDescriptionHistory.saveDescription(description);
      }
    };
    String message =
      LNG.get("CommandExecutionDialog.task.message.saving_description");
    task.execute(getOwner(), getTitle(), message);
  }

  /**
   * Executa o comando.
   */
  protected void execute() {
    CommandSubmission commandSubmission;
    String clientName = Client.getInstance().getClientName();
    commandSubmission =
      new CommandSubmission(this.configurator, project.getId(), clientName);
    submitCommand(commandSubmission);
  }

  /**
   * Submete o comando para execuo.
   *
   * @param commandSubmission a submisso.
   */
  protected void submitCommand(CommandSubmission commandSubmission) {
    boolean mailAtEnd = this.emailCheckBox.isSelected();
    commandSubmission.setMailAtEnd(mailAtEnd);

    String commandDescription = this.commandDescriptionTextField.getText();
    commandSubmission.setDescription(commandDescription);

    String platformFilter = this.sgaPanel.getPlatformFilter();
    commandSubmission.setPlatform(platformFilter);

    Priority priority = getSelectedPriority();
    commandSubmission.setPriority(priority);

    ExecutionType type = this.configurator.getExecutionType();
    switch (type) {
      case SIMPLE:
        configureSimpleCommand(commandSubmission);
        break;
      case MULTIPLE:
        configureMultipleCommand(commandSubmission);
        break;
      default:
        throw new IllegalArgumentException(String.format(
          "Tipo de execuo %s desconhecido.", type));
    }
    submit(commandSubmission);
    saveDescription(commandDescription);
  }

  /**
   * Retorna a prioridade de execuo do algoritmo
   *
   * @return a prioridade do algoritmo
   */
  private Priority getSelectedPriority() {
    /** Prioridade de execuo do algoritmo. */
    Priority priority = Priority.getDefault();
    if (this.enablePriority) {
      for (Priority p : Priority.values()) {
        if (p.ordinal() == this.priorityComboBox.getSelectedIndex()) {
          priority = p;
          break;
        }
      }
    }
    return priority;
  }

  /**
   * Atualiza o boto Executar.
   */
  protected void updateCanExecute() {
    if (canExecute()) {
      this.executeButton.setEnabled(true);
    }
    else {
      this.executeButton.setEnabled(false);
    }

    if (this.sgaPanel.hasAvailableServers()) {
      setSGAStatus(" ", null);
    }
    else {
      final String text =
        LNG.get("CommandExecutionDialog.label.no_sga_available");
      setSGAStatus(text, ApplicationImages.ICON_ERROR_16);
    }

  }

  /**
   * Atribui a mensagem e o cone de status do SGA.
   *
   * @param text a mensagem de status.
   * @param icon o cone de status.
   */
  protected void setSGAStatus(String text, ImageIcon icon) {
    this.sgaStatusLabel.setText(text);
    this.sgaStatusLabel.setIcon(icon);
  }

  /**
   * Atribui a mensagem e o cone de status dos parmetros do configurador.
   *
   * @param text a mensagem de status.
   * @param icon o cone de status.
   */
  protected void setParameterStatus(String text, ImageIcon icon) {
    this.parameterStatusLabel.setText(text);
    this.parameterStatusLabel.setIcon(icon);
  }

  /**
   * Determina se o comando pode ser executado.
   *
   * @return verdadeiro se o comando est pronto para execuo ou falso, caso
   *         contrrio.
   */
  protected boolean canExecute() {
    return validationResult.isWellSucceded()
      && this.sgaPanel.isSelectionReady();
  }

  /**
   * Coloca todos os widgets na janela do configurador de seleo.
   */
  private void populatePanel() {
    Container contentPane = getContentPane();
    contentPane.removeAll();
    JPanel selectionPanel = new JPanel();
    selectionPanel.setLayout(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.WEST;
    c.gridx = 1;
    c.gridy = 1;
    c.gridwidth = 2;
    c.insets = new Insets(2, 2, 2, 2);
    c.weightx = 1.0;
    c.weighty = 0.0;
    c.fill = GridBagConstraints.HORIZONTAL;
    selectionPanel.add(this.sgaPanel, c);
    c.gridy++;
    c.gridwidth = 1;
    c.weightx = 0;
    c.fill = GridBagConstraints.NONE;
    selectionPanel.add(this.commandDescriptionLabel, c);
    c.gridx++;
    c.fill = GridBagConstraints.HORIZONTAL;
    c.weightx = 1;
    selectionPanel.add(this.commandDescriptionTextField, c);
    c.fill = GridBagConstraints.NONE;
    c.gridx = 2;
    c.gridy++;
    c.gridwidth = 2;
    c.weightx = 1;
    selectionPanel.add(this.previousCommandDescriptionComboBox, c);
    if (this.enablePriority) {
      c.gridx = 1;
      c.gridy++;
      c.gridwidth = 1;
      c.weightx = 0;
      c.fill = GridBagConstraints.NONE;
      selectionPanel.add(this.priorityLabel, c);
      c.gridx++;
      c.weightx = 1;
      selectionPanel.add(this.priorityComboBox, c);
    }
    c.gridy++;
    c.gridx = 1;
    c.gridwidth = 2;
    selectionPanel.add(this.emailCheckBox, c);
    c.gridy++;
    selectionPanel.add(this.parameterStatusLabel, c);
    c.gridy++;
    selectionPanel.add(this.sgaStatusLabel, c);

    contentPane.add(selectionPanel, BorderLayout.CENTER);
    contentPane.add(this.buttonPanel, BorderLayout.SOUTH);
    contentPane.validate();
    getRootPane().setDefaultButton(this.executeButton);
    validate();
    pack();
    Dimension currentSize = getSize();
    int currentWidth = (int) currentSize.getWidth();
    int currentHeight = (int) currentSize.getHeight();
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    int screenWidth = (int) screenSize.getWidth();
    int screenHeight = (int) screenSize.getHeight();
    int newWidth = Math.min(currentWidth, screenWidth);
    int newHeight = Math.min(currentHeight, screenHeight);
    setSize(newWidth, newHeight);
    validate();
    center(getOwner());
  }

  /**
   * Cria o painel de seleo de SGAs.
   *
   * @throws IllegalStateException Tipo de execuo no esperado.
   */
  private void createSGASelectionPanel() {
    ExecutionType type = this.configurator.getExecutionType();
    switch (type) {
      case SIMPLE:
        sgaPanel = new SGASingleSelectionPanel(configurator);
        setTitle(LNG.get("CommandExecutionDialog.title.simple"));
        break;
      case MULTIPLE:
        sgaPanel = new SGAMultipleSelectionPanel(configurator);
        setTitle(LNG.get("CommandExecutionDialog.title.multiple"));
        break;
      default:
        throw new IllegalStateException("Tipo de execuo desconhecido: "
          + configurator.getExecutionType());
    }
    this.sgaPanel.populateGUIComponents();
    this.sgaPanel.addSGASelectionListener(new SGASelectionListener() {
      @Override
      public void wasChangedPlatform() {
        // ignora este evento.
      }

      @Override
      public void wasChangedServer() {
        updateCanExecute();
      }
    });
  }

  /**
   * Configura a submisso do comando (execuo simples).
   *
   * @param commandSubmission informao sobre a submisso do comando.
   */
  private void configureSimpleCommand(CommandSubmission commandSubmission) {
    Collection<SGASet> serverCollection =
      this.sgaPanel.getSelectedServerCollection();
    if (serverCollection != null) {
      // Seleo manual
      SGASet sgaServer = serverCollection.iterator().next();
      commandSubmission.configureSimpleExecution(sgaServer.getName());
    }
    else {
      commandSubmission.configureSimpleExecution();
    }
  }

  /**
   * Configura a submisso do comando (execuo mltipla).
   *
   * @param commandSubmission informao sobre a submisso do comando.
   */
  private void configureMultipleCommand(CommandSubmission commandSubmission) {
    Collection<SGASet> serverCollection =
      this.sgaPanel.getSelectedServerCollection();
    //  Seleo automtica de servidores.
    if (serverCollection == null) {
      Integer executionCount =
        ((SGAMultipleSelectionPanel) this.sgaPanel)
        .getExecutionCountForMultipleExecution();
      commandSubmission.configureMultipleExecution(executionCount);
    }
    else {
      // Seleo manual de servidores.
      List<String> serverList = new LinkedList<String>();
      for (SGASet sgaSet : serverCollection) {
        serverList.add(sgaSet.getName());
      }
      Integer executionCountPerSGA =
        ((SGAMultipleSelectionPanel) this.sgaPanel)
        .getExecutionCountPerSGAForMultipleExecution();
      commandSubmission.configureMultipleExecution(serverList,
        executionCountPerSGA);
    }
  }

  /**
   * Submete um algoritmo para execuo no escalonador.
   *
   * @param commandSubmission informaes sobre a submisso do comando.
   */
  private void submit(CommandSubmission commandSubmission) {
    Set<CommandInfo> submittedCommands =
      SchedulerProxy.submitCommand(this, commandSubmission);
    if (submittedCommands != null) {
      for (CommandRequestedListener listener : listeners) {
        listener.commandsWereRequested(submittedCommands);
      }
    }
  }

  /**
   * Estabelece um valor default para a descrio do comando que vai ser
   * executado.
   *
   * @param description descrio do comando
   */
  private void setCommandDescription(String description) {
    commandDescriptionTextField.setText(description);
  }

  /**
   * Retorna o configurador do algoritmo.
   *
   * @return o configurador
   */
  protected AlgorithmConfigurator getConfigurator() {
    return configurator;
  }

  /**
   * Retorna o projeto.
   *
   * @return o projeto.
   */
  protected CommonClientProject getProject() {
    return project;
  }
}
