/*
 * $Id: ExecutorLocal.java 149543 2014-02-11 13:02:45Z oikawa $
 */

package csbase.client.applications.executorlocal;

import java.io.File;
import java.io.IOException;

import tecgraf.javautils.core.externalcommand.ExternalCommand;
import csbase.client.applications.ApplicationProject;
import csbase.client.desktop.DesktopFrame;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;

/**
 * A classe <code>ExecutorLocal</code> implementa a aplicao que executa
 * aplicativos locais.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ExecutorLocal extends ApplicationProject {

  /**
   * Thread para execuo da aplicao local
   */
  private Thread exec;

  /**
   * Arquivo passada como parmetro pela rvore de projetos
   */
  private ClientProjectFile fileToOpen;

  /**
   * Construtor padro da aplicao.
   * 
   * @param id id da aplicao.
   */
  public ExecutorLocal(String id) {
    super(id);
  }

  /**
   * Solicita a execuo de uma aplicao local. Espera que ela termine e
   * finaliza a aplicao.
   */
  private void executeLocal() {
    DesktopFrame desktop = DesktopFrame.getInstance();
    ExternalCommand command = null;
    String appCommand = getCommand();
    String appDir = getApplicationCommandDir();
    System.out.println("Caminho da aplicao:" + appDir);
    File workingDir = new File(appDir);
    try {
      command = new ExternalCommand(appCommand, null, workingDir);
      if (command.waitFor() == 0) {
        System.out.println("A aplicao terminou OK");
      }
    }
    // Para o caso de sair com killApplication
    catch (InterruptedException e) {
      if (command != null) {
        command.destroy();
      }
    }
    catch (IOException e) {
      StandardErrorDialogs.showErrorDialog(desktop.getDesktopFrame(),
        getName(), getString("msg.error.io"));
    }
    catch (SecurityException e) {
      StandardErrorDialogs.showErrorDialog(desktop.getDesktopFrame(),
        getName(), getString("msg.error.security"));
    }
    finally {
      finishApplication();
    }
  }

  /**
   * Pega a linha de comando para a execuo da aplicao e substitui os
   * parmetros
   * 
   * @return a linha de comando pronta para executar
   */
  private String getCommand() {
    DesktopFrame desktop = DesktopFrame.getInstance();
    String server = ClientRemoteMonitor.getInstance().getServerPath();
    String project = desktop.getProject().getName();
    String owner = (String) desktop.getProject().getUserId();
    String user = (String) User.getLoggedUser().getId();
    String pass = ClientRemoteMonitor.getInstance().getPassword();
    String[] srvPort = server.split(":");
    String wioPort = getWIOPort();
    String file = getFilePath();
    String[][] aParam =
      {
          { "USER", "PASS", "SERVER", "PORT", "PROJECT", "WIO", "OWNER", "FILE" },
          { user, pass, srvPort[0], srvPort[1], project, wioPort, owner, file } };
    String appCommand = getApplicationCommand();
    for (int i = 0; i < aParam[0].length; i++) {
      appCommand = appCommand.replaceAll(aParam[0][i], aParam[1][i]);
    }
    return appCommand;
  }

  /**
   * Obtem a porta do servico de wio
   * 
   * @return .
   */
  private String getWIOPort() {
    try {
      String port = ClientRemoteLocator.wioService.getServerPort();
      if (port == null) {
        return "";
      }
      return port;
    }
    catch (Exception e) {
      DesktopFrame desktop = DesktopFrame.getInstance();
      StandardErrorDialogs.showErrorDialog(desktop.getDesktopFrame(),
        desktop.getTitle(), getString("msg.error.paramwio"));
      return "";
    }
  }

  /**
   * Obtm o caminho para o arquivo a ser aberto na rvore de projetos. Se no
   * h arquivo a ser aberto ou ser a propriedade
   * <code>APP_PROJECT_DIR_PROPERTY</code> no foi definida, retorna uma
   * <code>String</code> vazia.
   * 
   * @return o caminho para o arquivo a ser aberto na rvore de projetos
   */
  private String getFilePath() {
    if (fileToOpen == null) {
      return "";
    }
    return fileToOpen.getStringPath();
  }

  /**
   * Cria a Thread que executa a aplicao
   * 
   * @return a Thread de execuo da aplicao
   */
  private Thread createExecuteThread() {
    final Thread newThread = new Thread() {
      @Override
      public void run() {
        executeLocal();
      }
    };
    return newThread;
  }

  /**
   * Trmino ASAP da aplicao.
   */
  @Override
  public void killApplication() {
    try {
      exec.interrupt();
    }
    catch (SecurityException e) {
    }
  }

  /**
   * Mtodo que permite sempre ao usurio matar a aplicao.
   * 
   * @return o flag indicativo.
   */
  @Override
  public boolean userCanKillApplication() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void postInitialization() {
    exec = createExecuteThread();
    exec.start();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(String name, Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(PROJECT_FILE_MESSAGE)) {
      fileToOpen = (ClientProjectFile) value;
    }
  }
}
