/**
 * $Id: LogAdministration.java 169707 2015-11-12 10:52:44Z lmoreira $
 */
package csbase.client.applications.logadministration;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JToolBar;
import javax.swing.SortOrder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;

import tecgraf.ftc.common.logic.RemoteFileChannel;
import tecgraf.ftc.common.logic.RemoteFileChannelImpl;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.gui.table.AbstractColumn;
import tecgraf.javautils.gui.table.ColumnsObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableBuilder;
import tecgraf.javautils.gui.table.ObjectTableBuilder.ColumnsWidthPolicy;
import tecgraf.javautils.gui.table.ObjectTableBuilder.SelectionMode;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.ClientLocalFile;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.logic.LogFile;
import csbase.logic.LogFileInfo;
import csbase.logic.diagnosticservice.LogType;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.LogAdministrationServiceInterface;

/**
 * Aplicao para administrao dos arquivos de log. Ela utiliza um servio que
 * implementa a interface {@link LogAdministrationServiceInterface}.
 * 
 * @author Tecgraf
 */
public class LogAdministration extends Application {
  /**
   * Painel com abas referentes aos tipos de arquivos de log.
   */
  JTabbedPane logTypeTabbedPane = new JTabbedPane();

  /**
   * Construtor de tabela
   */
  ObjectTableBuilder<LogFileInfo> tableBuilder;

  /**
   * Mapa com as tabelas de cada tipo de arquivo de log
   */
  private Map<String, JTable> logFileTableMap = new HashMap<String, JTable>();

  /**
   * Mapa com o diretrio inicial (limite) de cada tabela de arquivo de log
   */
  private Map<String, String> initialLogDirectoryMap =
    new HashMap<String, String>();

  /**
   * Mapa com o diretrio corrente de cada tabela de arquivo de log
   */
  private Map<String, String> currentLogDirectoryMap =
    new HashMap<String, String>();

  /**
   * Dilogo para enviar por email arquivos de log
   */
  private SendLogFilesDialog sendLogFileDialog;

  /**
   * Diretrio selecionado como destino da ao de exportar arquivos de log
   */
  private File exportTargetDirectory;

  /**
   * Flag para indicar que todos os arquivos exportados que j existam no
   * diretrio local devem ser sobrescritos
   */
  private boolean overwriteAll = false;

  /**
   * {@link LogFileInfo} dos arquivos de log selecionados
   */
  private LogFileInfo[] selectedLogFilesInfo;

  /**
   * Ao para enviar os arquivos de log selecionados
   */
  private SendLogFileAction sendLogFileAction = new SendLogFileAction();

  /**
   * Ao para exportar os arquivos de log selecionados
   */
  private SaveLogFileAction saveLogFileAction = new SaveLogFileAction();

  /**
   * Ao para recarregar a tabela de arquivos de log corrente
   */
  private RefreshLogFileTableAction refreshLogFileTableAction =
    new RefreshLogFileTableAction();

  /**
   * Ao para carregar na tabela os arquivos de log do diretrio pai do
   * corrente
   */
  private LoadParentDirectoryAction loadParentDirectoryAction =
    new LoadParentDirectoryAction();

  /**
   * Ao para carregar na tabela os arquivos de log do diretrio inicial
   */
  private LoadHomeDirectoryAction loadHomeDirectoryAction =
    new LoadHomeDirectoryAction();

  /**
   * @param id identificador da aplicao
   */
  public LogAdministration(String id) {
    super(id);
    tableBuilder = createTableBuilder();
    buildFrame();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() throws ApplicationException {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /*****************************************************************************
   * Contrutores da interface
   ****************************************************************************/

  /**
   * Constri o frame da aplicao
   */
  private void buildFrame() {
    final ApplicationFrame frame = getApplicationFrame();

    frame.setLayout(new BorderLayout());
    frame.setSize(500, 300);
    frame.setJMenuBar(buildMenu());

    final Container appContentPane = frame.getContentPane();
    appContentPane.add(buildToolBar(), BorderLayout.NORTH);

    buildTypeTabbedPane();
    frame.add(logTypeTabbedPane, BorderLayout.CENTER);

    sendLogFileDialog = new SendLogFilesDialog(this);
    sendLogFileDialog.setVisible(false);
  }

  /**
   * Constri o painel com aba para cada tipo de arquivo de log.
   */
  private void buildTypeTabbedPane() {
    final ApplicationFrame frame = getApplicationFrame();

    LoadLogTypesTask task = new LoadLogTypesTask();
    if (!task.execute(frame, getString("loadlogfilestask.title"),
      getString("loadlogfilestask.message"))) {
      return;
    }
    LogType[] logTypes = task.getResult();

    if (logTypes.length == 0) {
      JPanel panel = new JPanel(false);
      JLabel label = new JLabel(getString("no.initial.directory"));
      label.setHorizontalAlignment(JLabel.CENTER);
      panel.setLayout(new GridLayout(1, 1));
      panel.add(label);
      logTypeTabbedPane.addTab(getString("error.tab.title"), panel);

      saveLogFileAction.setEnabled(false);
      sendLogFileAction.setEnabled(false);
      refreshLogFileTableAction.setEnabled(false);
      loadParentDirectoryAction.setEnabled(false);
      return;
    }

    for (LogType type : logTypes) {
      SortableTable logFileTable = buildFileTable(type.name, type.dir);
      logFileTableMap.put(type.name, logFileTable);
      currentLogDirectoryMap.put(type.name, type.dir);
      initialLogDirectoryMap.put(type.name, type.dir);
      logTypeTabbedPane.addTab(type.name, buildFileTablePanel(logFileTable));
    }

    logTypeTabbedPane.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        if (e.getSource() instanceof JTabbedPane) {
          String type = getActiveLogType();
          if (currentLogDirectoryMap.get(type).equals(
            initialLogDirectoryMap.get(type))) {
            loadParentDirectoryAction.setEnabled(false);
          }
          else {
            loadParentDirectoryAction.setEnabled(true);
          }
        }
      }
    });
  }

  /**
   * Constri o painel com a lista de arquivos de logs
   * 
   * @param c componente para adicionar ao painel
   * 
   * @return o painel com a lista de arquivos de logs
   */
  private JPanel buildFileTablePanel(Component c) {
    JPanel fileTablePanel = new JPanel(new BorderLayout());
    fileTablePanel.add(new JScrollPane(c), BorderLayout.CENTER);

    return fileTablePanel;
  }

  /**
   * Constri a lista de arquivos de log. A lista  exibida em uma tabela do
   * tipo {@link SortableTable}
   * 
   * @param type tipo dos arquivos de log
   * @param dir diretrio do qual os arquivos sero obtidos
   * 
   * @return a {@link SortableTable} com os arquivos de log disponveis
   */
  private SortableTable buildFileTable(String type, String dir) {
    final ApplicationFrame frame = getApplicationFrame();

    LoadLogFilesTask task = new LoadLogFilesTask(type, dir);
    if (!task.execute(frame, getString("loadlogfilestask.title"),
      getString("loadlogfilestask.message"))) {
      return null;
    }
    LogFileInfo[] logFiles = task.getResult();

    if (logFiles == null) {
      showInvalidParentDirectoryErroDialog();
      return null;
    }

    List<LogFileInfo> logFileList = new ArrayList<LogFileInfo>();
    for (LogFileInfo logFile : logFiles) {
      logFileList.add(logFile);
    }

    SortableTable logFileTable = tableBuilder.build(logFileList);
    logFileTable.getColumnModel().getColumn(0).setMaxWidth(36);

    logFileTable.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          LogFileInfo[] selectedFiles = getSelectedFilesInfo();
          if (selectedFiles.length == 1 && selectedFiles[0].isDirectory) {
            refreshLogFileTable(getActiveLogType(), selectedFiles[0].path);
          }
        }
      }
    });

    return logFileTable;
  }

  /**
   * Cria a barra de ferramentas da aplicao.
   * 
   * @return barra de ferramentas.
   */
  private JToolBar buildToolBar() {
    JToolBar toolBar = new JToolBar();
    toolBar.setFloatable(false);
    toolBar.add(sendLogFileAction);
    toolBar.add(saveLogFileAction);
    toolBar.add(new JToolBar.Separator());
    toolBar.add(refreshLogFileTableAction);
    toolBar.add(loadParentDirectoryAction);
    loadParentDirectoryAction.setEnabled(false);
    toolBar.add(loadHomeDirectoryAction);
    return toolBar;
  }

  /**
   * Constri o menu da aplicao
   * 
   * @return menu da aplicao
   */
  private JMenuBar buildMenu() {
    JMenu menuFile = new JMenu(getString("menu.file"));

    //criando sub-itens do menu menuFile
    JMenuItem sendLogFileMenu = new JMenuItem(sendLogFileAction);
    menuFile.add(sendLogFileMenu);

    JMenuItem saveLogFileMenu = new JMenuItem(saveLogFileAction);
    menuFile.add(saveLogFileMenu);

    menuFile.addSeparator();

    JMenuItem refreshLogFileListMenu = new JMenuItem(refreshLogFileTableAction);
    menuFile.add(refreshLogFileListMenu);

    JMenuItem loadParentDirectoryMenu =
      new JMenuItem(loadParentDirectoryAction);
    menuFile.add(loadParentDirectoryMenu);

    JMenuItem loadHomeDirectoryMenu = new JMenuItem(loadHomeDirectoryAction);
    menuFile.add(loadHomeDirectoryMenu);

    menuFile.addSeparator();

    JMenuItem exitMenu = new JMenuItem(new ApplicationExitAction(this));
    menuFile.add(exitMenu);

    //criando barra de menu
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(menuFile);

    return menuBar;
  }

  /*****************************************************************************
   * Aes
   ****************************************************************************/

  /**
   * Ao para carregar a tabela ativa com os arquivos do diretrio pai do
   * diretrio corrente
   * 
   * @author Tecgraf
   */
  private class LoadParentDirectoryAction extends AbstractAction {
    /**
     * Construtor padro.
     * 
     */
    protected LoadParentDirectoryAction() {
      super();
      putValue(SMALL_ICON, ApplicationImages.ICON_UP_16);
      putValue(SHORT_DESCRIPTION, getString("action.to.parent.tooltip"));
      putValue(NAME, getString("action.to.parent"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      refreshLogFileTable(getActiveLogType(),
        getParentDir(getActiveDirectory()));
    }
  }

  /**
   * Ao para carregar a tabela ativa com os arquivos do diretrio inicial
   * 
   * @author Tecgraf
   */
  private class LoadHomeDirectoryAction extends AbstractAction {
    /**
     * Construtor padro.
     * 
     */
    protected LoadHomeDirectoryAction() {
      super();
      putValue(SMALL_ICON, ApplicationImages.ICON_BROWSELOCALFILE_16);
      putValue(SHORT_DESCRIPTION, getString("action.to.home.tooltip"));
      putValue(NAME, getString("action.to.home"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      refreshLogFileTable(getActiveLogType(), getInitialDirectory());
    }
  }

  /**
   * Ao para recarregar a tabela ativa com os arquivos do diretrio corrente
   * 
   * @author Tecgraf
   */
  private class RefreshLogFileTableAction extends AbstractAction {
    /**
     * Construtor padro.
     */
    protected RefreshLogFileTableAction() {
      super();
      putValue(SMALL_ICON, ApplicationImages.ICON_REFRESH_16);
      putValue(SHORT_DESCRIPTION, getString("action.refresh.loglist.tooltip"));
      putValue(NAME, getString("action.refresh.loglist"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      refreshLogFileTable(getActiveLogType(), getActiveDirectory());
    }
  }

  /**
   * Ao para enviar os arquivos de log
   * 
   * @author Tecgraf
   */
  private class SendLogFileAction extends AbstractAction {

    /**
     * Construtor padro.
     */
    protected SendLogFileAction() {
      super();
      putValue(SMALL_ICON, ApplicationImages.ICON_SENDMAIL_16);
      putValue(SHORT_DESCRIPTION, getString("action.send.email.tooltip"));
      putValue(NAME, getString("action.send.email"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent arg0) {
      selectedLogFilesInfo = getSelectedFilesInfo();

      if (selectedLogFilesInfo.length == 0) {
        showEmptyListErroDialog();
      }
      else {
        sendLogFileDialog.updateSelectedFiles(selectedLogFilesInfo);
        sendLogFileDialog.setVisible(true);
      }
    }
  }

  /**
   * Ao para exportar os arquivos de log
   * 
   * @author Tecgraf
   */
  private class SaveLogFileAction extends AbstractAction {
    /**
     * Construtor padro.
     */
    protected SaveLogFileAction() {
      super();
      putValue(SMALL_ICON, ApplicationImages.ICON_EXPORT_16);
      putValue(SHORT_DESCRIPTION, getString("action.save.log.tooltip"));
      putValue(NAME, getString("action.save.log"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent arg0) {
      selectedLogFilesInfo = getSelectedFilesInfo();

      if (selectedLogFilesInfo.length == 0) {
        showEmptyListErroDialog();
      }
      else {
        ClientLocalFile localDir =
          ClientLocalFileChooserUtil.browseSingleDirectoryInOpenMode(
            getApplicationFrame(), Collections.<String>emptyList(),
            getString("savelogdialog.approve.button"), false, null);

        if (localDir != null) {
          if (!localDir.exists()) {
            showInvalidDirectoryErroDialog();
            return;
          }

          exportTargetDirectory = new File(localDir.getStringPath());

          DownloadLogFilesTask task = new DownloadLogFilesTask();
          if (!task.execute(getApplicationFrame(),
            getString("savelogfilestask.message"),
            getString("savelogfilestask.title"))) {
            showExportFileErroDialog();
          }

          LogFile[] logFiles = task.getResult();

          overwriteAll = false;
          for (LogFile file : logFiles) {
            try {
              saveFile(exportTargetDirectory, file);
            }
            catch (IOException e) {
              showExportFileErroDialog();
            }
          }
        }
      }
    }
  }

  /*****************************************************************************
   * Tarefas remotas
   ****************************************************************************/

  /**
   * Tarefa remota para buscar os arquivos de logs disponveis
   * 
   * @author Tecgraf
   */
  private class LoadLogTypesTask extends RemoteTask<LogType[]> {
    /**
     * {@inheritDoc}
     */
    @Override
    protected void performTask() throws Exception {
      LogAdministrationServiceInterface logAdministrationService =
        ClientRemoteLocator.logAdministrationService;
      LogType[] typeList = logAdministrationService.getLogTypes();
      setResult(typeList);
    }
  }

  /**
   * Tarefa remota para buscar os arquivos de logs disponveis
   * 
   * @author Tecgraf
   */
  private class LoadLogFilesTask extends RemoteTask<LogFileInfo[]> {
    /**
     * Tipo dos arquivos de log
     */
    String type;
    /**
     * Diretrio de onde buscar os arquivos
     */
    String dir;

    /**
     * @param type tipo dos arquivos de log
     * @param dir diretrio de onde buscar os arquivos
     */
    public LoadLogFilesTask(String type, String dir) {
      this.type = type;
      this.dir = dir;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void performTask() throws Exception {
      LogAdministrationServiceInterface logAdministrationService =
        ClientRemoteLocator.logAdministrationService;
      LogFileInfo[] fileList = logAdministrationService.listLogFiles(type, dir);
      setResult(fileList);
    }
  }

  /**
   * Tarefa remota para obter o contedo dos arquivos de logs selecionados
   * 
   * @author Tecgraf
   */
  private class DownloadLogFilesTask extends RemoteTask<LogFile[]> {
    /**
     * {@inheritDoc}
     */
    @Override
    protected void performTask() throws Exception {
      LogAdministrationServiceInterface logAdministrationService =
        ClientRemoteLocator.logAdministrationService;

      String[] selectedFilesPath = new String[selectedLogFilesInfo.length];

      for (int i = 0; i < selectedLogFilesInfo.length; i++) {
        selectedFilesPath[i] = selectedLogFilesInfo[i].path;
      }

      setResult(logAdministrationService.downloadLogFile(selectedFilesPath));
    }
  }

  /*****************************************************************************
   * Colunas
   ****************************************************************************/

  /**
   * Coluna que representa se o arquivo  um diretrio ou no
   * 
   * @author Tecgraf
   */
  private class DirColumn extends AbstractColumn<LogFileInfo> {

    /**
     * Construtor
     */
    protected DirColumn() {
      super(ImageIcon.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return "";
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(LogFileInfo obj) {
      if (obj.isDirectory) {
        return ApplicationImages.ICON_FOLDER_16;
      }
      else {
        return ApplicationImages.ICON_FILE_16;
      }
    }
  }

  /**
   * Coluna que representa o nome de um arquivo de log
   * 
   * @author Tecgraf
   */
  private class NameColumn extends AbstractColumn<LogFileInfo> {

    /**
     * Construtor
     */
    protected NameColumn() {
      super(String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return getString("fileloglist.column.name.name");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(LogFileInfo obj) {
      return obj.name;
    }
  }

  /**
   * Coluna que representa o tamanho de um arquivo de log
   * 
   * @author Tecgraf
   */
  private class SizeColumn extends AbstractColumn<LogFileInfo> {

    /**
     * Construtor
     */
    protected SizeColumn() {
      super(Long.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return getString("fileloglist.column.name.size");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(LogFileInfo obj) {
      return obj.size;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public TableCellRenderer createTableCellRenderer() {
      return new DefaultTableCellRenderer() {
        @Override
        public Component getTableCellRendererComponent(JTable table,
          Object value, boolean isSelected, boolean hasFocus, int row,
          int column) {
          super.getTableCellRendererComponent(table, value, isSelected,
            hasFocus, row, column);

          Long v = (Long) value;
          setText(FormatUtils.formatSize(v, 0));
          setHorizontalAlignment(JLabel.RIGHT);
          return this;
        }
      };
    }
  }

  /**
   * Coluna que representa data de atualizao de um arquivo de log
   * 
   * @author Tecgraf
   */
  private class LastModifiedColumn extends AbstractColumn<LogFileInfo> {

    /**
     * Construtor
     */
    protected LastModifiedColumn() {
      super(Long.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return getString("fileloglist.column.name.last");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(LogFileInfo obj) {
      return obj.lastModified;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public TableCellRenderer createTableCellRenderer() {
      return new DefaultTableCellRenderer() {
        @Override
        public Component getTableCellRendererComponent(JTable table,
          Object value, boolean isSelected, boolean hasFocus, int row,
          int column) {
          super.getTableCellRendererComponent(table, value, isSelected,
            hasFocus, row, column);

          String v =
            csbase.logic.Utilities.getFormattedDate(new Date((Long) value),
              "dd/MM/yyyy HH:mm:ss");
          setText(v.toString());
          setHorizontalAlignment(JLabel.RIGHT);
          return this;
        }
      };
    }

  }

  /*****************************************************************************
   * Utilitrios
   ****************************************************************************/

  /**
   * Exibe dilogo de erro para informar que nenhum arquivo foi selecionado
   */
  private void showEmptyListErroDialog() {
    showError(getString("emptylist.error.msg"));
  }

  /**
   * Exibe dilogo de erro para informar que o diretrio de destno no existe
   */
  private void showInvalidDirectoryErroDialog() {
    showError(getString("invalid.directory.error.msg"));
  }

  /**
   * Exibe dilogo de erro para informar que o diretrio corrente no tem pai
   */
  private void showInvalidParentDirectoryErroDialog() {
    showError(getString("no.parent.error.msg"));
  }

  /**
   * Exibe dilogo de erro na exportao de arquivos
   */
  private void showExportFileErroDialog() {
    showError(getString("savelog.error.msg"));
  }

  /**
   * Cria um ObjectTableBuilder para a tabela de arquivos
   * 
   * @return o ObjectTableBuilder
   */
  @SuppressWarnings("unchecked")
  ObjectTableBuilder<LogFileInfo> createTableBuilder() {
    ObjectTableBuilder<LogFileInfo> tableBuilder =
      new ObjectTableBuilder<LogFileInfo>(new DirColumn(), new NameColumn(),
        new SizeColumn(), new LastModifiedColumn());
    tableBuilder
      .setColumnsWidthPolicy(ColumnsWidthPolicy.ADJUST_BY_DATA_DYNAMIC);
    tableBuilder.setSelectionMode(SelectionMode.SINGLE_SELECTION);
    tableBuilder.setSortOrder(1, SortOrder.ASCENDING);

    return tableBuilder;
  }

  /**
   * Obtm a tabela de arquivos ativa no momento
   * 
   * @return a tabela ativa
   */
  SortableTable getActiveFileTable() {
    return SortableTable.class.cast(logFileTableMap.get(logTypeTabbedPane
      .getTitleAt(logTypeTabbedPane.getSelectedIndex())));
  }

  /**
   * Obtm o diretrio corrente da lista ativa
   * 
   * @return diretrio corrente
   */
  String getActiveDirectory() {
    return currentLogDirectoryMap.get(logTypeTabbedPane
      .getTitleAt(logTypeTabbedPane.getSelectedIndex()));
  }

  /**
   * Obtm o diretrio inicial da lista ativa
   * 
   * @return diretrio inicial
   */
  String getInitialDirectory() {
    return initialLogDirectoryMap.get(logTypeTabbedPane
      .getTitleAt(logTypeTabbedPane.getSelectedIndex()));
  }

  /**
   * Obtm o tipo de log ativo
   * 
   * @return tipo de log ativo
   */
  String getActiveLogType() {
    return logTypeTabbedPane.getTitleAt(logTypeTabbedPane.getSelectedIndex());
  }

  /**
   * Obtm o JScrollPane da Tab ativa.
   * 
   * @return o JScrollPane
   */
  private JScrollPane getActiveScrollPane() {
    JPanel panel = JPanel.class.cast(logTypeTabbedPane.getSelectedComponent());

    for (Component c : panel.getComponents()) {
      if (c instanceof JScrollPane) {
        return JScrollPane.class.cast(c);
      }
    }

    return null;
  }

  /**
   * Extri {@link LogFileInfo} das linhas selecionadas da tabela com os
   * arquivos disponveis
   * 
   * @return {@link LogFileInfo} dos arquivos selecionados
   */
  @SuppressWarnings("unchecked")
  private LogFileInfo[] getSelectedFilesInfo() {
    int[] selectedRows = getActiveFileTable().getSelectedRows();
    ColumnsObjectTableModel<LogFileInfo> tableModel =
      (ColumnsObjectTableModel<LogFileInfo>) getActiveFileTable().getModel();

    LogFileInfo[] infos = new LogFileInfo[selectedRows.length];
    for (int i = 0; i < selectedRows.length; i++) {
      infos[i] =
        tableModel.getRow(getActiveFileTable().convertRowIndexToModel(
          selectedRows[i]));
    }

    return infos;
  }

  /**
   * Recria a tabela com os arquivos de log da aba ativa
   * 
   * @param type tipo do arquivo de log
   * @param dir diretrio do qual os arquivos devem ser listados
   */
  private void refreshLogFileTable(String type, String dir) {
    SortableTable fileTable = buildFileTable(type, dir);
    if (fileTable != null) {
      getActiveScrollPane().getViewport().setView(fileTable);
      logFileTableMap.put(
        logTypeTabbedPane.getTitleAt(logTypeTabbedPane.getSelectedIndex()),
        fileTable);
      currentLogDirectoryMap.put(type, dir);

      if (currentLogDirectoryMap.get(type).equals(
        initialLogDirectoryMap.get(type))) {
        loadParentDirectoryAction.setEnabled(false);
      }
      else {
        loadParentDirectoryAction.setEnabled(true);
      }
    }
  }

  /**
   * Grava localmente um arquivo
   * 
   * @param directory diretrio onde o arquivo ser gravado
   * @param logFile arquivo de log para ser gravado
   * @throws IOException
   */
  private void saveFile(File directory, LogFile logFile) throws IOException {
    File localFile =
      new File(directory.getCanonicalPath() + File.separator + logFile.name);

    if (localFile.exists()) {
      boolean overwriteOnce = false;
      if (!overwriteAll) {
        String[] opts = { "Sim para todos", "Sim", "No" };
        int opt;
        if (logFile.isDirectory) {
          opt =
            showOptionDialog(getApplicationFrame(),
              "Diretrio " + localFile.getAbsoluteFile()
                + " j existe. Sobrescrever?", opts);
        }
        else {
          opt =
            showOptionDialog(getApplicationFrame(),
              "Arquivo " + localFile.getAbsoluteFile()
                + " j existe. Sobrescrever?", opts);
        }

        switch (opt) {
          case 0:
            overwriteAll = true;
            break;
          case 1:
            overwriteOnce = true;
            break;
          case 2:
            break;
        }
      }

      if (overwriteAll || overwriteOnce) {
        FileUtils.delete(localFile);
      }
      else {
        return;
      }
    }

    if (logFile.isDirectory) {
      localFile.mkdir();
      for (LogFile child : logFile.children) {
        saveFile(localFile, child);
      }
    }
    else {
      if (!localFile.createNewFile()) {
        throw new IOException("Erro ao criar arquivo "
          + localFile.getAbsolutePath());
      }
      else {
        if (!localFile.canWrite()) {
          localFile.setWritable(true);
        }
        byte data[] = readRemoteFile(logFile.channel);
        OutputStream output = null;
        try {
          output =
            new BufferedOutputStream(new FileOutputStream(
              localFile.getCanonicalPath()));
          output.write(data);
        }
        finally {
          FileUtils.close(output);
        }
      }
    }
  }

  /**
   * Obtm o diretrio pai de um diretrio
   * 
   * @param dir o diretrio
   * 
   * @return o diretrio pai ou o prprio diretrio se esta no tiver pai
   */
  String getParentDir(String dir) {
    // Busca o separador "/" e caso no ache usa o separador do SO
    int lastSeparatorIdx = dir.lastIndexOf("/");
    if (lastSeparatorIdx == -1) {
      lastSeparatorIdx = dir.lastIndexOf(File.separator);
    }

    if (lastSeparatorIdx == 0) {
      return dir.substring(lastSeparatorIdx, lastSeparatorIdx + 1);
    }

    if (lastSeparatorIdx == dir.length() - 1) {
      return getParentDir(dir.substring(0, dir.length() - 1));
    }
    else {
      return dir.substring(0, lastSeparatorIdx);
    }
  }

  /**
   * Faz a leitura de arquivos remotos usando o FTC.
   * 
   * @param channel canal de leitura do arquivo
   * 
   * @return array de bytes com o contedo do arquivo
   */
  public byte[] readRemoteFile(RemoteFileChannelInfo channel) {

    try {
      RemoteFileChannel rfc =
        new RemoteFileChannelImpl(channel.getIdentifier(), false,
          channel.getHost(), channel.getPort(), channel.getKey());
      rfc.open(true);
      int fileSize = (int) rfc.getSize();
      byte[] buffer = new byte[fileSize];
      if (fileSize != 0) {
        rfc.read(buffer);
      }
      rfc.close();
      return buffer;
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }
}
