/*
 * $Id$
 */
package csbase.client.applications.imageviewer;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.HeadlessException;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.Transparency;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;

import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.imageviewer.actions.effects.ImageViewerBlurAction;
import csbase.client.applications.imageviewer.actions.effects.ImageViewerGrayAction;
import csbase.client.applications.imageviewer.actions.effects.ImageViewerLightAction;
import csbase.client.applications.imageviewer.actions.effects.ImageViewerRGBAction;
import csbase.client.applications.imageviewer.actions.io.ImageViewerCloseAction;
import csbase.client.applications.imageviewer.actions.io.ImageViewerOpenAction;
import csbase.client.applications.imageviewer.actions.io.ImageViewerRestoreAction;
import csbase.client.applications.imageviewer.actions.io.ImageViewerSaveAction;
import csbase.client.applications.imageviewer.actions.io.ImageViewerSaveAsAction;
import csbase.client.applications.imageviewer.actions.view.ImageViewerZoomFitAction;
import csbase.client.applications.imageviewer.actions.view.ImageViewerZoomInAction;
import csbase.client.applications.imageviewer.actions.view.ImageViewerZoomOutAction;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;

/**
 * A classe <code>ImageViewer</code> implementa a aplicao de exibio de
 * imagens em um projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ImageViewer extends Application {

  /**
   * Fator de passo de zoom.
   */
  public static final double ZOOM_STEP_FACTOR = 1.05;

  /**
   * Painel onde ser exibida a imagem.
   */
  final private ImgPanel imagePanel;

  /**
   * Flag de arquivo alterado
   */
  private boolean needSave = false;
  /**
   * Imagem
   */
  private Image currentImage;

  /**
   * Imagem
   */
  private Image originalImage;

  /**
   * Nvel de zoom
   */
  private double zoom = 1.0;

  /**
   * Retorna o zoom.
   * 
   * @return zoom
   */
  public final double getZoom() {
    return zoom;
  }

  /**
   * Ajuste do zoom
   * 
   * @param zoom zoom
   */
  public final void setZoom(double zoom) {
    if (zoom < 0.05 || zoom > 10.0) {
      return;
    }
    this.zoom = zoom;
    updateImagePanelSize();
    imagePanel.repaint();
  }

  /**
   * Ajuste do tamanho do painel de imagens.
   */
  private void updateImagePanelSize() {
    if (ready) {
      final int t2X = (int) (Math.round(width * zoom));
      final int t2Y = (int) (Math.round(height * zoom));
      final Dimension dimension = new Dimension(t2X, t2Y);
      imagePanel.setSize(dimension);
      imagePanel.setMinimumSize(dimension);
      imagePanel.setPreferredSize(dimension);
    }
  }

  /**
   * Retorna a imagem corrente
   * 
   * @return image
   */
  public final Image getCurrentImage() {
    return currentImage;
  }

  /**
   * Ajuste de necessidade de salvar arquivo.
   * 
   * @param flag indicativo de imagem alterada.
   */
  final public void setSaveNeeded(final boolean flag) {
    needSave = flag;
    updateTitle();
  }

  /**
   * This method returns a buffered image with the contents of an image
   * 
   * @return o buffer
   */
  public BufferedImage getCurrentBufferedImage() {
    if (currentImage == null) {
      return null;
    }

    if (currentImage instanceof BufferedImage) {
      final BufferedImage bi = (BufferedImage) currentImage;
      if (bi.getType() == BufferedImage.TYPE_INT_RGB) {
        return bi;
      }
    }
    //    currentImage = new ImageIcon(currentImage).getImage();

    // Create a buffered image with a format that's compatible with the screen
    BufferedImage bimage = null;
    GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
    final int w = currentImage.getWidth(null);
    final int h = currentImage.getHeight(null);
    try {
      int transparency = Transparency.OPAQUE;
      GraphicsDevice gs = ge.getDefaultScreenDevice();
      GraphicsConfiguration gc = gs.getDefaultConfiguration();
      bimage = gc.createCompatibleImage(w, h, transparency);
    }
    catch (HeadlessException e) {
      // The system does not have a screen
    }
    if (bimage == null) {
      // Create a buffered image using the default color model
      int type = BufferedImage.TYPE_INT_RGB;
      bimage = new BufferedImage(w, h, type);
    }
    // Copy image to buffered image
    Graphics g = bimage.createGraphics();
    // Paint the image onto the buffered image
    g.drawImage(currentImage, 0, 0, null);
    g.dispose();
    return bimage;
  }

  /**
   * Ajusta a nova imagem
   * 
   * @param image a nova imagem
   */
  public final void setCurrentImage(Image image) {
    this.currentImage = image;
    updateImagePanel();
  }

  /**
   * Largura da figura
   */
  private int width = 0;

  /**
   * Altura da figura
   */
  private int height = 0;

  /**
   * Indica se a imagem est pronta
   */
  private boolean ready = false;

  /**
   * Observador utilizado na formao da imagem
   */
  private ImgObserver observer = null;

  /**
   * Arquivo de visualizao corrrente.
   */
  private ClientFile currentFile;

  /**
   * Retorna
   * 
   * @return currentFile
   */
  public final ClientFile getCurrentFile() {
    return currentFile;
  }

  /**
   * Cria a frame da aplicao
   */
  private void createMainFrame() {
    final ApplicationFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(createMenuBar());
    mainFrame.setSize(new Dimension(800, 600));
    Container cp = mainFrame.getContentPane();
    cp.add(createToolBar(), BorderLayout.NORTH);
    cp.add(createViewer(), BorderLayout.CENTER);
  }

  /**
   * Atualiza o ttulo da janela, inserindo o nome do arquivo atual.
   */
  private void updateTitle() {
    final String filename =
      (currentFile == null ? null : currentFile.getName());
    final ApplicationFrame frame = getApplicationFrame();

    String title = getName();
    if (filename != null) {
      title = title + " - " + filename;
    }

    if (needSave) {
      title = title + "*";
    }

    frame.setTitle(title);
  }

  /**
   * Abre o arquivo dado.
   * 
   * @param newFile arquivo a ser aberto
   * @return indicativo de sucesso.
   */
  public boolean openFile(ClientProjectFile newFile) {
    if (newFile == null) {
      return false;
    }
    Image newImage;
    try {
      newImage = ImageViewerIO.readTask(this, newFile);
      if (newImage == null) {
        return false;
      }
    }
    catch (Exception e) {
      showError(e.getMessage());
      return false;
    }
    originalImage = newImage;
    currentImage = originalImage;

    updateImagePanel();
    return true;
  }

  /**
   * Atualizao do painel de imagem.
   */
  private void updateImagePanel() {
    if (currentImage == null) {
      return;
    }
    observer = new ImgObserver();

    // Se a largura ou altura no estiverem especificados,
    // notifica o observador (observer).
    width = currentImage.getWidth(observer);
    height = currentImage.getHeight(observer);

    // Parece no ser necesssrio
    Toolkit.getDefaultToolkit().prepareImage(currentImage, width, height,
      observer);
    ready = ((width >= 0) && (height >= 0));

    updateImagePanelSize();
    // Chamando repaint(), caso o observador no tenha sido notificado.
    imagePanel.repaint();
  }

  /**
   * Fecha o arquivo dado.
   */
  public void closeFile() {
    originalImage = null;
    currentImage = null;
    observer = null;

    width = -1;
    height = -1;

    ready = false;

    // Chamando repaint(), caso o observador no tenha sido notificado.
    imagePanel.repaint();

    setCurrentFile(null);

  }

  /**
   * Cria o painel de visualizaao de imagens.
   * 
   * @return painel de visualizaao de imagens
   */
  private JComponent createViewer() {
    final JScrollPane scrollPane = new JScrollPane(imagePanel);
    final int hNeeded = JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED;
    final int vNeeded = JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED;

    scrollPane.setHorizontalScrollBarPolicy(hNeeded);
    scrollPane.setVerticalScrollBarPolicy(vNeeded);
    return scrollPane;
  }

  /**
   * Montagem da toolbar.
   * 
   * @return o painel.
   */
  private JToolBar createToolBar() {
    final JToolBar toolBar = new JToolBar(getName());
    toolBar.setFloatable(false);
    toolBar.add(new ImageViewerOpenAction(this));
    toolBar.add(new ImageViewerSaveAction(this));
    toolBar.add(new ImageViewerSaveAsAction(this));
    toolBar.addSeparator();
    toolBar.add(new ImageViewerZoomInAction(this));
    toolBar.add(new ImageViewerZoomOutAction(this));
    toolBar.add(new ImageViewerZoomFitAction(this));
    toolBar.addSeparator();
    toolBar.add(new ImageViewerRestoreAction(this));
    return toolBar;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu.
   */
  private JMenu createHelpMenu() {
    final JMenu menu = new JMenu(getString("menu.help"));
    menu.add(new ApplicationAboutAction(this));
    return menu;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu.
   */
  private JMenu createFileMenu() {
    final JMenu menu = new JMenu(getString("menu.file"));
    menu.add(new ImageViewerOpenAction(this));
    menu.add(new ImageViewerSaveAction(this));
    menu.add(new ImageViewerSaveAsAction(this));
    menu.add(new ImageViewerCloseAction(this));
    menu.addSeparator();
    menu.add(new ApplicationExitAction(this));
    return menu;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu.
   */
  private JMenu createViewMenu() {
    final JMenu menu = new JMenu(getString("menu.view"));
    menu.add(new ImageViewerZoomInAction(this));
    menu.add(new ImageViewerZoomOutAction(this));
    menu.add(new ImageViewerZoomFitAction(this));
    return menu;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu.
   */
  private JMenu createEffectsMenu() {
    final JMenu menu = new JMenu(getString("menu.effects"));
    menu.add(new ImageViewerGrayAction(this));
    menu.add(new ImageViewerLightAction(this));
    menu.add(new ImageViewerRGBAction(this));
    menu.add(new ImageViewerBlurAction(this));
    menu.addSeparator();
    menu.add(new ImageViewerRestoreAction(this));
    return menu;
  }

  /**
   * Montagem do menu.
   * 
   * @return .
   */
  private JMenuBar createMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(createFileMenu());
    menuBar.add(createViewMenu());
    menuBar.add(createEffectsMenu());
    menuBar.add(createHelpMenu());
    return menuBar;
  }

  /**
   * Trmino ASAP da aplicao.
   */
  @Override
  public void killApplication() {
  }

  /**
   * Mtodo que permite sempre ao usurio matar a aplicao.
   * 
   * @return o flag indicativo.
   */
  @Override
  public boolean userCanKillApplication() {
    if (needSave) {
      final String msg = getString("msg.close.without.save");
      final JFrame frame = getApplicationFrame();
      final String title = getName();
      final int answer = StandardDialogs.showYesNoDialog(frame, title, msg);
      if (answer == 0) {
        return true;
      }
      return false;
    }
    return true;
  }

  /**
   * Recebe o arquivo para ser aberto diretamente
   * 
   * @param name .
   * @param value .
   */
  @Override
  public void sendMessage(String name, Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(PROJECT_FILE_MESSAGE)) {
      ClientProjectFile file = (ClientProjectFile) value;
      final boolean opened = openFile(file);
      if (opened) {
        setCurrentFile(file);
      }
    }
  }

  /**
   * Classe que implementa o observador utilizado na formao da imagem
   */
  class ImgObserver implements ImageObserver {
    /**
     * {@inheritDoc}
     */
    @Override
    public boolean imageUpdate(Image i, int f, int x, int y, int w, int h) {
      synchronized (ImageViewer.this) {
        if (((f & ERROR) == ERROR) || ((f & ABORT) == ABORT)) {
          currentImage = null;
          return false;
        }
        if ((f & WIDTH) == WIDTH) {
          width = w;
        }
        if ((f & HEIGHT) == HEIGHT) {
          height = h;
        }
        if ((f & ALLBITS) == ALLBITS) {
          ready = true;
          imagePanel.revalidate();
          imagePanel.repaint();
          return false;
        }
        return true;
      }
    }
  }

  /**
   * Painel de Exibio de imagens
   */
  class ImgPanel extends JPanel {
    /**
     * Construtor
     */
    public ImgPanel() {
      super();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void paintComponent(Graphics g) {
      if (currentImage == null) {
        super.paintComponent(g);
        return;
      }

      super.paintComponent(g);
      if (!ready) {
        return;
      }
      Graphics2D g2d = (Graphics2D) g;

      final int sizeX = (Math.round(imagePanel.getWidth()));
      final int sizeY = (Math.round(imagePanel.getHeight()));

      final int t2X = (int) (Math.round(width * zoom));
      final int t2Y = (int) (Math.round(height * zoom));

      int t3X = (int) (Math.round((sizeX - t2X) / 2.0));
      int t3Y = (int) (Math.round((sizeY - t2Y) / 2.0));
      if (t3X < 0) {
        t3X = 0;
      }
      if (t3Y < 0) {
        t3Y = 0;
      }

      final AffineTransform transf = new AffineTransform();
      transf.setToIdentity();
      transf.scale(zoom, zoom);
      transf.translate(t3X, t3Y);
      g2d.drawImage(currentImage, transf, observer);
    }
  }

  /**
   * Constri e exibe o dilogo de execuo de algoritmos
   * 
   * @param id .
   */
  public ImageViewer(final String id) {
    super(id);
    imagePanel = new ImgPanel();
    createMainFrame();
  }

  /**
   * Restaura a imagem original carregada
   */
  public void restoreOriginalImage() {
    if (originalImage == null) {
      return;
    }
    setCurrentImage(originalImage);
  }

  /**
   * Ajuste o arquivo corrente.
   * 
   * @param file novo arquivo.
   */
  public void setCurrentFile(ClientFile file) {
    if (file == null) {
      currentFile = null;
    }
    else {
      currentFile = file;
    }
    updateTitle();
  }
}
