package csbase.client.applications.preferenceviewer.tree;

import java.awt.Component;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JTree;
import javax.swing.tree.DefaultTreeCellRenderer;

import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.preferenceviewer.PreferencePanel;
import csbase.client.applications.preferenceviewer.PreferenceViewer;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.util.PreferencesUtil;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Renderizador de ns da rvore que exibe as categorias de preferncias.
 * 
 * @author Tecgraf
 */
class PreferenceTreeCellRenderer extends DefaultTreeCellRenderer {

  /** Referncia para o painel que exibe todas as preferncias. */
  private PreferencePanel preferencePanel;

  /** Referncia para a aplicao. */
  private PreferenceViewer app;

  /** Mapeia o nome do enum de uma preferncia para o identificador da app. */
  private Map<String, String> appMemoization;

  /** Mapeia a categoria de uma preferncia para o seu cone. */
  private Map<PreferenceCategory, Icon> iconMemoization;

  /**
   * Construtor padro.
   * 
   * @param preferencePanel
   * 
   * @param app - referncia para a aplicao.
   */
  public PreferenceTreeCellRenderer(PreferencePanel preferencePanel,
    PreferenceViewer app) {
    this.preferencePanel = preferencePanel;
    this.app = app;

    this.appMemoization = new HashMap<String, String>();
    this.iconMemoization = new HashMap<PreferenceCategory, Icon>();

    ApplicationManager am = ApplicationManager.getInstance();
    for (ApplicationRegistry registry : am.getAllApplicationRegistries()) {
      String enumName = PreferencesUtil.getAppEnumName(registry);
      appMemoization.put(enumName, registry.getId());
    }

    createAllIcons();
  }

  /** Constri todos os icones das categorias. */
  private void createAllIcons() {
    PreferenceManager pm = PreferenceManager.getInstance();
    PreferenceCategory root = pm.loadPreferences();

    defineIcons(root);
  }

  /**
   * Define o cone para a categoria e seus filhos recursivamente.
   * 
   * @param pc - categoria.
   */
  private void defineIcons(PreferenceCategory pc) {
    iconMemoization.put(pc, createPreferenceNodeIcon(pc));

    for (PreferenceCategory child : pc.getCategories()) {
      defineIcons(child);
    }
  }

  /**
   * Retorna o cone equivalente ao n da rvore. Esta aplicao adota as
   * seguintes condies:
   * 
   * - Preferncias de aplicao usam os prprios cones das aplicaes; <br/>
   * - Outras preferncias devem possuir seus cones no repositrio da
   * aplicao. O nome do arquivo deve ser <nome da enumerao>.gif;
   * 
   * @param pc - categoria de preferncia.
   * @return cone do n que exibe a categoria de preferncias.
   */
  private Icon createPreferenceNodeIcon(PreferenceCategory pc) {
    Icon icon = null;

    if (appMemoization.containsKey(pc.getId())) {
      ApplicationManager am = ApplicationManager.getInstance();
      String appId = appMemoization.get(pc.getId());
      icon = am.getApplicationIcon(appId);
    }
    else {
      int begin = pc.getId().lastIndexOf(".") + 1;
      int end = pc.getId().length();
      if (begin != -1) {
        String iconName = pc.getId().substring(begin, end) + ".gif";

        try (InputStream input = app.getResource(new String[] { iconName })) {
          if (input != null) {
            BufferedImage image = ImageIO.read(input);
            icon = new ImageIcon(image);
          }
        }
        catch (IOException e) {
          // faz nada
        }
      }
    }
    return icon;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTreeCellRendererComponent(JTree tree, Object value,
    boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
    JLabel cell =
      (JLabel) super.getTreeCellRendererComponent(tree, value, sel, expanded,
        leaf, row, hasFocus);

    PreferenceCategory pc = (PreferenceCategory) value;

    if (preferencePanel.getModifiedCategories() != null
      && preferencePanel.getModifiedCategories().contains(pc)) {
      cell.setText(pc.getLabel() + "*");
    }
    else {
      cell.setText(pc.getLabel() + " ");
    }
    cell.setIcon(iconMemoization.get(pc));

    return cell;
  }

}
