/*
 * $Id$
 */
package csbase.client.desktop;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.remote.srvproxies.UserProxy;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.remote.ClientRemoteLocator;

/**
 * Tela para envio de notificaes entre usurios.
 * 
 * @author Tecgraf/PUC-Rio
 */
final public class NotificationCompositionFrame extends
  AbstractNotificationFrame {

  /**
   * Opo de todos os usurios
   */
  final private JRadioButton allUsersButton = new JRadioButton();

  /**
   * Opo de usurios do projeto
   */
  final private JRadioButton projectUsersButton = new JRadioButton();

  /**
   * Opo de usurios conectados
   */
  final protected JCheckBox connectedUsersCheck = new JCheckBox();

  /**
   * Boto para selecionar todos os usurios
   */
  final private JButton selectAllUsersButton = new JButton();

  /**
   * Tabela de usurios.
   */
  private JTable usersTable;

  /**
   * Inicializao
   */
  @Override
  protected void preShowInit() {
    this.updateUsers();
    this.updateProjectUsersStatus();
  }

  /**
   * Obtem a lista de todos os usurios cadastrados.
   * 
   * @return lista com todos os usurios cadastrados
   */
  protected List<UserOutline> getUserIdsList() {
    String msg = LNG.get("notification.message.waiting");
    String title = getTitle();
    Vector<UserOutline> allOutlines =
      UserProxy.getAllOutlines(getOwner(), title, msg);

    User loggedUser = User.getLoggedUser();
    Object loggedUserOutline = null;
    try {
      loggedUserOutline = loggedUser.getOutline();
      allOutlines.remove(loggedUserOutline);
    }
    catch (Exception e) {
      // Nada a fazer. O usrio loggado deve existir...
      e.printStackTrace();
    }
    return allOutlines;
  }

  /**
   * Verifica se o projeto est sendo compartilhado com algum usurio.
   * 
   * @return true se o projeto est sendo compartilhado com algum usurio
   */
  private boolean projectHasUsers() {
    CommonClientProject project = getOpenedProject();
    if (project == null || !project.isShared()) {
      return false;
    }
    return !project.getUsersRO().isEmpty() || !project.getUsersRW().isEmpty();
  }

  /**
   * Cria a tabela de usurios.
   * 
   * @return tabela de usurios
   */
  private JTable createUsersTable() {
    SortableTable table =
      new SortableTable(new NotificationRecipientsTableModel());
    table.setRowSelectionAllowed(true);
    table.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    // preenche todo o espao vertical do scroll pane
    table.setFillsViewportHeight(true);
    return table;
  }

  /**
   * Cria o painel de opes de envio
   * 
   * @return painel de opes
   */
  private JPanel createUsersTablePanel() {
    /*
     * tabela com os usurios
     */
    usersTable = createUsersTable();
    JScrollPane tableScrollPane = new JScrollPane(usersTable);
    tableScrollPane.setPreferredSize(new Dimension(250, 150));

    /*
     * radio-buttons
     */
    ButtonGroup buttonGroup = new ButtonGroup();
    allUsersButton.setText(LNG.get("notification.all.users.label"));
    projectUsersButton.setText(LNG
      .get("notification.current.project.users.label"));
    buttonGroup.add(allUsersButton);
    buttonGroup.add(projectUsersButton);

    allUsersButton.setSelected(true);
    this.updateProjectUsersStatus();
    allUsersButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        updateUsers();
      }
    });
    projectUsersButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        updateUsers();
      }
    });

    /*
     * checkbox "somente conectados"
     */
    connectedUsersCheck.setText(LNG.get("notification.connected.users.label"));
    connectedUsersCheck.setSelected(true);
    connectedUsersCheck.addItemListener(new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        updateUsers();
      }
    });

    /*
     * boto para seleo de todos os usurios
     */
    selectAllUsersButton.setText(LNG.get("notification.selectall.users.label"));
    this.selectAllUsersButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        final int numRows = usersTable.getRowCount();
        if (numRows == 0) {
          return;
        }
        usersTable.setRowSelectionInterval(0, numRows - 1);
      }
    });

    JPanel panel = new JPanel(new GridBagLayout());
    String userTitle = LNG.get("notification.user.label");
    panel.setBorder(BorderFactory.createTitledBorder(userTitle));

    Insets border = new Insets(0, 5, 0, 5);

    GBC gbc = new GBC(0, 0).northwest().insets(border);
    panel.add(allUsersButton, gbc);

    gbc = new GBC(0, 1).northwest().insets(border);
    panel.add(projectUsersButton, gbc);

    gbc =
      new GBC(1, 0).height(2).width(2).northwest().insets(0, 5, 0, 10).both();
    panel.add(tableScrollPane, gbc);

    gbc = new GBC(1, 3).northwest().insets(10, 0, 5, 20);
    panel.add(connectedUsersCheck, gbc);

    gbc = new GBC(2, 3).east().insets(10, 0, 10, 10).pushx();
    panel.add(selectAllUsersButton, gbc);

    return panel;
  }

  /**
   * Atualizao de status de usurios do projeto
   */
  private void updateProjectUsersStatus() {
    projectUsersButton.setEnabled(projectHasUsers());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getWindowTitle() {
    return String.format("%s - %s", LNG.get("notification.composition.title"),
      LNG.get("SERVER"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void addSubPanels(Container mainPane) {
    /*
     * painel para seleo de usurios
     */
    GBC gbc = new GBC().both(1.0, 0.5).northwest().insets(10);
    mainPane.add(createUsersTablePanel(), gbc);
    /*
     * painel com a mensagem
     */
    JPanel messagePanel =
      createEditionPanel(LNG.get("notification.compose.panel.title"));
    gbc = new GBC(0, 1).both().northwest().insets(0, 10, 10, 10);
    mainPane.add(messagePanel, gbc);
    /*
     * painel com os botes
     */
    gbc = new GBC(0, 2).horizontal().bottom(10);
    mainPane.add(createButtonsPane(), gbc);
  }

  /**
   * Consulta ao projeto aberto.
   * 
   * @return o projeto aberto.
   */
  private CommonClientProject getOpenedProject() {
    return DesktopFrame.getInstance().getProject();
  }

  /**
   * Preenche a tabela.
   * 
   * @param users lista de usurios
   */
  private void fillUsersTable(List<UserOutline> users) {
    int[] modelIndexes = getSelectedModelIndexes();
    NotificationRecipientsTableModel model =
      (NotificationRecipientsTableModel) usersTable.getModel();
    if (users == null) {
      model.setUsers();
    }
    else {
      List<Integer> selectedUsersIndexes = model.setUsers(users, modelIndexes);
      for (Integer modelIndex : selectedUsersIndexes) {
        int i = usersTable.convertRowIndexToView(modelIndex);
        usersTable.addRowSelectionInterval(i, i);
      }
    }
  }

  /**
   * Retorna um conjunto com os {@link UserOutline} de todos os usurios que tm
   * acesso ao projeto (RO ou RW).
   * 
   * @return conjunto com todos os usurios que tm acesso ao projeto (RO ou RW)
   */
  protected Set<UserOutline> getProjectUsersIdsSet() {
    CommonClientProject project = getOpenedProject();
    if (project == null) {
      return null;
    }
    Set<UserOutline> prjUsers = new HashSet<UserOutline>();
    addUserOutlines(prjUsers, project.getUsersRO());
    addUserOutlines(prjUsers, project.getUsersRW());
    addOwnerOutline(prjUsers, project.getUserId());
    return prjUsers;
  }

  /**
   * Acrescenta o usurio dono do projeto se o mesmo <b>no</b> o usurio
   * loggado no sistema.
   * 
   * @param prjUsers lista de usurios
   * @param ownerUserId id do usurio dono do projeto.
   */
  private void addOwnerOutline(Set<UserOutline> prjUsers, Object ownerUserId) {
    Object loggedUserId = User.getLoggedUser().getId();
    if (loggedUserId.equals(ownerUserId)) {
      return;
    }
    try {
      User ownerUser = User.getUser(ownerUserId);
      prjUsers.add(ownerUser.getOutline());
    }
    catch (Exception e) {
      // Nada a fazer. O usrio loggado deve existir...
      e.printStackTrace();
    }
  }

  /**
   * Acrescenta os {@link UserOutline} associados a um conjunto de usurios a um
   * conjunto previamente definido.
   * 
   * @param prjUsers - conjunto final de <code>UserOutline</code>
   * @param usersIds - conjunto com os identificadores dos usurios que sero
   *        adicionados
   */
  private void addUserOutlines(Set<UserOutline> prjUsers, Set<Object> usersIds) {
    Object loggedUserId = User.getLoggedUser().getId();
    for (Object userId : usersIds) {
      if (loggedUserId.equals(userId)) {
        continue;
      }
      try {
        User user = User.getUser(userId);
        prjUsers.add(user.getOutline());
      }
      catch (Exception e) {
        continue;
      }
    }
  }

  /**
   * Atualizao de usurios
   */
  private void updateUsers() {
    List<UserOutline> usersList;
    if (allUsersButton.isSelected()) {
      usersList = getUserIdsList();
    }
    else {
      usersList = new ArrayList<UserOutline>(getProjectUsersIdsSet());
    }

    if (this.connectedUsersCheck.isSelected()) {
      try {
        // FIXME implementar getLoggedUsersSet em LoginService
        UserOutline[] connectedUsersArray =
          ClientRemoteLocator.server.getLoggedUsers();
        List<UserOutline> connectedUsersList =
          Arrays.asList(connectedUsersArray);
        usersList.retainAll(connectedUsersList);
      }
      catch (RemoteException e) {
        e.printStackTrace();
      }
    }
    fillUsersTable(usersList);
  }

  /**
   * Constri e exibe o dilogo de execuo de algoritmos
   */
  public NotificationCompositionFrame() {
    buildUI();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Object> getRecipientsIDs() {
    int[] selectedRows = getSelectedModelIndexes();
    NotificationRecipientsTableModel model =
      (NotificationRecipientsTableModel) usersTable.getModel();
    return model.getIDsFor(selectedRows);
  }

  /**
   * Obtm os ndices <b>do modelo</b> correspondentes s linhas selecionadas.
   * 
   * @return ndices <b>do modelo</b> correspondentes s linhas selecionadas
   */
  private int[] getSelectedModelIndexes() {
    int[] selectedRows = usersTable.getSelectedRows();
    for (int i = 0; i < selectedRows.length; i++) {
      selectedRows[i] = usersTable.convertRowIndexToModel(selectedRows[i]);
    }
    return selectedRows;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasRecipients() {
    return usersTable.getSelectedRowCount() > 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ActionListener getSendButtonActionListener() {
    return new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        if (!hasRecipients()) {
          StandardDialogs.showErrorDialog(NotificationCompositionFrame.this,
            null, LNG.get("notification.composition.error.noRecipients"));
          return;
        }
        sendMessage(!connectedUsersCheck.isSelected());
      }
    };
  }
}
