package csbase.client.preferences;

import java.awt.Font;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.border.TitledBorder;

import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.util.PreferenceEditorListener;

/**
 * Classe que encapsula o editor de um valor de preferncia.
 * 
 * Editores de preferncias possuem a responsabilidade de manipular toda a
 * lgica que envolve a edio de um valor de preferncia. Essa lgica envolve a
 * criao de uma cpia do valor original da preferncia e, com isso, permite
 * que o usurio possa cancelar a edio sem modificar o contedo original do
 * valor.
 * 
 * @param <T> tipo do valor editado.
 * 
 * @author Tecgraf
 */
public abstract class PreferenceEditor<T> {

  /** Referncia para o valor da preferncia. */
  private PreferenceValue<T> pv;

  /** Objeto que est sendo editado. */
  private T value;

  /** Fonte usada no editor. */
  protected static Font font = new Font(Font.SANS_SERIF, Font.PLAIN, 13);

  /** Lista com todos os ouvintes deste editor. */
  private List<PreferenceEditorListener> listeners;

  /**
   * Construtor padro.
   * 
   * @param pv objeto que encapsula o valor da preferncia.
   */
  public PreferenceEditor(PreferenceValue<T> pv) {
    if (pv == null) {
      throw new IllegalArgumentException("value no pode ser nulo.");
    }
    this.pv = pv;
    this.listeners = new ArrayList<PreferenceEditorListener>();
  }

  /**
   * Rtulo da preferncia.
   * 
   * @return rtulo da preferncia.
   */
  public String getLabel() {
    return pv.getLabel();
  }

  /**
   * Seta o valor que est sendo editado.
   * 
   * @param newValue valor editado.
   */
  public void setValue(T newValue) {
    this.value = newValue;
  }

  /**
   * Retorna o valor.
   * 
   * @return valor.
   */
  public T getValue() {
    return value;
  }

  /** Persiste o valor que est sendo editado. */
  public void commit() {
    pv.setValue(value);
  }

  /**
   * Retorna o componente de edio da preferncia.
   * 
   * @param showDefaultValue true se for para criar o componente com o valor
   *        default da preferncia, false caso contrrio.
   * 
   * @return componente de edio da preferncia.
   */
  public JComponent getComponent(boolean showDefaultValue) {
    if (showDefaultValue) {
      setValue(pv.getDefaultValue());
    }
    else {
      PreferenceValue<T> v = pv.clone();
      setValue(v.getValue());
    }

    return createComponent();
  }

  /**
   * Retorna a poltica de visibilidade do valor da preferncia.
   * 
   * @return poltica de visibilidade do valor da preferncia.
   */
  public boolean isEditable() {
    return PreferencePolicy.READ_WRITE.equals(pv.getPolicy());
  }

  /**
   * Adiciona ouvinte do editor de preferncia.
   * 
   * @param l ouvinte do editor.
   */
  public void addListener(PreferenceEditorListener l) {
    listeners.add(l);
  }

  /**
   * True se este editor possui ouvintes cadastrados, false caso contrrio.
   * 
   * @return true se este editor possui ouvintes cadastrados, false caso
   *         contrrio.
   */
  public boolean hasListeners() {
    return !listeners.isEmpty();
  }

  /**
   * Remove ouvinte do editor de preferncia.
   * 
   * @param l ouvinte do editor.
   */
  public void removeListener(PreferenceEditorListener l) {
    listeners.remove(l);
  }

  /** Notifica todos os ouvintes deste editor. */
  public void notifyListeners() {
    for (PreferenceEditorListener l : listeners) {
      l.action();
    }
  }

  /**
   * Define uma borda no painel e usa o rtulo da preferncia como ttulo.
   * 
   * @param panel painel.
   */
  protected void setTitledBorder(JPanel panel) {
    setTitledBorder(panel, pv.getLabel());
  }

  /**
   * Define uma borda no painel e usa o rtulo da preferncia como ttulo.
   * 
   * @param panel painel.
   * @param title ttulo do painel.
   */
  protected void setTitledBorder(JPanel panel, String title) {
    panel.setBorder(BorderFactory.createTitledBorder(null, title,
      TitledBorder.LEADING, TitledBorder.DEFAULT_POSITION, font, null));
  }

  /**
   * Retorna o texto de internacionalizao.
   * 
   * @param tag chave do texto a ser internacionalizado.
   * @return texto internacionalizado.
   */
  protected String getString(String tag) {
    return pv.getPreferenceBundle().get(tag);
  }

  /**
   * Retorna true se existe um valor para a dada propriedade, false caso
   * contrrio.
   * 
   * @param tag chave do texto a ser internacionalizado.
   * @return true se existe um valor para a dada propriedade, false caso
   *         contrrio.
   */
  protected boolean hasString(String tag) {
    return pv.getPreferenceBundle().has(tag);
  }

  /**
   * Retorna o texto de internacionalizao.
   * 
   * @param tag chave do texto a ser internacionalizado.
   * @return texto internacionalizado.
   */
  protected String getPrefixedString(String tag) {
    return getString(getClass().getSimpleName() + "." + tag);
  }

  /**
   * Cria o componente de edio da preferncia.
   * 
   * @return cria o componente de edio da preferncia.
   */
  protected abstract JComponent createComponent();
}
