/*
 * $Id$
 */
package csbase.client.project;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;
import java.util.Date;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.border.BevelBorder;

import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.MetalThemeAdapter;
import csbase.logic.User;
import csbase.logic.Utilities;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe que cria um dilogo que permite ao usurio adicionar uma descrio
 * para um <code>ProjectFile</code>.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class AppendDescriptionDialog {
  /**
   * Dilogo
   */
  private DesktopComponentDialog descriptionDialog;

  /**
   * Boto default para fechar o dilogo.
   */
  private JButton closeButton;

  /**
   * Campo de descrio do arquivo.
   */
  private JTextArea description;

  /**
   * Referncia para o dilogo pai
   */
  private FilePropertiesDialog owner;

  /**
   * Cria o dilogo.
   * 
   * @param owner referncia para o dilogo pai.
   */
  public AppendDescriptionDialog(FilePropertiesDialog owner) {
    this.owner = owner;
    descriptionDialog = makeDialog();
  }

  /**
   * Constri o dilogo.
   * 
   * @return dilogo.
   */
  private DesktopComponentDialog makeDialog() {
    final JDialog dlg = owner.getDialog();
    final String title = LNG.get("PRJ_FILE_ADD_DESCRIPTION");
    DesktopComponentDialog aDialog = new DesktopComponentDialog(dlg, title);
    aDialog.setModal(true);
    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(makeFieldPanel());
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);
    aDialog.getContentPane().add(mainPanel);
    aDialog.pack();
    aDialog.center(dlg);
    aDialog.getRootPane().setDefaultButton(closeButton);
    return aDialog;
  }

  /**
   * Constri o painel de botes.
   * 
   * @return painel de botes.
   */
  private Component makeButtonPanel() {
    JPanel panel = new JPanel();
    JButton addButton = new JButton(LNG.get("PRJ_FILE_ADD"));
    panel.add(addButton);
    addButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        addDescription();
      }
    });
    closeButton = new JButton(LNG.get("UTIL_CLOSE"));
    panel.add(closeButton);
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        descriptionDialog.close();
      }
    });
    return panel;
  }

  /**
   * Adiciona uma mensagem do usurio  descrio do arquivo.
   */
  private void addDescription() {
    String date = Utilities.getFormattedDate((new Date()).getTime());
    String userLogin = User.getLoggedUser().getLogin();
    String userName = User.getLoggedUser().getName();
    MessageFormat formatter =
      new MessageFormat(LNG.get("PRJ_FILE_MSG_SAVED_DESCRIPTION"));
    String message = description.getText().trim();
    Object[] arg = { date, userLogin, userName, message };
    message = formatter.format(arg);
    /* Adiciona texto  descrio - chamada remota */
    appendFileDescription(message);
    /* Atualiza dilogo original */
    owner.updateDescription();

    descriptionDialog.close();

  }

  /**
   * Adiciona a descrio a um arquivo
   * 
   * @param message Texto a ser adiconado a uma descrio
   */
  private void appendFileDescription(final String message) {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        owner.getFile().appendDescription(message);
      }
    };
    task.execute(owner.getDialog(), owner.getTitle(),
      LNG.get("PRJ_FILE_ADD_DESCRIPTION_TASK_MSG"));
  }

  /**
   * Constri o painel de campos.
   * 
   * @return painel de campos.
   */
  private Component makeFieldPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();

    // Instrues
    c.gridx = 0;
    c.gridy = 0;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(8, 10, 10, 3); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.WEST;
    JLabel instruction =
      new JLabel("Por favor, entre com a descrio desejada:");
    instruction.setBackground(MetalThemeAdapter.getInstance().getSecondary3());
    panel.add(instruction, c);

    // Campo de Descrio
    c.gridx = 0;
    c.gridy = 1;
    c.gridwidth = 1;
    c.gridheight = 8;
    c.weightx = 1;
    c.weighty = 1;
    c.insets = new Insets(3, 10, 10, 8); // tlrb
    c.fill = GridBagConstraints.BOTH;
    c.anchor = GridBagConstraints.WEST;
    description = new JTextArea(8, 30);
    description.setLineWrap(true);
    description.setWrapStyleWord(true);
    description.setBorder(new BevelBorder(BevelBorder.LOWERED));
    JScrollPane scrollPane = new JScrollPane(description);
    panel.add(scrollPane, c);
    return panel;
  }

  /**
   * Exibe/Oculta o dilogo.
   * 
   * @param visible indicativo
   */
  public void setVisible(boolean visible) {
    descriptionDialog.setVisible(visible);
  }
}
