/*
 * UserContainerSelection.java
 * 
 * $Author$ $Revision$ - $Date: 2006-08-22 16:07:04 -0300 (Tue,
 * 22 Aug 2006) $
 */
package csbase.client.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.logic.UserOutline;

/**
 * A classe <code>UserContainerSelection</code> modela um componente
 * ContainerSelection que apresenta duas listas de usurios. Este componente
 * deve ser usado em qualquer dilogo que se deseje gerenciar usurios.
 */
public class UserContainerSelection {
  /** Componente de seleo dos usurios. */
  private ContainerSelection<UserTarget> selectableItems = null;

  /** Indica se o container deve ser habilitado ou no. */
  private boolean enabled;

  /** Usurios disponveis */
  private Vector<UserOutline> availableUsers;

  /** Usurios selecionados */
  private Vector<UserOutline> selectedUsers;

  /** Define se sero exibidos ttulos nas tabelas do continer. */
  private boolean showTitles;

  public static JPanel getUserContainerSelection(boolean enabled,
    Vector<UserOutline> availableUsers, Vector<UserOutline> selectedUsers) {
    return new UserContainerSelection(enabled, availableUsers, selectedUsers)
      .getPanel();
  }

  /**
   * Cria o painel com o componente que permite selecionar os usurios. Usa um
   * componente que mostra dois conjuntos, disponveis e escolhidos, nos quais o
   * usurio pode movimentar elementos entre eles.
   * 
   * @return Panel do container.
   */
  public JPanel getPanel() {
    /* Especifica o formatador dos elementos disponveis para seleo. */
    ObjectTableProvider firstFormatter = getFormatForAvailableItems();
    /* Especifica o formatador dos elementos escolhidos. */
    ObjectTableProvider secondFormatter = getFormatForSelectedItems();
    /*
     * Cria o componente que mostra os dois conjuntos de elementos: um com os
     * disponveis e o outro com os escolhidos.
     */
    if (this.showTitles) {
      selectableItems =
        new ContainerSelection<UserTarget>(firstFormatter, secondFormatter,
          enabled, false, LNG.get("IAS_AVAILABLE_USERS"),
          LNG.get("IAS_SELECTED_USERS"));
    }
    else {
      selectableItems =
        new ContainerSelection<UserTarget>(firstFormatter, secondFormatter,
          enabled, false);
    }

    /* Monta a lista dos usurios disponveis e selecionados. */
    if ((selectedUsers != null) || (selectedUsers.size() > 0)) {
      availableUsers.removeAll(selectedUsers);
    }
    /*
     * Carrega o conjunto de elementos disponveis com os usurios e grupos
     * cadastrados.
     */
    selectableItems.loadItems(UserTarget.toUserTarget(availableUsers),
      UserTarget.toUserTarget(selectedUsers));
    selectableItems.adjustTableColumns();
    return selectableItems.getPanel();
  }

  /**
   * Retorna os ids dos usurios selectionados.
   * 
   * @return lista de usurios selecionados.
   */
  public Object[] getSelectedItems() {
    List<UserTarget> selectedUsers = selectableItems.getSelectedItems();
    final Object[] userIds = new Object[selectedUsers.size()];
    for (int i = 0; i < selectedUsers.size(); i++) {
      userIds[i] = selectedUsers.get(i).getUserId();
    }
    return userIds;
  }

  /**
   * Retorna o id do Usurio selecionado
   * 
   * @param index .
   * 
   * @return .
   */
  public Object getUserId(int index) {
    List<UserTarget> users = selectableItems.getSelectedItems();
    if (users == null) {
      return null;
    }
    UserTarget user = users.get(index);
    if (user == null) {
      return null;
    }
    return user.getUserId();
  }

  /**
   * Adiciona um listener de modificao do container de seleo. Esse listener
   *  notificado sempre que algum elemento  colocado ou removido do container
   * com os elementos selecionados.
   * 
   * @param l O listener de alterao do contedo do container.
   */
  public void addContainerSelectionListener(ContainerSelectionListener l) {
    selectableItems.addContainerSelectionListener(l);
  }

  /**
   * Devolve o formatador dos elementos disponveis para seleo.
   * 
   * @return .
   */
  private ObjectTableProvider getFormatForAvailableItems() {
    return new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_USER"), LNG.get("IAS_USER_NAME") };
      }

      /**
       * {@inheritDoc}
       */
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class, String.class };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        UserTarget target = (UserTarget) item;
        if (target == null) {
          return null;
        }
        String[] format = { target.getLogin(), target.getName() };
        return format;
      }
    };
  }

  /**
   * Devolve o formatador dos elementos escolhidos.
   * 
   * @return .
   */
  private ObjectTableProvider getFormatForSelectedItems() {
    return new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_USER"), LNG.get("IAS_USER_NAME") };
      }

      /**
       * {@inheritDoc}
       */
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class, String.class };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        UserTarget target = (UserTarget) item;
        if (target == null) {
          return null;
        }
        String[] format = { target.getLogin(), target.getName() };
        return format;
      }
    };
  }

  public ContainerSelection<UserTarget> getContainerSelection() {
    return selectableItems;
  }

  /**
   * Construtor
   * 
   * @param enabled Indica se o container deve ser habilitado ou no.
   * @param availableUsers Usurios disponveis.
   * @param selectedUsers Usurio j selecionados.
   */
  public UserContainerSelection(boolean enabled,
    Vector<UserOutline> availableUsers, Vector<UserOutline> selectedUsers) {
    this(enabled, availableUsers, selectedUsers, false);
  }

  public UserContainerSelection(boolean enabled,
    Vector<UserOutline> availableUsers, Vector<UserOutline> selectedUsers,
    boolean showTitles) {
    this.enabled = enabled;
    this.availableUsers = availableUsers;
    this.selectedUsers = selectedUsers;
    this.showTitles = showTitles;
  }

  public void setEnabled(boolean enabled) {
    this.selectableItems.setEnabled(enabled);
  }

  public void reset() {
    List<UserTarget> allUsers = new ArrayList<UserTarget>();
    if (this.selectableItems.hasAvailableItems()) {
      allUsers.addAll(this.selectableItems.getAvailableItems());
    }
    if (this.selectableItems.hasSelectedItems()) {
      allUsers.addAll(this.selectableItems.getSelectedItems());
    }
    this.selectableItems.clear();
    this.selectableItems.loadItems(allUsers);
  }
}

/**
 * A classe <code>UserTarget</code> modela um elemento que pode ser um grupo ou
 * um usurio.
 */
class UserTarget {
  UserOutline userOutline;

  String getLogin() {
    return userOutline.getLogin();
  }

  String getName() {
    return userOutline.getName();
  }

  Object getUserId() {
    return userOutline.getId();
  }

  static Vector<UserTarget> toUserTarget(Vector<UserOutline> outlines) {
    int size = outlines.size();
    Vector<UserTarget> targets = new Vector<UserTarget>(size);
    for (int i = 0; i < size; i++) {
      UserOutline user = outlines.elementAt(i);
      targets.add(i, new UserTarget(user));
    }
    return targets;
  }

  UserTarget(UserOutline userOutline) {
    this.userOutline = userOutline;
  }
}
