/*
 * $Author$ $Date$
 * $Release: $
 */
package csbase.client.util.sga;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import csbase.client.kernel.ClientException;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;

/**
 * Define um painel para seleo mltipla de SGAs.
 * 
 * @author ururahy
 * 
 */
public class SGAMultipleSelectionPanel extends SGASelectionPanel {
  /**
   * Rtulo do nmero de servidores.
   */
  private JLabel numServerLabel;

  /**
   * Campo de texto do nmero de servidores.
   */
  private JTextField numServerTextField;

  /**
   * Rtulo de nmero de execues.
   */
  private JLabel numExecLabel;

  /**
   * Campo de texto do nmero de execues.
   */
  private JTextField numExecTextField;

  /**
   * Caixa de seleo mltipla de servidores.
   */
  private ContainerSelection<SGASet> sgaServerContainerSelection;

  /**
   * Painel do nmero de servidores.
   */
  private JPanel numCPUPanel;

  /**
   * Painel do nmero de execues.
   */
  private JPanel numExecPanel;

  /**
   * Define um painel para seleo mltipla de SGAs.
   * 
   * @param configurator o configurador de algoritmos.
   */
  public SGAMultipleSelectionPanel(AlgorithmConfigurator configurator) {
    super(configurator);
    this.numServerLabel =
      new JLabel(LNG.get("SGAMultipleSelectionPanel.label.num_exec"));
    this.numServerTextField = new MyJTextField();
    this.sgaServerContainerSelection = makeSGASelectionContainer();
    this.numCPUPanel =
      this.createLabeledTextPanel(numServerLabel, numServerTextField);
    this.numExecLabel =
      new JLabel(LNG.get("SGAMultipleSelectionPanel.label.num_execution"));
    this.numExecTextField = new MyJTextField();
    this.numExecPanel =
      this.createLabeledTextPanel(numExecLabel, numExecTextField);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Collection<SGASet> getManualServerCollection() {
    return this.sgaServerContainerSelection.getSelectedItems();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SGASet getManualSelectedServer() {
    Collection<SGASet> serverCollection = getSelectedServerCollection();
    if (serverCollection.isEmpty()) {
      return null;
    }
    return serverCollection.iterator().next();
  }

  /**
   * Obtm o nmero de execues que sero disparadas na execuo mltipla.
   * 
   * @return nmero de execues
   */
  public Integer getExecutionCountForMultipleExecution() {
    if (this.numServerTextField.getText().equals("")) {
      return null;
    }
    return Integer.parseInt(this.numServerTextField.getText());
  }

  /**
   * Obtm o nmero de execues por sga que ser utilizado na execuo mltipla
   * manual.
   * 
   * @return .
   */
  public Integer getExecutionCountPerSGAForMultipleExecution() {
    if (this.numExecTextField.getText().equals("")) {
      return null;
    }
    return Integer.parseInt(this.numExecTextField.getText());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasAvailableServers() {
    return this.sgaServerContainerSelection.hasItems();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSelectionReady() {
    if (!this.hasAvailableServers()) {
      return false;
    }
    if (!isManualServerSelection()) {
      if (this.numServerTextField.getText().equals("")) {
        return false;
      }
      return true;
    }
    if (this.numExecTextField.getText().equals("")) {
      return false;
    }
    Collection<SGASet> serverCollection = getSelectedServerCollection();
    if (serverCollection.isEmpty()) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void populateGUIComponents() {
    this.removeAll();
    this.setLayout(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.WEST;
    c.gridx = 1;
    c.gridy = 1;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.insets = new Insets(2, 2, 2, 2);
    c.weightx = 1.0;
    c.weighty = 0.0;
    c.fill = GridBagConstraints.HORIZONTAL;
    this.add(this.platformFilterPanel, c);
    c.insets = new Insets(2, 2, 2, 2);
    c.gridx = 1;
    c.gridy++;
    this.add(this.automaticServerRadioButton, c);
    c.insets = new Insets(2, 20, 2, 2);
    c.gridy++;
    this.add(this.numCPUPanel, c);
    c.insets = new Insets(2, 2, 2, 2);
    c.gridx = 1;
    c.gridy++;
    this.add(this.manualServerRadioButton, c);
    c.insets = new Insets(2, 20, 2, 2);
    c.gridy++;
    this.add(this.numExecPanel, c);
    c.insets = new Insets(2, 20, 2, 2);
    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridy++;
    this.add(this.sgaServerContainerSelection.getPanel(), c);
    this.revalidate();
    this.updateEnabledServers();
  }

  /**
   * Cria um painel para a insero de um Label/Texto.
   * 
   * @param label Ttulo que aparece ao lado esquerdo da caixa de texto.
   * @param textField Caixa de texto que aparece ao lado direto do ttulo.
   * 
   * @return o painel contendo o ttulo com a caixa de texto.
   */
  private JPanel createLabeledTextPanel(JLabel label, JTextField textField) {
    JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.WEST;
    c.insets = new Insets(2, 2, 2, 2);
    c.gridx = 1;
    c.gridy = 1;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0.0;
    c.weighty = 0.0;
    c.fill = GridBagConstraints.NONE;
    panel.add(label, c);
    c.weightx = 1.0;
    c.gridx++;
    panel.add(textField, c);
    return panel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateEnabledServers() {
    if (this.hasAvailableServers()) {
      if (this.manualServerRadioButton.isSelected()) {
        this.sgaServerContainerSelection.setEnabled(true);
        this.numServerTextField.setText("");
        this.numServerLabel.setEnabled(false);
        this.numServerTextField.setEnabled(false);
        this.numExecLabel.setEnabled(true);
        this.numExecTextField.setText("1");
        this.numExecTextField.setEnabled(true);
      }
      else {
        this.sgaServerContainerSelection.setEnabled(false);
        this.numServerTextField.setEnabled(true);
        ((MyJTextField) this.numServerTextField).setText("1");
        this.numServerLabel.setEnabled(true);
        this.numExecTextField.setText("");
        this.numExecLabel.setEnabled(false);
        this.numExecTextField.setEnabled(false);
      }
      this.manualServerRadioButton.setEnabled(true);
      this.automaticServerRadioButton.setEnabled(true);
    }
    else {
      this.manualServerRadioButton.setEnabled(false);
      this.automaticServerRadioButton.setEnabled(false);
      this.numServerLabel.setEnabled(false);
      this.sgaServerContainerSelection.setEnabled(false);
      this.numServerTextField.setEnabled(false);
      this.numExecLabel.setEnabled(false);
      this.numExecTextField.setEnabled(false);
    }
  }

  /**
   * Recarrega os SGAs disponveis (podem ser filtrados por plataforma). Se, na
   * lista de todos os SGAs disponveis (no apenas os filtrados) houver mais de
   * uma plataforma, mantm a coluna que indica a plataforma do SGA.
   */
  @Override
  public void reloadSGAs() {
    try {
      sgaServerContainerSelection.loadItems(this.loadServers());
      updateEnabledServers();
    }
    catch (ClientException e) {
      e.printStackTrace();
    }
  }

  /**
   * Cria um componente do tipo ContainerSelection para a seleo de SGAs.
   * 
   * @return O ContainerSelection criado e j populado com os SGAs disponveis.
   */
  public ContainerSelection<SGASet> makeSGASelectionContainer() {
    ContainerSelection<SGASet> container = null;
    try {
      // Exibe uma coluna com a plataforma no SGA, caso haja mais
      // de uma plataforma.
      Vector<SGASet> sgas = this.loadServers();
      boolean showPlats = false;
      Iterator<SGASet> sgasIterator = sgas.iterator();
      if (sgasIterator.hasNext()) {
        SGASet sga = sgasIterator.next();
        String platform = sga.getPlatformId();
        while (sgasIterator.hasNext()) {
          sga = sgasIterator.next();
          if (!platform.equals(sga.getPlatformId())) {
            showPlats = true;
            break;
          }
        }
      }
      container =
        new ContainerSelection<SGASet>(
          new SGASetTableProvider(LNG.get("MON_GROUP_SERVERS_TITLE"), showPlats),
          new SGASetTableProvider(LNG.get("MON_GROUP_SERVERS_TITLE"), showPlats),
          true, false, LNG.get("MON_GROUP_SELECTABLE_SERVERS"), LNG
            .get("MON_GROUP_SELECTED_SERVERS"));
      container.loadItems(sgas);
      container.adjustTableColumns();
      container.addContainerSelectionListener(new ContainerSelectionListener() {
        @Override
        public void containerChanged() {
          fireChangedServer();
        }
      });
    }
    catch (ClientException e) {
      e.printStackTrace();
    }
    return container;
  }

  /**
   * A classe MyJTextField cria um componente do tipo JTextField onde os
   * caracteres vlidos para a insero na caixa de texto consiste apenas de
   * dgitos numricos.
   */
  private class MyJTextField extends JTextField {

    /**
     * Construtor.
     */
    public MyJTextField() {
      addKeyListener(new KeyAdapter() {
        @Override
        public void keyTyped(KeyEvent ev) {
          if ((!Character.isDigit(ev.getKeyChar()))
            && (!(ev.getKeyChar() == KeyEvent.VK_BACK_SPACE))) {
            ev.consume();
          }
        }

        @Override
        public void keyReleased(KeyEvent ev) {
          if ((Character.isDigit(ev.getKeyChar()))
            || (ev.getKeyChar() == KeyEvent.VK_BACK_SPACE)
            || (ev.getKeyChar() == KeyEvent.VK_DELETE)) {
            fireChangedServer();
          }
        }
      });
      this.setColumns(5);
    }
  }

  /**
   * Provedor de informaes da caixa de seleo mltipla de servidores.
   */
  private class SGASetTableProvider extends DefaultObjectTableProvider {

    /**
     * Ttulo da coluna.
     */
    private String columnTitle;

    /**
     * Determina se as plataformas devem ser mostradas.
     */
    private boolean showPlats;

    /**
     * Construtor.
     * 
     * @param columnTitle ttulo da coluna.
     * @param showPlats determina se as plataformas dos servidores devem ser
     *        mostradas.
     */
    private SGASetTableProvider(String columnTitle, boolean showPlats) {
      this.columnTitle = columnTitle;
      this.showPlats = showPlats;
    }

    /**
     * Construtor.
     * 
     * @param columnTitle ttulo da coluna.
     */
    private SGASetTableProvider(String columnTitle) {
      this(columnTitle, false);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getColumnNames() {
      if (showPlats) {
        return new String[] { columnTitle,
            LNG.get("MON_GROUP_SERVERS_PLATFORM") };
      }
      return new String[] { columnTitle };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getColumnClasses() {
      if (showPlats) {
        return new Class[] { String.class, String.class };
      }
      return new Class[] { String.class };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object[] getCellValues(Object arg0) {
      SGASet sgaSet = (SGASet) arg0;
      if (showPlats) {
        return new String[] { sgaSet.getName(), sgaSet.getPlatformId() };
      }
      return new String[] { sgaSet.getName() };
    }

  }
}
