/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Frame;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.gui.StatusBar;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.events.FinishedOrSystemFailureCommandsLoadingEvent;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.commandsmonitor.actions.ExecuteCommandAction;
import csbase.client.applications.commandsmonitor.actions.KillCommandsAction;
import csbase.client.applications.commandsmonitor.actions.RefreshCommandsAction;
import csbase.client.applications.commandsmonitor.actions.RemoveCommandsAction;
import csbase.client.applications.commandsmonitor.actions.SchedulerSwitchAction;
import csbase.client.applications.commandsmonitor.actions.SchedulerSwitchAction.StateChangedEvent;
import csbase.client.applications.commandsmonitor.actions.ShowCommandDetailsAction;
import csbase.client.applications.commandsmonitor.actions.ShowCommandDirAction;
import csbase.client.applications.commandsmonitor.actions.ShowCommandLogAction;
import csbase.client.applications.commandsmonitor.actions.ShowCommandParametersAction;
import csbase.client.applications.commandsmonitor.dal.xml.XmlConfigDAO;
import csbase.client.applications.commandsmonitor.events.DoubleClickEvent;
import csbase.client.applications.commandsmonitor.events.SelectionChangedEvent;
import csbase.client.applications.commandsmonitor.models.CommandsFilterDTO;
import csbase.client.applications.commandsmonitor.models.CommandsTableDTO;
import csbase.client.applications.commandsmonitor.models.TablesTabDTO;
import csbase.client.applications.commandsmonitor.table.CommandsTableFactory;
import csbase.client.desktop.Task;
import csbase.client.util.event.EventListener;
import csbase.client.util.event.EventManager;
import csbase.client.util.event.IEvent;
import csbase.logic.CommandStatus;
import csbase.logic.RemoteFileInputStream;
import csbase.remote.ApplicationServiceInterface;
import csbase.remote.ClientRemoteLocator;

/**
 * Aplicao de monitorao de comandos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class CommandsMonitor extends Application {

  /** Gerencia o envio de eventos. */
  private EventManager eventManager;

  /** Aes de comando presentes na interface. */
  /** Mostra os parmetros do comando selecionado. */
  private ShowCommandParametersAction showParametersAction;
  /** Mostra o log do comando selecionado. */
  private ShowCommandLogAction showLogAction;
  /** Mostra os detalhes do comando selecionado. */
  private ShowCommandDetailsAction showDetailsAction;
  /** Cria uma cpia do comando selecionado, em memria, para execuo. */
  private ExecuteCommandAction executeCommandAction;
  /** Mata o comando selecionado em execuo. */
  private KillCommandsAction killCommandsAction;
  /** Destroi a persistncia do comando selecionado, terminado. */
  private RemoveCommandsAction removeCommandsAction;

  /**
   * Construtor.
   * 
   * @param id identificador nico da aplicao.
   * 
   * @throws ApplicationException caso no seja possvel criar dinamicamente a
   *         interface.
   */
  public CommandsMonitor(String id) throws ApplicationException {
    super(id);

    this.eventManager = new EventManager();

    try {
      /*
       * Cria as aes de comandos. Deve ser criado antes pois  utilizado pelo
       * painel principal e pela barra de menu.
       */
      createActions();
      /*
       * Adiciona a ao que dever ocorrer quando um comando for clicado duas
       * vezes.
       */
      eventManager.addEventListener(showParametersAction,
        DoubleClickEvent.class);

      // Cria o objeto de acesso as configuraes.
      XmlConfigDAO config = createConfigDAO();

      // Cria o painel da aplicao.
      getApplicationFrame().getContentPane().add(createContentPane(config));
      // Cria a barra de menu da aplicao.
      getApplicationFrame().setJMenuBar(createMenuBar(config));
      getApplicationFrame().pack();
    }
    catch (ApplicationException e) {
      throw e;
    }
    catch (Exception e) {
      throw new ApplicationException(e);
    }
  }

  /**
   * Adiciona um ouvinte interessado nas alteraes na seleo dos comandos.
   * 
   * @param listener ouvinte interessado.
   */
  public void addSelectionChangedListener(
    EventListener<SelectionChangedEvent> listener) {
    eventManager.addEventListener(listener, SelectionChangedEvent.class);
  }

  /**
   * Inicializa o monitor de comando de cada tabela presente.
   * 
   * {@inheritDoc}
   */
  @Override
  public void startApplication() throws ApplicationException {
    eventManager.fireEvent(new ActionEvent(ActionEvent.Type.started));
    super.startApplication();
  }

  /**
   * Interrompe a monitorao de comandos.
   * 
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    eventManager.fireEvent(new ActionEvent(ActionEvent.Type.killed));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Cria as aes de comandos.
   */
  private void createActions() {
    showParametersAction = new ShowCommandParametersAction(this);
    showLogAction = new ShowCommandLogAction(this);
    showDetailsAction = new ShowCommandDetailsAction(this);
    executeCommandAction = new ExecuteCommandAction(this);
    killCommandsAction = new KillCommandsAction(this);
    removeCommandsAction = new RemoveCommandsAction(this);
  }

  /**
   * Cria a barra de menu.
   * 
   * @param config As configuraes da aplicao.
   * 
   * @return A barra de menu.
   */
  private JMenuBar createMenuBar(XmlConfigDAO config) {
    JMenuBar menuBar = new JMenuBar();
    menuBar.add(createCommandsMenu());
    /* Verifica se o menu do escalonador deve ser exibido. */
    if (acceptCommandsFromStatus(CommandStatus.SCHEDULED, config)) {
      menuBar.add(createSchedulerMenu());
    }
    return menuBar;
  }

  /**
   * Cria o menu de administrao do agendador de comandos.
   * 
   * @return O menu de administrao do agendador de comandos.
   */
  private JMenu createSchedulerMenu() {
    JMenu schedulerMenu =
      new JMenu(getString("CommandsMonitor.menu.scheduler"));
    /** Item de menu de auto-reload. */
    SchedulerSwitchAction switchAction = new SchedulerSwitchAction(this);
    final JCheckBoxMenuItem switchItem = new JCheckBoxMenuItem(switchAction);
    boolean isBlocked = switchAction.isSchedulerBlocked();
    switchItem.setSelected(isBlocked);
    showBloquedQueueWarning(isBlocked);
    switchAction
      .addStateChangedListener(new EventListener<StateChangedEvent>() {
        @Override
        public void eventFired(StateChangedEvent event) {
          boolean isBlocked = event.isSchedulerBlocked();
          showBloquedQueueWarning(isBlocked);
          switchItem.setSelected(isBlocked);
        }
      });
    schedulerMenu.add(switchItem);
    return schedulerMenu;
  }

  /**
   * Exibe/esconde alerta sobre fila bloqueada.
   * 
   * @param show <code>true</code> se o alerta deve ser exibido,
   *        <code>false</code> para remov-lo
   */
  private void showBloquedQueueWarning(boolean show) {
    StatusBar sb = getApplicationFrame().getStatusBar();
    if (show) {
      sb.setWarning(getString("CommandsMonitor.warning.blockedQueue"));
    }
    else {
      sb.clearStatus();
    }
  }

  /**
   * Cria o menu Comandos.
   * 
   * @return O menu.
   */
  private JMenu createCommandsMenu() {
    JMenu commandsMenu = new JMenu(getString("CommandsMonitor.menu.commands"));
    commandsMenu.add(new JMenuItem(showParametersAction));
    commandsMenu.add(new JMenuItem(showLogAction));
    commandsMenu.add(new JMenuItem(showDetailsAction));
    commandsMenu.add(new JMenuItem(executeCommandAction));
    commandsMenu.add(new JMenuItem(killCommandsAction));
    commandsMenu.add(new JMenuItem(removeCommandsAction));

    if (hasAdvancedMenu()) {
      commandsMenu.addSeparator();
      JMenu advancedSubMenu = createAdvancedMenu();
      commandsMenu.add(advancedSubMenu);
    }

    commandsMenu.addSeparator();
    commandsMenu.add(new ApplicationExitAction(this));

    return commandsMenu;
  }

  /**
   * Testa se o menu avanado est habilitado na configurao do sistema.
   * 
   * @return verdadeiro se o menu deve ser mostrado ou falso, caso contrrio.
   */
  private boolean hasAdvancedMenu() {
    final ConfigurationManager cnfManager = ConfigurationManager.getInstance();
    if (cnfManager == null) {
      return false;
    }
    try {
      final Class<CommandsMonitor> propClass = CommandsMonitor.class;
      final Configuration cnf = cnfManager.getConfiguration(propClass);
      final String propName = "enable.advanced.menu";
      return cnf.getOptionalBooleanProperty(propName, false);
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Cria o menu Avanado.
   * 
   * @return O menu.
   */
  private JMenu createAdvancedMenu() {
    JMenu advancedSubMenu =
      new JMenu(getString("CommandsMonitor.menu.advanced"));
    advancedSubMenu.add(new JMenuItem(new RefreshCommandsAction(this)));
    advancedSubMenu.add(new JMenuItem(new ShowCommandDirAction(this)));
    return advancedSubMenu;
  }

  /**
   * Cria o painel da aplicao.
   * 
   * @param config As configuraes da aplicao.
   * 
   * @return O painel da aplicao.
   * 
   * @throws ApplicationException em caso de falha na aplicao.
   */
  private JComponent createContentPane(XmlConfigDAO config)
    throws ApplicationException {
    JPanel panel = new JPanel(new BorderLayout());
    panel.add(createToolBar(), BorderLayout.NORTH);
    panel.add(createCenterPane(config), BorderLayout.CENTER);

    // Mostrando a barra de status.
    this.getApplicationFrame().showStatusBar();
    if (acceptCommandsFromStatus(CommandStatus.FINISHED, config)
      || acceptCommandsFromStatus(CommandStatus.SYSTEM_FAILURE, config)) {
      // Criando o listener que ir escutar a cache e atualizar a barra de status.
      final EventListener<FinishedOrSystemFailureCommandsLoadingEvent> loadingListener =
        new EventListener<FinishedOrSystemFailureCommandsLoadingEvent>() {
          @Override
          public void eventFired(
            final FinishedOrSystemFailureCommandsLoadingEvent event) {
            SwingThreadDispatcher.invokeLater(new Runnable() {
              @Override
              public void run() {
                StatusBar statusbar = getApplicationFrame().getStatusBar();
                if (event.finished()) {
                  statusbar.clearStatus();
                }
                else {
                  /*
                   * A ordem dos parmetros no deve ser alterada aqui, com o
                   * risco de enviar uma mensagem errada ao usurio; ao invs
                   * disso, altere a mensagem. A ordem est documentada na
                   * descrio da mensagem.
                   */
                  Object[] args =
                    new Object[] { event.getLoaded(), event.getMissing(),
                        event.getTotal() - event.getMissing(), event.getTotal() };

                  statusbar.setStatus(getString(
                    "CommandsMonitor.statusbar.commands.loading", args));
                }
              }
            });
          }
        };
      // Adicionando e removendo o listener na cache, conforme o estado da aplicao.
      eventManager.addEventListener(new EventListener<ActionEvent>() {
        @Override
        public void eventFired(ActionEvent event) {
          if (event.type == ActionEvent.Type.started) {
            CommandsCache.getInstance().addEventListener(loadingListener);
          }
          else if (event.type == ActionEvent.Type.killed) {
            CommandsCache.getInstance().removeEventListener(loadingListener);
          }
        }
      }, ActionEvent.class);
    }

    return panel;
  }

  /**
   * Cria a barra de ferramentas contendo algumas aes de comando.
   * 
   * @return A barra de ferramentas contendo algumas aes de comando.
   */
  private JToolBar createToolBar() {
    JToolBar toolBar = new JToolBar();
    toolBar.setOrientation(SwingConstants.HORIZONTAL);
    toolBar.add(showParametersAction);
    toolBar.add(showLogAction);
    toolBar.add(showDetailsAction);
    toolBar.add(executeCommandAction);
    toolBar.add(killCommandsAction);
    toolBar.add(removeCommandsAction);
    return toolBar;
  }

  /**
   * Cria o painel que ir conter as abas que detm as tabelas.
   * 
   * @param config As configuraes da aplicao.
   * 
   * @return O painel que ir conter as abas que detm as tabelas.
   * 
   * @throws ApplicationException em caso de falha na aplicao.
   */
  private Component createCenterPane(XmlConfigDAO config)
    throws ApplicationException {
    /** Fbrica de tabelas. */
    CommandsTableFactory tablesFactory =
      new CommandsTableFactory(getApplicationRegistry(), config);

    List<TablesTabDTO> tabsInfo = config.getTabsInfo();
    if (tabsInfo.size() < 1) {
      throw new ApplicationException(
        "Deve existir pelo menos uma aba configurada.");
    }
    else if (tabsInfo.size() == 1) {
      TablesTabDTO tab = tabsInfo.get(0);
      final TablesPanel panel =
        new TablesPanel(tab.getTablesId(), tablesFactory,
          getApplicationRegistry(), eventManager);
      eventManager.addEventListener(new EventListener<ActionEvent>() {
        @Override
        public void eventFired(ActionEvent event) {
          if (event.type == ActionEvent.Type.started) {
            panel.start();
          }
          else if (event.type == ActionEvent.Type.killed) {
            panel.stop();
          }
        }
      }, ActionEvent.class);
      return panel;
    }
    else {
      final TablesTabbedPane panel =
        new TablesTabbedPane(config, tablesFactory, getApplicationRegistry(),
          eventManager);
      eventManager.addEventListener(new EventListener<ActionEvent>() {
        @Override
        public void eventFired(ActionEvent event) {
          if (event.type == ActionEvent.Type.started) {
            panel.start();
          }
          else if (event.type == ActionEvent.Type.killed) {
            panel.stop();
          }
        }
      }, ActionEvent.class);
      return panel;
    }
  }

  /**
   * Cria o objeto de acesso as configuraes da interface.
   * 
   * @return O objeto de acesso as configuraes da interface.
   * 
   * @throws ApplicationException Caso tenha ocorrido algum problema tentando
   *         ler as configuraes.
   */
  private XmlConfigDAO createConfigDAO() throws ApplicationException {
    final ApplicationServiceInterface appService =
      ClientRemoteLocator.applicationService;

    Task<XmlConfigDAO> task = new Task<XmlConfigDAO>() {
      @Override
      protected void performTask() throws Exception {
        /*
         * Obtm um {@link InputStream} para o xml de configurao que est no
         * servidor.
         */
        final String[] configXmlPath = new String[] { "config.xml" };
        RemoteFileChannelInfo info =
          appService.getApplicationResource(getId(), configXmlPath);
        if (info == null) {
          throw new RuntimeException("Arquivo no encontrado: config.xml");
        }
        InputStream in = new RemoteFileInputStream(info);
        // L as configuraes da interface.
        Reader reader = new InputStreamReader(in);
        // Cria o objeto de acesso aos dados da configurao.
        setResult(new XmlConfigDAO(reader));
      }
    };
    Frame frame = getApplicationFrame();
    if (task.execute(frame, getName(),
      getString("task.configdao.create.message"))) {
      return task.getResult();
    }

    final String msg = getString("task.configdao.create.error");
    throw new ApplicationException(msg, task.getError());
  }

  /**
   * Verifica se o filtro de qualquer uma das tabelas a serem mostradas aceitam
   * comandos no dado estado.
   * 
   * @param status Estado que deve ser aceito pelo filtro de qualquer tabela do
   *        visualizador.
   * @param config As configuraes da aplicao.
   * 
   * @return <tt>true</tt> se o filtro de qualquer uma das tabelas a serem
   *         mostradas aceitam comandos no dado estado.
   */
  public boolean acceptCommandsFromStatus(CommandStatus status,
    XmlConfigDAO config) {
    // Cria um cache de tabelas para facilitar no passo abaixo.
    Map<String, CommandsTableDTO> tablesById =
      new HashMap<String, CommandsTableDTO>();
    for (CommandsTableDTO tableInfo : config.getTablesInfo()) {
      tablesById.put(tableInfo.getId(), tableInfo);
    }
    // Cria um cache de filtros para facilitar no passo abaixo.
    Map<String, CommandsFilterDTO> filtersById =
      new HashMap<String, CommandsFilterDTO>();
    for (CommandsFilterDTO filterInfo : config.getFiltersInfo()) {
      filtersById.put(filterInfo.getId(), filterInfo);
    }

    // Procura pelo estado no filtro de cada tabela em cada aba.
    for (TablesTabDTO tabInfo : config.getTabsInfo()) {
      for (String tableId : tabInfo.getTablesId()) {
        CommandsTableDTO tableInfo = tablesById.get(tableId);
        CommandsFilterDTO filterInfo = filtersById.get(tableInfo.getFilterId());
        // Verifica se o filtro inclui ou exclui os estados que ele carrega.
        if (filterInfo.areStatusesIncluded()) {
          // Se inclui, basta procurar pelo estado que queremos no filtro.
          for (CommandStatus aStatus : filterInfo.getStatuses()) {
            if (aStatus.equals(status)) {
              return true;
            }
          }
        }
        else {
          // Se exclui, devemos verificar se o estado no est na lista.
          boolean excluded = false;
          for (CommandStatus aStatus : filterInfo.getStatuses()) {
            if (aStatus.equals(status)) {
              excluded = true;
              break;
            }
          }
          /*
           * Se ele no  excluido do filtro, ento o filtro aceita comandos com
           * aquele estado.
           */
          if (!excluded) {
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   * Evento indicando que a aplicao sofreu uma ao.
   * 
   * @author Tecgraf
   */
  static class ActionEvent implements IEvent {
    /**
     * Tipo de aes que pode ter ocorrido.
     * 
     * @author Tecgraf
     */
    enum Type {
      /**
       * A aplicao foi iniciada.
       */
      started,
      /**
       * A aplicao foi morta.
       */
      killed
    }

    /**
     * Tipo de ao sofrida pela aplicao.
     */
    private Type type;

    /**
     * Construtor.
     * 
     * @param type Tipo de ao sofrida pela aplicao.
     */
    ActionEvent(Type type) {
      this.type = type;
    }
  }
}
