/*
 * $Id:$
 */
package csbase.client.ias.server;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.WindowConstants;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.manager.server.ServerInfoManager;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ServerURI;
import csbase.logic.server.ServerInfo;
import csbase.logic.server.ServerInfoData;

/**
 * Representa um dilogo utilizado para criao e modificao de informaes de
 * servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ServerInfoDialog {

  /**
   * A janela pai deste dilogo.
   */
  private JFrame parent;

  /**
   * Informaes do servidor que est sendo alterado ou null, caso seja uma
   * incluso.
   */
  private ServerInfo serverInfo;

  /**
   * O dilogo propriamente dito.
   */
  private DesktopComponentDialog dialog;

  /**
   * O ttulo do dilogo.
   */
  private String windowLabel;

  /**
   * O boto de ao do dilogo. Este boto ser de incluso ou de alterao.
   */
  private JButton actionButton;

  /**
   * Campo de texto que contm o nome do servidor
   */
  private JTextField serverNameTextField;

  /**
   * Campo de seleo que indica se o servidor est suspenso.
   */
  private JCheckBox serverSuspendedCheckBox;

  /**
   * Quantidade de colunas do campo texto do nome do servidor e URI
   */
  private static final int SERVER_TEXTFIELD_COLUMNS = 25;

  /**
   * Campo que indica se o servidor  local
   */
  private JCheckBox serverIsLocalCheckBox;

  /**
   * Campo de texto para a URI do servidor
   */
  private JTextField serverURITextField;

  /**
   * Cria um dilogo para incluso de informaes um servidor
   * 
   * @param parent A janela pai do dilogo.
   */
  public ServerInfoDialog(JFrame parent) {
    this.parent = parent;
    this.windowLabel = LNG.get("IAS_SERVER_ADD_TITLE");
    this.display();
  }

  /**
   * Cria um dilogo para alterao de informaes de um servidor.
   * 
   * @param parent A janela pai do dilogo.
   * @param serverInfo O servidor a ser alterado.
   * 
   * @throws IllegalArgumentException Caso o servidor local seja nulo.
   */
  public ServerInfoDialog(JFrame parent, ServerInfo serverInfo) {
    if (serverInfo == null) {
      throw new IllegalArgumentException("O servidor no pode ser nulo.");
    }
    this.parent = parent;
    this.windowLabel = LNG.get("IAS_SERVER_UPDATE_TITLE");
    this.serverInfo = serverInfo;
    this.display();
  }

  /**
   * Cria os componentes do dilogo e o exibe.
   */
  private void display() {
    JPanel mainPanel = new JPanel(new BorderLayout());
    this.addContentPanel(mainPanel);
    this.addButtonPanel(mainPanel);
    this.dialog = new DesktopComponentDialog(this.parent, this.windowLabel);
    this.dialog.getContentPane().add(mainPanel);
    this.dialog.getRootPane().setDefaultButton(this.actionButton);
    this.dialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
    this.dialog.pack();
    this.dialog.center(this.parent);
    this.dialog.setVisible(true);
  }

  /**
   * Listener de informativos.
   * 
   * @author Tecgraf
   */
  private class ServerInfoDocumentListener implements DocumentListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void changedUpdate(DocumentEvent e) {
      this.update(e);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void insertUpdate(DocumentEvent e) {
      this.update(e);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeUpdate(DocumentEvent e) {
      this.update(e);
    }

    /**
     * @param e evento
     */
    private void update(DocumentEvent e) {
      Document serverNameDoc = serverNameTextField.getDocument();
      Document serverURIDoc = serverURITextField.getDocument();
      String serverName = "";
      String serverURIStr = "";

      try {
        serverName = serverNameDoc.getText(0, serverNameDoc.getLength()).trim();
        serverURIStr = serverURIDoc.getText(0, serverURIDoc.getLength()).trim();
      }
      catch (BadLocationException ble) {
        StandardErrorDialogs.showErrorDialog(dialog,
          LNG.get("IAS_SERVER_GET_TEXT_ERROR_TITLE"),
          LNG.get("IAS_SERVER_GET_TEXT_ERROR_MESSAGE"), ble);
      }
      if (serverName.length() == 0 || serverURIStr.length() == 0) {
        actionButton.setEnabled(false);
      }
      else {
        actionButton.setEnabled(true);
      }
    }
  }

  /**
   * Adiciona o painel com as informaes do servidor ao painel principal.
   * 
   * @param mainPanel O painel principal.
   */
  private void addContentPanel(JPanel mainPanel) {
    this.serverNameTextField = new JTextField(SERVER_TEXTFIELD_COLUMNS);
    this.serverURITextField = new JTextField(SERVER_TEXTFIELD_COLUMNS);
    if (this.serverInfo != null) {
      this.serverNameTextField.setText(this.serverInfo.getName());
      this.serverURITextField.setText(this.serverInfo.getURI().toString());
    }

    ServerInfoDocumentListener serverInfoDocListener =
      new ServerInfoDocumentListener();
    this.serverNameTextField.getDocument().addDocumentListener(
      serverInfoDocListener);
    this.serverURITextField.getDocument().addDocumentListener(
      serverInfoDocListener);
    this.serverSuspendedCheckBox = new JCheckBox();
    this.serverIsLocalCheckBox = new JCheckBox();
    if (this.serverInfo != null) {
      this.serverSuspendedCheckBox.setSelected(this.serverInfo.isSuspended());
      this.serverIsLocalCheckBox.setSelected(this.serverInfo.isLocal());
    }
    JPanel contentPanel =
      GUIUtils
        .createBasicGridPanel(new JComponent[][] {
            { new JLabel(LNG.get("IAS_SERVER_NAME")), this.serverNameTextField },
            { new JLabel(LNG.get("IAS_SERVER_URI")), this.serverURITextField },
            { new JLabel(LNG.get("IAS_SERVER_SUSPENDED")),
                this.serverSuspendedCheckBox },
            { new JLabel(LNG.get("IAS_SERVER_LOCAL")),
                this.serverIsLocalCheckBox } });
    contentPanel.setBorder(BorderFactory.createTitledBorder(LNG
      .get("IAS_SERVER_REGISTRATION")));
    mainPanel.add(contentPanel, BorderLayout.CENTER);
  }

  /**
   * Adiciona o painel de botes ao painel principal.
   * 
   * @param mainPanel O painel principal.
   */
  private void addButtonPanel(JPanel mainPanel) {
    JPanel buttonPanel = new JPanel();
    if (this.serverInfo == null) {
      this.actionButton = new JButton(LNG.get("IAS_ADD"));
      this.actionButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (!ServerURI.isValid(serverURITextField.getText())) {
            StandardDialogs.showErrorDialog(ServerInfoDialog.this.parent,
              LNG.get("IAS_SERVER_UPDATE_TITLE"), LNG.get("UTIL_INVALID_URI"));
            return;
          }

          try {
            if (addServerInfo()) {
              dialog.dispose();
            }
            else {
              StandardErrorDialogs.showErrorDialog(dialog, LNG
                .get("IAS_SERVER_ADD_TITLE"), MessageFormat.format(
                LNG.get("IAS_SERVER_ALREADY_EXISTS_MESSAGE"),
                new Object[] { serverNameTextField.getText() }));
            }
          }
          catch (Exception exception) {
            StandardErrorDialogs.showErrorDialog(dialog, LNG
              .get("IAS_SERVER_ADD_TITLE"), MessageFormat.format(
              LNG.get("IAS_SERVER_ADDING_ERROR"),
              new Object[] { serverNameTextField.getText() }), exception);
          }
        }
      });
    }
    else {
      this.actionButton = new JButton(LNG.get("IAS_UPDATE"));
      this.actionButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          if (!ServerURI.isValid(serverURITextField.getText())) {
            StandardDialogs.showErrorDialog(ServerInfoDialog.this.parent,
              LNG.get("IAS_SERVER_UPDATE_TITLE"), LNG.get("UTIL_INVALID_URI"));
            return;
          }
          try {
            if (updateServerInfo()) {
              dialog.dispose();
            }
            else {
              StandardErrorDialogs.showErrorDialog(dialog, LNG
                .get("IAS_SERVER_UPDATE_TITLE"), MessageFormat.format(
                LNG.get("IAS_SERVER_ALREADY_EXISTS_MESSAGE"),
                new Object[] { serverNameTextField.getText() }));
            }
          }
          catch (Exception exception) {
            StandardErrorDialogs.showErrorDialog(dialog, LNG
              .get("IAS_SERVER_UPDATE_TITLE"), MessageFormat.format(
              LNG.get("IAS_SERVER_UPDATING_ERROR"),
              new Object[] { serverInfo.getName() }), exception);
          }
        }
      });
    }
    if (this.serverInfo == null) {
      this.actionButton.setEnabled(false);
    }
    else {
      this.actionButton.setEnabled(true);
    }
    buttonPanel.add(this.actionButton);
    JButton cancelButton = new JButton(LNG.get("IAS_CANCEL"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        dialog.dispose();
      }
    });
    buttonPanel.add(cancelButton);
    mainPanel.add(buttonPanel, BorderLayout.SOUTH);
  }

  /**
   * Altera as informaes do servidor no gerenciador de servidores.
   * 
   * @return true, caso o servidor tenha sido alterado, ou false, caso j exista
   *         um servidor com o nome informado.
   * 
   * @throws Exception Caso ocorra algum erro durante a operao.
   */
  private boolean updateServerInfo() throws Exception {
    RemoteTask<Boolean> modifyLocalServerTask = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        this
          .setResult(new Boolean(
            ServerInfoManager.getInstance()
              .modifyServerInfo(
                serverInfo,
                new ServerInfoData(serverNameTextField.getText().trim(),
                  serverSuspendedCheckBox.isSelected(), serverIsLocalCheckBox
                    .isSelected(),
                  ServerURI.parse(serverURITextField.getText())))));
      }
    };
    if (modifyLocalServerTask.execute(this.dialog,
      LNG.get("IAS_SERVER_UPDATE_TITLE"),
      LNG.get("IAS_SERVER_UPDATING_MESSAGE"))) {
      return modifyLocalServerTask.getResult();
    }
    throw modifyLocalServerTask.getError();
  }

  /**
   * Adiciona um servidor ao gerenciador de servidores.
   * 
   * @return true, caso o servidor tenha sido inserido, ou false, caso j exista
   *         um servidor com o nome informado.
   * 
   * @throws Exception Caso ocorra algum erro durante a operao.
   */
  private boolean addServerInfo() throws Exception {
    final ServerInfoData newLocalServer =
      new ServerInfoData(this.serverNameTextField.getText().trim(),
        this.serverSuspendedCheckBox.isSelected(),
        serverIsLocalCheckBox.isSelected(),
        ServerURI.parse(this.serverURITextField.getText()));
    RemoteTask<Boolean> addLocalServerTask = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        this.setResult(new Boolean(ServerInfoManager.getInstance()
          .addServerInfo(newLocalServer)));
      }
    };
    if (addLocalServerTask.execute(this.dialog,
      LNG.get("IAS_SERVER_ADD_TITLE"), MessageFormat.format(
        LNG.get("IAS_SERVER_ADDING_MESSAGE"),
        new Object[] { newLocalServer.getName() }))) {
      Boolean result = addLocalServerTask.getResult();
      return result.booleanValue();
    }
    throw addLocalServerTask.getError();
  }
}
