package csbase.client.ias;

import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import csbase.client.desktop.RemoteTask;
import csbase.client.ias.PermissionInfoDialog.PermissionClassName;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ChoicePermission;
import csbase.logic.Permission;

/**
 * Painel para definio de valores de permisses que estendem ChoicePermission.
 * 
 * @see ChoicePermission
 */
public class ChoicePermissionPanel extends AbstractPermissionPanel {

  /** Objeto para percorrer os atributos de uma ChoicePermission. */
  AttributesNavigator navigator;

  /** Combo para seleo do nome do atributo de uma ChoicePermission. */
  private JLabel attributeName;

  /** Chave que mapeia o texto que ser exibido em {@link #attributeName}. */
  private String attributeNameKey;

  /**
   * Combo para seleo do valor para um atributo de uma ChoicePermission, caso
   * este seja exclusivo.
   */
  private JComboBox attributeValue;

  /** Valores possveis de um atributo. */
  private Map<String, Object> attributeRange;

  /** Map com atributos para criao/alterao de uma ChoicePermission. */
  private Map<String, ArrayList<Object>> map;

  /** Container para seleo de mltiplos valores de uma ChoicePermission. */
  private ContainerSelection<String> containerSelection;

  /** Ttulo do dilogo. */
  String windowLabel;

  /** Boto para navegar pelos atributos de uma ChoicePermission. */
  private JButton nextButton;

  /** Boto para navegar pelos atributos de uma ChoicePermission. */
  private JButton previousButton;

  /** Boto de incluso/alterao da permisso. */
  private JButton includePermissionButton;

  /** Indica se uma nova permisso est sendo criada. */
  private boolean isNew = true;

  /**
   * Construtor.
   * 
   * @param dialog Dilogo no qual este painel est contido.
   * @param windowLabel Ttulo do dilogo.
   */
  ChoicePermissionPanel(PermissionInfoDialog dialog, String windowLabel) {
    super(new GridBagLayout(), dialog);
    this.windowLabel = windowLabel;
    map = new HashMap<String, ArrayList<Object>>();

    nextButton = dialog.nextButton = new JButton(LNG.get("IAS_NEXT"));
    previousButton =
      dialog.previousButton = new JButton(LNG.get("IAS_PREVIOUS"));
    includePermissionButton = dialog.includePermissionButton;

    navigator = new AttributesNavigator();
    dialog.nextButton.setVisible(false);
    dialog.nextButton.setEnabled(true);
    dialog.previousButton.setVisible(false);
    dialog.previousButton.setEnabled(true);
    dialog.nextButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        navigator.next();
      }
    });
    dialog.previousButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        navigator.previous();
      }
    });
    attributeName = new JLabel();
    attributeValue = new JComboBox();
    attributeValue.setSelectedIndex(-1);
    attributeValue.setVisible(false);
    attributeValue.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent a) {
        if (attributeValue.getSelectedIndex() != -1) {
          ArrayList<Object> value = new ArrayList<Object>();
          value.add(attributeRange.get(attributeValue.getSelectedItem()));
          map.put(attributeNameKey, value);
          nextButton.setEnabled(true);
          includePermissionButton.setEnabled(!nextButton.isVisible());
        }
        else {
          nextButton.setEnabled(false);
          includePermissionButton.setEnabled(false);
        }
        if (!isNew) {
          ChoicePermissionPanel.this.dialog.checkDataChange();
        }
      }
    });
    this.containerSelection =
      new ContainerSelection<String>(new TableItemsFormatter(),
        new SelectedItemsFormatter(), true, false);
    this.containerSelection.adjustTableColumns();
    this.containerSelection
      .addContainerSelectionListener(new ContainerSelectionListener() {
        @Override
        public void containerChanged() {
          List<String> selectedItems = containerSelection.getSelectedItems();
          if (selectedItems.size() > 0) {
            ArrayList<Object> values = new ArrayList<Object>();
            for (int i = 0; i < selectedItems.size(); i++) {
              values.add(attributeRange.get(selectedItems.get(i)));
            }
            map.put(attributeNameKey, values);
            nextButton.setEnabled(true);
            includePermissionButton.setEnabled(!nextButton.isVisible());
          }
          else {
            nextButton.setEnabled(false);
            includePermissionButton.setEnabled(false);
          }
          if (!isNew) {
            ChoicePermissionPanel.this.dialog.checkDataChange();
          }
        }
      });
    this.containerSelection.getPanel();

    // Label com o nome do atributo
    GBC gbc = new GBC(0, 0).width(2).northwest().both();
    add(this.createChoicePermissionPanel(), gbc);

  }

  /**
   * @return Painel para escolha do(s) valor(es).
   */
  private JPanel createChoicePermissionPanel() {
    JPanel choicePermissionPanel = new JPanel(new GridBagLayout());

    // Label com o nome do atributo
    GBC gbc = new GBC(0, 0).insets(5, 5, 5, 5);
    choicePermissionPanel.add(attributeName, gbc);

    // Combo com os valores do atributo
    gbc = new GBC(1, 0).horizontal().insets(5, 5, 5, 5);
    choicePermissionPanel.add(attributeValue, gbc);

    // ContainerSelection com os valores do atributo
    gbc = new GBC(0, 0).both().insets(5, 5, 5, 5);
    choicePermissionPanel.add(this.containerSelection.getPanel(), gbc);

    return choicePermissionPanel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void importFrom(Permission permission) {
    isNew = false;
    map.putAll((Map<String, ArrayList<Object>>) ((ChoicePermission) permission)
      .getAttributes());
    navigator.init();

  }

  /**
   * {@inheritDoc}
   */
  @Override
  void exportTo(Permission permission) {
    ((ChoicePermission) permission).setAttributes(map);
  }

  /**
   * Formatador dos itens da tabela ordenvel, que ser exibida no
   * ContainerSelection de edio de uma ChoicePermission.
   * 
   * @author Jorge Marques
   */
  class TableItemsFormatter extends DefaultObjectTableProvider {
    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getColumnNames() {
      return new String[] { LNG.get("IAS_AVAILABLE_ITEMS") };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getColumnClasses() {
      return new Class<?>[] { String.class };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object[] getCellValues(Object arg0) {
      return new Object[] { arg0.toString() };
    }
  }

  /**
   * Formatador dos itens da tabela ordenvel, que ser exibida no
   * ContainerSelection de edio de uma ChoicePermission.
   * 
   * @author Jorge Marques
   */
  class SelectedItemsFormatter extends DefaultObjectTableProvider {
    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getColumnNames() {
      return new String[] { LNG.get("IAS_SELECTED_ITEMS") };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getColumnClasses() {
      return new Class<?>[] { String.class };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object[] getCellValues(Object arg0) {
      return new Object[] { arg0.toString() };
    }
  }

  /**
   * Classe que faz a navegao entre os atributos de uma ChoicePermission.
   */
  private class AttributesNavigator {
    /**
     * Lista com os pares {String attributeName, Boolean isMultiSelection} de
     * domnio de uma permisso, onde <code>isMultiSelection</code> indica se 
     * permitido a seleo de mais de um valor deste atributo.
     */
    ArrayList<Entry<String, Object>> domainList;

    /** ndice do atributo que est sendo exibido. */
    int position;

    /** Permisso cujos atributos so navegados por uma instncia desta classe. */
    ChoicePermission choicePermission;

    /**
     * Map com os pares {String attributeName, Map<String, Object>
     * attributeRange} dos atributos de uma permisso, onde
     * <code>attributeRange</code> indica os valores possveis de um atributo.
     */
    Map<String, Map<String, Object>> domainAttributesRangeMap =
      new HashMap<String, Map<String, Object>>();

    /**
     * Inicia o navegador de atributos de uma ChoicePermission.
     */
    public void init() {
      Window window = ChoicePermissionPanel.this.dialog;
      String windowLabel = ChoicePermissionPanel.this.windowLabel;
      PermissionClassName item =
        ChoicePermissionPanel.this.dialog.getPermissionClassName();

      /*
       * Cria uma instncia da permisso selecionada na comboBox do dilogo que
       * contm este painel.
       */
      try {
        Class<?> selectedClass = Class.forName(item.getClassName());
        Constructor<?> constructor =
          selectedClass.getConstructor(new Class[] {});
        choicePermission =
          (ChoicePermission) constructor.newInstance(new Object[] {});
      }
      catch (Exception e) {
        StandardErrorDialogs.showErrorDialog(window, LNG.get("ERRO") + " - "
          + windowLabel, e);
      }

      /*
       * Obtm a lista de atributos desta permisso e a lista de valores
       * possveis para cada atributo.
       */
      RemoteTask<Void> task = new RemoteTask<Void>() {
        @Override
        public void performTask() throws Exception {
          Map<String, Object> domainMap = choicePermission.getDomain();

          domainList =
            new ArrayList<Entry<String, Object>>(domainMap.entrySet());
          for (int i = 0; i < domainList.size(); i++) {
            String attrName = domainList.get(i).getKey();
            domainAttributesRangeMap.put(attrName,
              choicePermission.getRange(attrName));
          }
        }
      };
      if (!task.execute(window, LNG.get("desktop.msg.wait"),
        LNG.get("desktop.msg.wait"))) {
        domainList = new ArrayList<Entry<String, Object>>();
        attributeName.setVisible(false);
        attributeValue.setVisible(false);
        containerSelection.getPanel().setVisible(false);
      }
      position = -1;

      /* Navega para o primeiro atributo desta permisso. */
      next();
    }

    /**
     * Navega para o atributo anterior.
     */
    void previous() {
      if (position > 0) {
        position--;
        String key = domainList.get(position).getKey();
        attributeRange = domainAttributesRangeMap.get(key);
        go(key);
      }
    }

    /**
     * Navega para o atributo seguinte.
     */
    void next() {
      if (domainList.size() - 1 > position) {
        position++;
        String key = domainList.get(position).getKey();
        attributeRange = domainAttributesRangeMap.get(key);
        go(key);
      }
    }

    /**
     * Navega para o primeiro atributo.
     */
    void rewind() {
      attributeValue.setSelectedIndex(-1);
      attributeValue.invalidate();
      position = -1;
      next();
    }

    /**
     * Exibe na tela os campos e valores correspondentes ao atributo cujo nome
     * foi recebido como parmetro deste mtodo.
     * 
     * @param key Nome do atributo
     */
    void go(String key) {
      previousButton.setVisible((position > 0) ? true : false);
      nextButton.setVisible((domainList.size() - 1 > position) ? true : false);
      attributeNameKey = key;
      attributeName.setText(LNG.get(attributeNameKey));
      Boolean isMultiSelect = (Boolean) domainList.get(position).getValue();
      ArrayList<Object> mapValue = map.get(key);

      // Exibe o ContainerSelection para seleo mltipla.
      if (isMultiSelect.booleanValue()) {
        attributeName.setVisible(false);
        attributeValue.setVisible(false);
        if (mapValue != null) {
          ArrayList<Object> idList = mapValue;
          Iterator<String> keyIterator = attributeRange.keySet().iterator();
          List<String> selectedItems = new ArrayList<String>();
          while (keyIterator.hasNext()) {
            Object attributeKey = keyIterator.next();
            for (int i = 0; i < idList.size(); i++) {
              if (idList.get(i).equals(attributeRange.get(attributeKey))) {
                selectedItems.add((String) attributeKey);
                break;
              }
            }
          }
          containerSelection.loadItems(attributeRange.keySet(), selectedItems);
        }
        else {
          containerSelection.loadItems(attributeRange.keySet(),
            new ArrayList<String>());
        }
        containerSelection.getPanel().setVisible(true);
      }

      // Exibe a JComboBlox para seleo nica.
      else {
        containerSelection.getPanel().setVisible(false);
        attributeName.setVisible(true);
        attributeValue.setVisible(true);
        attributeValue.setModel(new DefaultComboBoxModel(attributeRange
          .keySet().toArray()));
        if (mapValue != null) {
          Object attrValue = (mapValue).get(0);
          Object attrName = getAttributeName(attrValue);
          attributeValue.setSelectedItem(attrName);
        }
        else {
          attributeValue.setSelectedIndex(-1);
        }
      }
    }

    /**
     * Este mtodo recupera o nome de um atributo a partir de seu valor.
     * 
     * @param attrValue Valor do atributo
     * @return Nome do atributo.
     */
    private Object getAttributeName(Object attrValue) {
      Set<Entry<String, Object>> entries = attributeRange.entrySet();
      for (Entry<?, ?> entry : entries) {
        if (entry.getValue().equals(attrValue)) {
          return entry.getKey();
        }
      }
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void clearFields() {
    map.clear();
    navigator.rewind();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  boolean hasChanged(Permission permission) {
    return !((ChoicePermission) permission).getAttributes().equals(map);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean aFlag) {
    super.setVisible(aFlag);
    includePermissionButton.setEnabled(!aFlag);
    if (aFlag) {
      navigator.init();
    }
    else {
      nextButton.setVisible(false);
      previousButton.setVisible(false);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void changeClass() {
    super.changeClass();
    navigator.init();
  }
}
