/*
 * $Id: ExportFilesToDirAction.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.project.action;

import java.awt.event.ActionEvent;
import java.io.File;
import java.security.AccessControlException;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.project.ProjectFileContainer;
import csbase.logic.ClientProjectFile;

/**
 * Ao para exportar arquivos/diretrios para um diretrio. Usada quando o
 * usurio selecionou um diretrio ou mltiplos arquivos/diretrios.
 * <p>
 * Esta ao tenta primeiro acessar diretamente o sistema de arquivos local,
 * utilizando o <code>JFileChooser</code>. Caso no consiga (no caso dos jars
 * no estarem assinados), utiliza a API JNLP.
 */
public class ExportFilesToDirAction extends CommonFileExportAction {

  /**
   * Mensagem de erro para o caso em que se tentou exportar mltiplos arquivos
   * via JNLP.
   */
  private static final String JNLP_CANT_EXPORT_MULTIPLE =
    "JNLP_CANT_EXPORT_MULTIPLE";

  /**
   * Cria a ao para exportao para um diretrio.
   * 
   * @param container objeto contendo arquivos de projeto.
   */
  public ExportFilesToDirAction(ProjectFileContainer container) {
    super(container);
  }

  /**
   * Seleciona o diretrio de destino local da transferncia.
   * 
   * @return diretrio escolhido pelo usurio, ou <code>null</code> caso a
   *         operao tenha sido cancelada
   */
  private File getUserTarget() {
    JFileChooser chooser = getFileChooser();
    chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
    chooser.setMultiSelectionEnabled(false);
    int result = chooser.showSaveDialog(getWindow());
    if (result == JFileChooser.APPROVE_OPTION) {
      return chooser.getSelectedFile();
    }
    return null;
  }

  /**
   * Mtodo que dispara a ao. Dispara uma <code>ExportTask</code> encarregada
   * de fazer a transferncia dos arquivos selecionados para o disco local do
   * usurio.
   * <p>
   * Caso as permisses de acesso no possibilitem o acesso direto ao sistema de
   * arquivos local, a transferncia  feita via JNLP.
   * 
   * @param e informaes sobre o evento ocorrido.
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    ClientProjectFile[] selectedFiles = getSelectedFiles();
    if (selectedFiles == null || selectedFiles.length == 0) {
      return;
    }
    try {
      /*
       * Primeiro vamos tentar usar o JFileChooser nativo (assumindo que as
       * permisses de acesso possibilitam isto). Se no der certo, o catch mais
       * abaixo usar o file chooser do JNLP.
       */
      File userTarget = getUserTarget();
      if (userTarget == null) {
        return;
      }
      /*
       * Se o dir de destino ainda no existe, tentamos cri-lo. Se houver algum
       * erro, informamos o usurio e abortamos.
       */
      if (userTarget.exists() == false) {
        if (userTarget.mkdir() == false) {
          showError(LNG.get(CRITICAL_ERROR));
          return;
        }
      }

      /*
       * agora transferimos os arquivos, um de cada vez
       */
      Collection<ClientProjectFile> sourceFiles = selectTopFiles(selectedFiles);
      final String userTargetPath = userTarget.getAbsolutePath();
      for (Iterator<ClientProjectFile> iter = sourceFiles.iterator(); iter
        .hasNext();) {
        final ClientProjectFile sourceFile = iter.next();
        long fileSize = getFileSize(sourceFile);
        if (fileSize == -1L) {
          /*
           * houve algum erro na comunicao com o servidor, perguntamos ao
           * usurio se ele deseja prosseguir com a exportao dos demais
           * arquivos. A mensagem de erro j foi exibida pela task remota
           * executada pelo mtodo getFileSize()
           */
          int option =
            StandardDialogs.showYesNoDialog(getWindow(), LNG.get(TITLE),
              LNG.get("ExportFilesToDirAction.continue.question"));
          if (option == JOptionPane.NO_OPTION) {
            break;
          }
          continue;
        }
        if (fileSize == 0L && !confirmEmptyFileExport(sourceFile)) {
          break;
        }
        File targetFile =
          new File((userTargetPath + File.separator + sourceFile.getName()));
        if (targetFile.exists()) {
          if (!confirmOverwrite(targetFile)) {
            continue;
          }
          FileUtils.delete(targetFile);
        }
        transferFile(sourceFile, targetFile);
      }
    }
    catch (AccessControlException ex1) {
      /*
       * Acesso ao sistema de arquivos negado (rodando em "sandbox"): interao
       * com sistema de arquivos deve ser feita via JNLP. No  possvel
       * exportar diretrios nem mais de um arquivo.
       */
      if (selectedFiles.length > 1) {
        /*
         * TODO no poderamos transferir todos os arquivos, um por vez?
         */
        showError(JNLP_CANT_EXPORT_MULTIPLE);
        return;
      }

      transferFileUsingJNLP(selectedFiles[0]);
    }
  }
}
