package csbase.client.rest;

import java.awt.Window;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.ServerSocket;
import java.net.URI;
import java.rmi.RemoteException;
import java.security.DigestException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import javax.ws.rs.core.Configuration;

import org.glassfish.grizzly.http.server.CLStaticHttpHandler;
import org.glassfish.grizzly.http.server.HttpServer;
import org.glassfish.jersey.grizzly2.httpserver.GrizzlyHttpServerFactory;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.server.model.ModelProcessor;
import org.glassfish.jersey.server.model.Resource;
import org.glassfish.jersey.server.model.ResourceMethod;
import org.glassfish.jersey.server.model.ResourceModel;
import org.glassfish.jersey.server.spi.Container;
import org.glassfish.jersey.server.spi.ContainerLifecycleListener;

import csbase.client.Client;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.NotificationPanel;
import csbase.client.desktop.RemoteTask;
import csbase.client.login.LoginInterface;
import csbase.client.login.PreLogin;
import csbase.client.rest.ReloadFilter.RequestCounter;
import csbase.logic.SecureKey;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.RestServiceInterface;

import io.swagger.jaxrs.config.BeanConfig;

import tecgraf.javautils.core.lng.LNG;

/**
 * Controlador REST para o cliente.
 */
public class RestController {
  /**
   * Singleton
   */
  private static RestController instance;

  /**
   * Configuração de resources oferecidos.
   */
  private ResourceConfig resourceConfig;

  /**
   * Servidor HTTP
   */
  private HttpServer httpServer;

  /**
   * Container Jersey para atualização dinâmica de recursos
   */
  private Container container;

  /**
   * Token
   */
  private String token;

  /**
   * Usu�rio
   */
  private final LoginInterface userLogin;

  /**
   * Indicativo interno de developer mode (não demanda token)
   */
  private boolean developMode = false;

  /**
   * Pacotes de recursos da API rest
   */
  private final List<String> resourcePackages;

  /**
   * Conjunto de recursos por prefixo da aplicação (padrão appId/instanceId).
   */
  private final Map<String, Set<Object>> instancesByPrefix;

  /**
   * Mapa de prefixos da aplicação por classe de recurso.
   */
  private final Map<Class<?>, String> prefixByResourceClass;

  /**
   * Diretório de recursos para interface web da documentação
   */
  private String docFrontEndDirectory;

  /**
   * Bean de configuração de documentação Swagger.
   */
  private BeanConfig beanConfig;

  /**
   * Flag que inidica que o container HTTP deve ser reiniciado
   */
  private boolean shouldReload;

  /**
   * Define o diretório de recursos para interface web da documentação
   * @param docFrontEndDirectory diretório.
   */
  public void setDocFrontEndDirectory(String docFrontEndDirectory) {
    if (httpServer != null && httpServer.isStarted()) {
      throw new IllegalStateException("Server is already started");
    }
    this.docFrontEndDirectory = docFrontEndDirectory;
  }

  /**
   * Pporta do servidor HTTP
   */
  private int httpPort;

  /**
   * Método para recuperar a instância do singleton do controlador REST.
   * @return A instância única do controlador REST
   */
  public static RestController getInstance() {
    if (instance == null) {
      instance = new RestController();
    }
    return instance;
  }

  /**
   * Host padrão para acesso.
   */
  private static final String DEFAULT_HOST = "127.0.0.1";

  /**
   * Porta padrão do servidor REST
   */
  private static final int DEFAULT_PORT = 15000;

  /**
   * Determina uma porta TCP livre no host local
   * @return Uma porta TCP que não está sendo usada
   */
  private static int getAvailablePort() {
    ServerSocket socket = null;
    int port = -1;
    try {
      socket = new ServerSocket(0);
      port = socket.getLocalPort();
    }
    catch (IOException ex) {
    }
    finally {
      if (socket != null) {
        try {
          socket.close();
        }
        catch (IOException e) {
          throw new RuntimeException("You should handle this error.", e);
        }
      }
    }

    return port;
  }

  /**
   * Construtor privado do controlador REST, somente acessível via getInstance
   */
  private RestController() {
    this.userLogin = Client.getInstance().getLoginObject();

    //pacotes de recursos REST do cliente
    this.resourcePackages = new ArrayList<>();
    resourcePackages.add("csbase.client.rest.resources.v1.authentication");

    //objetos de recursos de aplicações
    instancesByPrefix = new HashMap<>();

    prefixByResourceClass = new HashMap<>();

    //Diretorio padrão do frontend swagger para documentação
    this.docFrontEndDirectory = "csbase/client/rest/swagger-ui/";

    if (userLogin instanceof PreLogin) {
      this.token = ((PreLogin) userLogin).getClientInstanceId();
      if (this.token.startsWith("TOKENID:")) {
        this.token = this.token.substring("TOKENID:".length());
      }
    }
    else {
      SecureKey sessionKey = new SecureKey();
      try {
        this.token = sessionKey.digest();
      }
      catch (DigestException e) {
        e.printStackTrace();
        this.token = userLogin.getClientInstanceId();
      }
    }
  }

  /**
   * Recupera o token de autenticação do usuário.
   * @return O token de login do usuário corrente.
   */
  public String getAuthToken() {
    return token;
  }

  /**
   * Verifica se o servidor está em modo Developer.
   * @return Retorna true no caso de estar no modo de desenvolvimento, ou false
   * caso contrario
   */
  public boolean isDeveloperMode() {
    return developMode;
  }

  /**
   * Verifica se um token de usuário é válido. No caso de estar no modo de
   * desenvolvimento, retorna true independentemente do token.
   * @param checkToken O token a ser verificado
   * @return true se o token passado é válido, ou false caso contrario
   */
  public boolean isTokenAuthorized(String checkToken) {
    return isDeveloperMode() || this.token.equals(checkToken);
  }

  /**
   * Recupera o login do usuario corrente.
   * @return a interface de login
   */
  public LoginInterface getUserLogin() {
    return userLogin;
  }

  /**
   * Encerra o servidor REST.
   */
  public void stopServer() {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();

      RemoteTask<Void> task = new RemoteTask<Void>() {
        @Override
        protected void performTask() throws Exception {
          ClientRemoteLocator.restService.unregisterClientHttpServer(token);
        }
      };
      final String msg = LNG.get("csbase.client.rest.server.unregistering");
      final DesktopFrame desktopFrame = DesktopFrame.getInstance();
      final Window window = (desktopFrame == null ? null : desktopFrame.getDesktopFrame());
      task.execute(window, null, msg);

      if (task.getError() != null) {
        task.getError().printStackTrace();
      }
    }
  }

  /**
   * Inicia o servidor REST na porta e host padrão.
   */
  public void startServer() {
    try {
      startServer(DEFAULT_HOST, DEFAULT_PORT);
    }
    catch (javax.ws.rs.ProcessingException ex) {
      //Tentar uma porta disponivel
      int availablePort = getAvailablePort();
      startServer(DEFAULT_HOST, availablePort);
    }
  }

  /**
   * Inicia o servidor REST no host padrão em uma porta específica.
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(int port) {
    startServer(DEFAULT_HOST, port);
  }

  /**
   * Inicia o servidor REST em uma porta e host específicos.
   * @param host Host a ser usado pelo servidor.
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(String host, int port) {
    this.httpPort = port;

    try {
      this.beanConfig = new BeanConfig();
      beanConfig.setVersion("1.0");
      beanConfig.setDescription(LNG.get("csbase.client.rest.server.description"));
      beanConfig.setBasePath("/");
      beanConfig.setResourcePackage(String.join(",", resourcePackages));
      beanConfig.setScan(true);

      this.resourceConfig = buildResourceConfig();

      this.resourceConfig.registerInstances(new ContainerLifecycleListener() {
        @Override
        public void onStartup(Container cont) {
          RestController.this.container = cont;
        }

        @Override
        public void onReload(Container cont) {
        }

        @Override
        public void onShutdown(Container cont) {
        }
      });

      final String baseURL = "http://" + host + ":" + port + "/";
      this.httpServer = GrizzlyHttpServerFactory.createHttpServer(URI.create(baseURL), this.resourceConfig);

      // Registro de UI do swagger
      CLStaticHttpHandler staticHttpHandler =
        new CLStaticHttpHandler(RestController.class.getClassLoader(), this.docFrontEndDirectory);
      httpServer.getServerConfiguration().addHttpHandler(staticHttpHandler, "/docs/");

      RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
        @Override
        protected void performTask() throws Exception {
          final RestServiceInterface restService = ClientRemoteLocator.restService;
          restService.registerClientHttpServer(token, baseURL);
          boolean devMode = restService.isClientDeveloperMode();
          setResult(devMode);
        }
      };

      final String taskMsg = LNG.get("csbase.client.rest.server.registering");
      final DesktopFrame desktopFrame = DesktopFrame.getInstance();
      final Window window = desktopFrame == null ? null : desktopFrame.getDesktopFrame();
      task.execute(window, null, taskMsg);

      final Exception error = task.getError();
      if (error == null) {
        this.developMode = task.getResult();
      }
      else {
        final String errMsg = error.getLocalizedMessage();
        final String msg = LNG.get("csbase.client.rest.server.start.error") + " - " + errMsg;
        addNotificationMessage(msg);
      }

      final String startMsg = LNG.get("csbase.client.rest.server.started");
      final String logMsg = startMsg + " - " + port + (isDeveloperMode() ? " (dev mode)" : "");
      addNotificationMessage(logMsg);

      final String apiMsg = LNG.get("csbase.client.rest.server.doc.notification") + " " + baseURL + "docs/";
      addNotificationMessage(apiMsg);
      logMessage("RestService", "Sevidor iniciado na porta " + port);
    }
    catch (Throwable ex) {
      ex.printStackTrace();
      //Repassar exce��o de porta invalida
      if (ex instanceof javax.ws.rs.ProcessingException) {
        throw ex;
      }
      final String apiMsg = LNG.get("csbase.client.rest.server.error") + " " + ex.getLocalizedMessage();
      addNotificationMessage(apiMsg);
    }
  }

  /**
   * Metodo para adicionar uma mensagem ao notification panel
   * @param msg Mensagem a ser impressa
   */
  private void addNotificationMessage(final String msg) {
    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    if (desktopFrame == null) {
      System.out.println("[RestController] " + msg);
      return;
    }

    final NotificationPanel notificationPanel = desktopFrame.getNotificationPanel();
    if (notificationPanel == null) {
      System.out.println("[RestController] " + msg);
      return;
    }

    notificationPanel
      .addNotificationLine(notificationPanel.makeNotificationLine(new Date(), "ClientRestService", msg, true));
  }

  /**
   * Metodo para registro de recursos no servidor REST.
   * @param classes Lista de classes de recursos a serem registradas.
   */
  public void registerService(Class<?>[] classes) {
    resourceConfig.registerClasses(classes);
  }

  /**
   * Adiciona uma lista de pacotes a lista de recursos a serem registrados no
   * servidor REST.
   * @param packages Lista de pacotes com resursos REST a serem adicionados
   */
  public void addResourcePackages(String... packages) {
    resourcePackages.addAll(Arrays.asList(packages));

    beanConfig.setResourcePackage(String.join(",", resourcePackages));
    beanConfig.setScan(true);

    if (container != null) {
      this.resourceConfig = buildResourceConfig();
      container.reload(this.resourceConfig);
    }
  }

  /**
   * Adiciona uma lista de instâncias de recursos a serem registrados no
   * servidor REST.
   * @param prefix O prefixo da aplicação
   * @param config Configuração de recursos REST a serem adicionados
   */
  public void addApplicationResources(String prefix, ResourceConfig config) {
    Set<Object> instances = config.getInstances();
    for (Object i : instances) {
      prefixByResourceClass.put(i.getClass(), prefix);
    }
    instancesByPrefix.put(prefix, instances);
    shouldReload = true;
    reloadContainer();
  }

  /**
   * Recarrega o container (ou registra um timer para fazer a recarga caso não
   * seja possível recarregar no momento).
   */
  private void reloadContainer() {
    RequestCounter counter = ReloadFilter.getCounter();
    synchronized (counter.getLock()) {
      if (shouldReload) {
        // Só recarrega efetivamente se não existirem requisições sendo
        // atendidas no momento.
        if (counter.isZero()) {
          if (container != null) {
            this.resourceConfig = buildResourceConfig();
            container.reload(this.resourceConfig);
          }
          shouldReload = false;
        }
        else {
          // Caso não seja poss�vel recarregar, registrar um timer para
          // tentar novamente.
          TimerTask timerTask = new TimerTask() {
            @Override
            public void run() {
              reloadContainer();
            }
          };
          Timer timer = new Timer("RestController reload timer");
          timer.schedule(timerTask, 300);
        }
      }
    }
  }

  /**
   * Remove uma lista de recursos registrados no servidor REST.
   * @param prefix O prefixo da aplicação para retirada de recursos
   */
  public void removeApplicationResources(String prefix) {
    instancesByPrefix.remove(prefix);

    prefixByResourceClass.entrySet().removeIf(entry -> entry.getValue().equals(prefix));

    shouldReload = true;
    reloadContainer();
  }

  /**
   * Consulta a porta REST usada pelo cliente.
   * @return porta.
   */
  public int getPort() {
    if (httpServer == null || !httpServer.isStarted()) {
      return -999;
    }
    return httpPort;
  }

  /**
   * Metodo para registro de pacotes com recursos no servidor REST.
   * @param packages Lista de packages de recursos a serem registradas.
   */
  public void registerPackages(String... packages) {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();
    }
    resourcePackages.addAll(Arrays.asList(packages));
    this.startServer();
  }

  /**
   * Faz log no eventlog do servidor.
   * @param type tipo
   * @param msg mensagem
   */
  private void logMessage(String type, String msg) {
    try {
      ClientRemoteLocator.eventLogService
        .addClientInformation(new String[] { "ClientRest" }, new String[] { type, msg });
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }
  }

  /**
   * Cria a configuração de recursos de acordo com os recursos atuais definidos
   * @return Uma nova configuração de recursos
   */
  private ResourceConfig buildResourceConfig() {
    final ResourceConfig rc = new ResourceConfig();

    // Processador de recursos das instâncias de aplicação
    rc.register(ApplicationApiModelProcessor.class);

    // Registro dos recursos REST do cliente
    rc.packages(true, resourcePackages.toArray(new String[resourcePackages.size()]));

    // Registra os recursos das aplicações
    instancesByPrefix.entrySet().forEach(e -> rc.registerInstances(e.getValue().toArray()));

    // Registro de recurso para serialização JSON
    rc.register(JacksonFeature.class);

    // Registro de recursos para tratamento de requisições e respostas
    rc.register(AuthFilter.class);
    rc.register(DebugMapper.class);
    rc.register(ReloadFilter.class);

    // Registro de classes do swagger
    rc.register(io.swagger.jersey.listing.ApiListingResourceJSON.class);
    rc.register(io.swagger.jaxrs.listing.ApiListingResource.class);
    rc.register(io.swagger.jaxrs.listing.SwaggerSerializers.class);
    return rc;
  }

  /**
   * Processador do modelo de recursos, usado para modificar os recursos
   * providos por instâncias de aplicação. Os recursos do próprio cliente não
   * são modificados.
   */
  public static class ApplicationApiModelProcessor implements ModelProcessor {

    /**
     * Cria um novo modelo de recursos, separados os recursos comuns a mais de
     * uma instância de aplicação em recursos próprios, prefixados pelo
     * identificador da instância.
     * <p>
     * {@inheritDoc}
     */
    @Override
    public ResourceModel processResourceModel(ResourceModel resourceModel, Configuration configuration) {
      ResourceModel.Builder newModelBuilder = new ResourceModel.Builder(false);

      for (Resource res : resourceModel.getResources()) {
        boolean isAppResource = false;
        for (Map.Entry<Class<?>, String> prefixByClass : RestController.getInstance().prefixByResourceClass
          .entrySet()) {
          if (res.getHandlerClasses().contains(prefixByClass.getKey())) {
            isAppResource = true;
            Resource newRes = createResource(res, prefixByClass.getValue(), prefixByClass.getKey(), true);
            newModelBuilder.addResource(newRes);
          }
        }
        if (!isAppResource) {
          newModelBuilder.addResource(res);
        }
      }

      return newModelBuilder.build();
    }

    /**
     * Cria um novo recurso, adicionando somente os métodos que pertencem à
     * instância especificada.
     * @param resource recurso original.
     * @param prefix prefixo da instância de aplicação.
     * @param clazz a classe que implementa o recurso na instância.
     * @param isParentResource verdadeiro se o recurso em questão é o recurso
     * pai.
     * @return o novo recurso.
     */
    private Resource createResource(Resource resource, String prefix, Class<?> clazz, boolean isParentResource) {
      final Resource.Builder resourceBuilder = Resource.builder().name(resource.getName());
      if (isParentResource) {
        resourceBuilder.path("/" + prefix + resource.getPath());
      }
      else {
        resourceBuilder.path(resource.getPath());
      }
      resource.getChildResources()
              .forEach((r) -> resourceBuilder.addChildResource(createResource(r, prefix, clazz, false)));

      for (final ResourceMethod resourceMethod : resource.getResourceMethods()) {
        Method classMethod = getClassMethod(resourceMethod);
        if (classMethod.getDeclaringClass().equals(clazz)) {
          resourceBuilder.addMethod(resourceMethod);
        }
      }

      return resourceBuilder.build();
    }

    /**
     * Obtem o método da classe que implementa o método do recurso.
     * @param resourceMethod o método do recurso.
     * @return o método da classe.
     */
    private Method getClassMethod(final ResourceMethod resourceMethod) {
      return resourceMethod.getInvocable().getDefinitionMethod();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ResourceModel processSubResource(ResourceModel resourceModel, Configuration configuration) {
      return resourceModel;
    }
  }

}
