package csbase.client.project.dialogs;

import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.text.MessageFormat;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.ScrollPaneConstants;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.tasks.CheckAreaReservedTask;
import csbase.client.util.ClientUtilities;
import csbase.client.util.NumericTextField;
import csbase.client.util.user.UserDialog;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Dilogo para construo/manuteno de projetos.
 * 
 * @author jnlopes
 */
public abstract class ProjectBuildingDialog extends JDialog {

  /**
   * O campo que indica se o projeto quer ou no uma reserva de rea.
   */
  final protected JCheckBox lockingBox = new JCheckBox(
    LNG.get("ProjectBuildingDialog.locking.checkbox"));

  /**
   * O campo com o tamanho da rea reservada.
   */
  final protected NumericTextField sizeField = new NumericTextField(20);

  /**
   * Campo com nome do dono do projeto.
   */
  final protected JTextField projectOwnerField = new JTextField(User
    .getLoggedUser().getLogin());

  /**
   * Campo de nome do projeto.
   */
  final protected JTextField nameField = new JTextField();

  /**
   * Campo de descrio do projeto.
   */
  final JTextArea descrField = new JTextArea();

  /**
   * Boto para busca de usurios.
   */
  final JButton searchButton = new JButton();

  /**
   * Projeto a ser criado.
   */
  CommonClientProject project;

  /**
   * Indica se o projeto utiliza reserva de rea.
   */
  final boolean allocation;

  /**
   * Boto de confirmao para criao de projeto.
   */
  final JButton okButton = new JButton();

  /**
   * Painel extra para preenchimento de dados que sero adicionados como
   * atributos na hashtable do projeto.
   */
  ProjectDataExtensionPanel extraPanel;

  /**
   * Instancia a viso de informaes de projeto usada para criao de projeto.
   * O dilogo  criado e apresentado.
   * 
   * @param owner O frame da aplicao pai.
   */
  ProjectBuildingDialog(final Window owner) {
    this(owner, null, null);
  }

  /**
   * Instancia a viso de informaes de projeto usada para criao de projeto.
   * O dilogo  criado e apresentado.
   * 
   * @param owner O frame da aplicao pai.
   * @param extraPanel Painel extra.
   */
  ProjectBuildingDialog(final Window owner,
    final ProjectDataExtensionPanel extraPanel) {
    this(owner, null, extraPanel);
  }

  /**
   * Instancia a viso de informaes de projeto usada para criao de projeto.
   * O dilogo  criado e apresentado.
   * 
   * @param owner O frame da aplicao pai.
   * @param project O projeto a ser modificado.
   */
  ProjectBuildingDialog(final Window owner, final CommonClientProject project) {
    this(owner, project, null);
  }

  /**
   * Instancia a viso de informaes de projeto usada para criao de projeto.
   * O dilogo  criado e apresentado.
   * 
   * @param owner O frame da aplicao pai.
   * @param project O projeto a ser modificado.
   * @param extraPanel Painel extra.
   */
  ProjectBuildingDialog(final Window owner, final CommonClientProject project,
    final ProjectDataExtensionPanel extraPanel) {
    super(owner);
    this.allocation = checkAreaReserved();
    this.project = project;
    this.extraPanel = extraPanel;
    clearDialog();
    buildDialog();
    addActions();
  }

  /**
   * Mostra um dilogo para que o usurio crie um projeto. O usurio informa os
   * dados do projeto. Caso a operao seja confirmada, o servidor  requisitado
   * para executar o servio de criao do projeto.
   * 
   * @param owner O frame da aplicao pai.
   * @param extraPanel Painel extra para que sejam preenchidos atributos do
   *        projeto.
   * 
   * @return O projeto criado.
   */
  public static CommonClientProject createProject(final Window owner,
    final ProjectDataExtensionPanel extraPanel) {
    final ProjectCreationDialog pcd =
      new ProjectCreationDialog(owner, extraPanel);
    pcd.setModalityType(ModalityType.APPLICATION_MODAL);
    pcd.pack();
    pcd.setLocationRelativeTo(owner);
    pcd.setVisible(true);
    return pcd.project;
  }

  /**
   * Mostra um dilogo para que o usurio altere um projeto. O usurio informa
   * os dados do projeto. Caso a operao seja confirmada, o servidor 
   * requisitado para executar o servio de alterao do projeto.
   * 
   * @param owner O frame da aplicao pai.
   * @param project O projeto que vai sofrer a alterao.
   * @param extraPanel Painel extra para que sejam preenchidos atributos do
   *        projeto.
   * @return O projeto alterado.
   */
  public static CommonClientProject updateProject(final Window owner,
    final CommonClientProject project,
    final ProjectDataExtensionPanel extraPanel) {
    final ProjectUpdateDialog pud =
      new ProjectUpdateDialog(owner, project, extraPanel);
    pud.setModalityType(ModalityType.APPLICATION_MODAL);
    pud.pack();
    pud.setLocationRelativeTo(owner);
    pud.setVisible(true);
    return project;
  }

  /**
   * Monta o dilogo de criao de projetos.
   */
  private void buildDialog() {
    nameField.setColumns(22);
    descrField.setColumns(22);
    descrField.setRows(3);

    projectOwnerField.setEditable(false);
    projectOwnerField.setColumns(10);
    sizeField.setColumns(10);

    final JScrollPane descrPane = new JScrollPane(descrField);
    descrPane
      .setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
    descrPane
      .setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);

    setLayout(new GridBagLayout());
    final Insets regularEdgeInsets = new Insets(5, 15, 5, 15);
    final Insets separatingEdgeInsets = new Insets(15, 15, 5, 15);

    add(new JLabel(LNG.get("ProjectCreationDialog.project.name.label")),
      new GBC(0, 0).west().insets(separatingEdgeInsets));
    add(nameField, new GBC(1, 0).horizontal().insets(separatingEdgeInsets));

    add(new JLabel(LNG.get("ProjectBuildingDialog.project.descr.label")),
      new GBC(0, 1).height(2).northwest().insets(regularEdgeInsets));
    add(descrPane, new GBC(1, 1).height(2).both().insets(regularEdgeInsets));

    int i = 3;
    final User user = User.getLoggedUser();
    if (user.isAdmin()) {
      searchButton.setAction(new AbstractAction() {

        @Override
        public void actionPerformed(ActionEvent e) {
          final String title =
            LNG.get("ProjectBuildingDialog.owner.selection.title");
          try {
            final Set<Object> ids =
              UserDialog.showUserSelectionDialog(ProjectBuildingDialog.this,
                getUsers(), false, title);

            if (ids != null) {
              final Object userId = ids.toArray()[0];
              projectOwnerField.setText((String) userId);
            }
          }
          catch (Exception e1) {
            e1.printStackTrace();
            StandardDialogs.showErrorDialog(ProjectBuildingDialog.this,
              LNG.get("ProjectBuildingDialog.obtaining.user.error.title"),
              LNG.get("ProjectBuildingDialog.obtaining.user.error.text"));
            return;
          }
        }
      });

      searchButton.setIcon(ApplicationImages.ICON_FIND_USER_16);
      searchButton.setMargin(new Insets(0, 0, 0, 0));
      searchButton.setToolTipText(LNG
        .get("ProjectBuildingDialog.search.owner.tooltip"));
      final JPanel ownerPanel = new JPanel(new GridBagLayout());
      ownerPanel.add(projectOwnerField, new GBC(0, 0).insets(0, 0, 0, 5));
      ownerPanel.add(searchButton, new GBC(1, 0).insets(0, 5, 0, 0));
      add(new JLabel(LNG.get("ProjectBuildingDialog.user.login.label")),
        new GBC(0, i).west().insets(separatingEdgeInsets));
      add(ownerPanel, new GBC(1, i).west().insets(separatingEdgeInsets));
      i++;
    }

    if (allocation) {
      // Check-box com a opo de reservar ou no uma rea de projeto
      lockingBox.setSelected(false);
      // Tamanho da rea a ser reservada
      final JLabel sizeLabel =
        new JLabel(LNG.get("ProjectBuildingDialog.locking.label"));
      sizeLabel.setEnabled(false);
      sizeField.setEnabled(false);
      // O tamnho somente  habilitado se a opo de reserva for selecionada
      lockingBox.addItemListener(new ItemListener() {

        @Override
        public void itemStateChanged(ItemEvent e) {
          final boolean condition = e.getStateChange() == ItemEvent.SELECTED;
          sizeLabel.setEnabled(condition);
          sizeField.setEnabled(condition);
        }
      });

      final JPanel lockPanel = new JPanel(new GridBagLayout());
      lockPanel.add(sizeField, new GBC(0, 0).insets(0, 0, 0, 5));
      lockPanel.add(sizeLabel, new GBC(1, 0).insets(0, 5, 0, 0));
      add(lockingBox, new GBC(0, i).west().insets(5, 12, 5, 0));
      add(lockPanel, new GBC(1, i).west().insets(regularEdgeInsets));
      i++;
    }

    if (extraPanel != null) {
      add(extraPanel, new GBC(0, i).both().width(2).insets(5, 12, 5, 0));
      i++;
    }

    final JPanel buttonPanel = makeButtonPanel();
    add(buttonPanel, new GBC(0, i).width(2).horizontal().insets(15));
  }

  /**
   * Limpa os campos do dilogo e atribui os listeners a esses campos.
   */
  private void clearDialog() {
    // Inicializa os campos
    nameField.setText("");
    descrField.setText("");

    /*
     * Adiciona um listener ao campo de nome para que, quando um nome for
     * digitado, o boto principal seja habilitado.
     */
    final Document document = nameField.getDocument();
    document.addDocumentListener(new DocumentListener() {

      @Override
      public void changedUpdate(DocumentEvent e) {
        // vazio
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        setConfirmButtonState();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        setConfirmButtonState();
      }

      private void setConfirmButtonState() {
        String name = nameField.getText().trim();
        if (name.isEmpty()) {
          okButton.setEnabled(false);
        }
        else {
          okButton.setEnabled(true);
        }
      }
    });
  }

  /**
   * Cria um painel com os botes que vo ser usados no dilogo de criao de
   * projeto. Na criao de um projeto, o usurio pode acionar o boto de
   * criao, que confirma a operao, ou pode acionar o boto de cancelamento,
   * que desiste da operao. O boto de criao somente  habilitado quando o
   * usurio tiver informado um nome do projeto.
   * 
   * @return O painel recm-criado.
   */
  private JPanel makeButtonPanel() {

    final JPanel panel = new JPanel(new FlowLayout(FlowLayout.CENTER, 11, 0));

    // Boto de criao de projeto
    initCreateButton();
    panel.add(okButton);

    // Boto de cancelamento da operao de criao.
    final JButton cancelButton =
      new JButton(LNG.get("ProjectCreationDialog.cancel.button"));
    cancelButton.addActionListener(new ActionListener() {

      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectBuildingDialog.this.dispose();
      }
    });
    panel.add(cancelButton);

    ClientUtilities.adjustEqualSizes(okButton, cancelButton);
    return panel;
  }

  /**
   * Inicializa o boto para confirmar criao de projeto.
   */
  private void initCreateButton() {
    setOkButton(okButton);
    okButton.setEnabled(false);
  }

  /**
   * Adiciona aes relacionadas ao pressionamento de teclas.
   */
  protected void addActions() {
    /*
     * Associao da tecla ESC com o fechamento da tela.
     */
    AbstractAction cancelAction = new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectBuildingDialog.this.dispose();
      }
    };

    KeyStroke cancelKeyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);

    JComponent component = this.getRootPane();

    addActionToComponent(cancelAction, component, cancelKeyStroke,
      JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);

  }

  /**
   * Adiciona uma action relativa ao pressionamento de uma tecla a um
   * componente.
   * 
   * @param action Ao que se deseja associar a um componente.
   * @param component Componente.
   * @param keyStroke Tecla associada  ao.
   * @param focusSituation Condio de pressionamento da tecla.
   */
  protected void addActionToComponent(AbstractAction action,
    JComponent component, KeyStroke keyStroke, int focusSituation) {

    InputMap inputMap = component.getInputMap(focusSituation);
    ActionMap actionMap = component.getActionMap();

    String actionMapKey = keyStroke.toString();
    inputMap.put(keyStroke, actionMapKey);
    actionMap.put(actionMapKey, action);
  }

  /**
   * Retorna uma lista de usurios do sistema
   * 
   * @return A lista de usurios do sistema.
   * @throws Exception se a tarefa de busca dos usurios o fizer
   */
  private List<UserOutline> getUsers() throws Exception {

    final RemoteTask<List<UserOutline>> rt =
      new RemoteTask<List<UserOutline>>() {

        /**
         * {@inheritDoc}
         */
        @Override
        protected void performTask() throws Exception {
          final List<UserOutline> allUsers = User.getAllOutlines();
          setResult(allUsers);
        }
      };

    final Window owner = getOwner();
    rt.execute(owner, LNG.get("ProjectCreationDialog.retrieving.users.title"),
      LNG.get("ProjectCreationDialog.retrieving.users.msg"));

    if (rt.wasCancelled()) {
      StandardDialogs.showErrorDialog(owner,
        LNG.get("ProjectCreationDialog.users.retrieval.error.title"),
        LNG.get("ProjectCreationDialog.users.retrieval.error.msg"));
      return null;
    }

    if (rt.getStatus() != true) {
      final Exception exception = rt.getError();
      throw exception;
    }

    final List<UserOutline> users = rt.getResult();
    if (users == null) {
      StandardDialogs.showErrorDialog(owner,
        LNG.get("ProjectCreationDialog.no.users.title"),
        LNG.get("ProjectCreationDialog.no.users.msg"));
    }
    return users;
  }

  /**
   * Verifica se o projeto utiliza reserva de rea.
   * 
   * @return Boolean indicando se o projeto utiliza reserva de rea.
   */
  private boolean checkAreaReserved() {

    final Window window = getOwner();
    final CheckAreaReservedTask task = new CheckAreaReservedTask();
    task.execute(window, LNG.get("ProjectCreationDialog.area.reserved.title"),
      LNG.get("ProjectCreationDialog.area.reserved.msg"));

    if (task.wasCancelled()) {
      final String title =
        LNG.get("ProjectCreationDialog.area.reserved.cancelled.title");
      final String msg =
        LNG.get("ProjectCreationDialog.area.reserved.cancelled.msg");
      StandardDialogs.showErrorDialog(window, title, msg);
      return false;
    }

    if (task.getStatus() != true) {
      final Exception exception = task.getError();
      final String title =
        LNG.get("ProjectCreationDialog.area.reserved.exception.title");
      final String msg =
        LNG.get("ProjectCreationDialog.area.reserved.exception.msg") + "\n"
          + exception.getMessage();
      StandardDialogs.showErrorDialog(window, title, msg);
      return false;
    }

    return task.getResult();
  }

  /**
   * Monta o boto de confirmao.
   * 
   * @param confirmButton O boto de confirmao.
   */
  abstract void setOkButton(final JButton confirmButton);

}

/**
 * Dilogo para criao de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
class ProjectCreationDialog extends ProjectBuildingDialog {

  /**
   * Instancia a viso de informaes de projeto usada para criao de projeto.
   * O dilogo  criado e apresentado.
   * 
   * @param owner O frame da aplicao pai.
   * @param extraPanel Painel extra.
   */
  ProjectCreationDialog(final Window owner,
    final ProjectDataExtensionPanel extraPanel) {
    super(owner, extraPanel);
  }

  /**
   * Mtodo que define o comportamento da ao de criao de projeto. Se o
   * projeto  criado com sucesso, este dilogo  fechado. Caso contrrio, 
   * exibida uma mensagem ao usurio informando o motivo pelo qual o projeto no
   * foi criado, e este dilogo permanece.
   */
  private void createProjectAction() {
    final User loggedUser = User.getLoggedUser();
    /*
     * Caso o usurio seja o administrador, ento o dono do projeto sendo criado
     * pode ser outro usurio (usurios comuns no podem criar projetos no nome
     * de outros).
     */
    final Object ownerId;
    if (loggedUser.isAdmin()) {
      ownerId = projectOwnerField.getText();
    }
    else {
      ownerId = loggedUser.getId();
    }

    final String projectName = nameField.getText();
    if (!ClientUtilities.isValidFileName(projectName)) {
      StandardDialogs.showErrorDialog(ProjectCreationDialog.this, getTitle(),
        LNG.get("ProjectCreationDialog.character.error.message"));
      return;
    }
    final String description = descrField.getText();

    final Hashtable<String, Object> panelDataTable =
      new Hashtable<String, Object>();
    if (extraPanel != null) {
      panelDataTable.putAll(extraPanel.getPanelData());
    }

    final Window window = ProjectCreationDialog.this.getOwner();
    if (lockingBox != null && lockingBox.isSelected()) {
      final String stas = sizeField.getText();
      if (stas.trim().isEmpty()) {
        StandardDialogs.showErrorDialog(window,
          LNG.get("ProjectCreationDialog.invalid.space.definition.title"),
          LNG.get("ProjectCreationDialog.invalid.space.definition.msg"));
        return;
      }

      final int spaceToAllocate = Integer.parseInt(stas);
      if (!createProjectWithAreaReserved(window, projectName, description,
        String.valueOf(ownerId), panelDataTable, spaceToAllocate)) {
        return;
      }
    }
    else {
      project =
        createProject(window, projectName, description,
          String.valueOf(ownerId), panelDataTable);
      if (project == null) {
        return;
      }
    }
    ProjectCreationDialog.this.dispose();

  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void addActions() {

    super.addActions();

    /*
     * Associao da tecla ENTER pressionada com a criao do projeto, quando o
     * foco estiver no campo de nome do projeto.
     */
    AbstractAction okAction = new AbstractAction() {

      @Override
      public void actionPerformed(ActionEvent e) {
        createProjectAction();
      }
    };

    KeyStroke okKeyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
    addActionToComponent(okAction, nameField, okKeyStroke,
      JComponent.WHEN_FOCUSED);
  }

  /**
   * Cria um projeto (sem reserva de rea).
   * 
   * @param window O frame sobre o qual a task de criao de projeto deve rodar.
   * @param projectName O nome do projeto.
   * @param description A descrio do projeto.
   * @param ownerId O id do dono do projeto.
   * @param extraData Hashtable de atributos a serem salvos no projeto.
   * @return O projeto criado.
   */
  private CommonClientProject createProject(final Window window,
    final String projectName, final String description, final String ownerId,
    final Hashtable<String, Object> extraData) {

    final RemoteTask<CommonClientProject> task =
      new RemoteTask<CommonClientProject>() {

        @Override
        public void performTask() throws Exception {
          final ProjectServiceInterface psi =
            ClientRemoteLocator.projectService;
          final CommonProjectInfo cpi = new CommonProjectInfo();
          cpi.name = projectName;
          cpi.userId = ownerId;
          cpi.description = description;
          if (extraData != null) {
            cpi.getAttributes().putAll(extraData);
          }
          final CommonClientProject prj = psi.createProject(cpi);
          setResult(prj);
        }
      };
    if (task.execute(window, getTitle(),
      LNG.get("ProjectCreationDialog.waiting.create.title"))) {
      return task.getResult();
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void setOkButton(final JButton confirmButton) {

    //okAction 

    confirmButton.setAction(new AbstractAction() {

      @Override
      public void actionPerformed(ActionEvent e) {
        createProjectAction();
      }
    });
    confirmButton.setText(LNG.get("ProjectCreationDialog.creation.button"));
  }

  /**
   * Cria um projeto com reserva de rea e painel extra.
   * 
   * @param window O frame sobre o qual a task de criao de projeto deve rodar.
   * @param projectName O nome do projeto.
   * @param description A descrio do projeto.
   * @param ownerId O id do dono do projeto.
   * @param extraData Hashtable de atributos a serem salvos no projeto.
   * @param spaceToAllocate O espao em GB a ser alocado.
   * @return True, se o projeto foi criado com sucesso. False, caso contrrio.
   */
  private boolean createProjectWithAreaReserved(final Window window,
    final String projectName, final String description, final String ownerId,
    final Hashtable<String, Object> extraData, final int spaceToAllocate) {
    final RemoteTask<Void> task = new RemoteTask<Void>() {

      @Override
      public void performTask() throws Exception {
        final ProjectServiceInterface psi = ClientRemoteLocator.projectService;
        final CommonProjectInfo cpi = new CommonProjectInfo();
        cpi.name = projectName;
        cpi.userId = ownerId;
        cpi.description = description;
        // TODO - A linha abaixo parece desnecessria. Confirmar.
        //    cpi.setAttribute(LOCKING_AREA_SIZE, new Long(lockingAreaSize));
        if (extraData != null) {
          cpi.getAttributes().putAll(extraData);
        }
        psi.createProjectWithAllocation(cpi, spaceToAllocate);
      }
    };
    if (task.execute(window, getTitle(),
      LNG.get("ProjectCreationDialog.waiting.create.message"))) {
      /*
       * Informa que o projeto foi criado, mas est bloqueado
       */
      final String message =
        LNG.get("ProjectCreationDialog.project.creation.withlock.message");
      StandardDialogs.showInfoDialog(window, getTitle(),
        MessageFormat.format(message, new Object[] { projectName }));
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTitle() {
    return LNG.get("ProjectCreationDialog.creation.title");
  }

}

/**
 * Dilogo para alterao de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
class ProjectUpdateDialog extends ProjectBuildingDialog {

  /**
   * Instancia a viso de informaes de projeto usada para alterao de
   * projeto. O dilogo  criado e apresentado.
   * 
   * @param owner O frame da aplicao pai.
   * @param project O projeto que vai sofrer a alterao.
   * @param extraPanel Painel extra.
   */
  ProjectUpdateDialog(final Window owner, final CommonClientProject project,
    final ProjectDataExtensionPanel extraPanel) {
    super(owner, project, extraPanel);
    fillDialog();
  }

  /**
   * Preenche o dilogo com as informaes correntes do projeto a ser
   * modificado.
   */
  private void fillDialog() {
    nameField.setText(project.getName());
    nameField.setEnabled(false);
    descrField.setText(project.getDescription());
    projectOwnerField.setText(String.valueOf(project.getUserId()));
    searchButton.setEnabled(false);
    final long lockingAreaSize = project.getLockingAreaSize();
    if (lockingAreaSize > 0) {
      lockingBox.setSelected(true);
      sizeField.setEnabled(false);
      sizeField.setText(String.valueOf(lockingAreaSize));
    }
    lockingBox.setEnabled(false);
  }

  /**
   * Altera dados do projeto.
   */
  private void updateProject() {
    final String description = descrField.getText();
    updateProject(getOwner(), project, description);
    ProjectUpdateDialog.this.dispose();
  }

  /**
   * Altera dados do projeto.
   * 
   * @param window O frame sobre o qual a task de criao de projeto deve rodar.
   * @param ccp O projeto a ser modificado.
   * @param description A nova descrio do projeto.
   * 
   *        NOTA: No momento apenas a descrio do projeto pode ser alterada.
   */
  private void updateProject(final Window window,
    final CommonClientProject ccp, final String description) {

    ccp.setDescription(description);
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        ccp.modify();
      }
    };
    if (task.execute(window, getTitle(),
      LNG.get("ProjectUpdateDialog.waiting.update.message"))) {
      /*
       * Informa que a modificao ocorreu com sucesso
       */
      StandardDialogs.showInfoDialog(window, getTitle(),
        LNG.get("ProjectUpdateDialog.update.success.message"));
    }
  }

  /**
   * Informa o ttulo que vai ser utilizado na janela de alterao de projeto.
   * 
   * @return O ttulo da janela.
   */
  @Override
  public String getTitle() {
    return LNG.get("ProjectUpdateDialog.update.title");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void setOkButton(JButton confirmButton) {
    confirmButton.setAction(new AbstractAction() {

      @Override
      public void actionPerformed(ActionEvent e) {
        updateProject();
      }
    });
    confirmButton.setText(LNG.get("ProjectUpdateDialog.update.button"));
  }

}
