package csbase.client.rest;

import java.util.Date;
import java.util.Locale;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.ContainerResponseContext;
import javax.ws.rs.container.ContainerResponseFilter;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.Provider;

import com.sun.jersey.core.util.Base64;

import csbase.client.ClientServerManager;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.NotificationPanel;
import csbase.client.login.UserPasswordLogin;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.exception.CSBaseException;
import csbase.logic.Session;
import csbase.remote.UserPasswordAuthenticator;
import tecgraf.javautils.core.lng.LNG;

/**
 * Filtro de autenticao.
 *
 * @author Tecgraf/PUC-Rio
 */
@Provider
public class AuthFilter implements ContainerRequestFilter,
  ContainerResponseFilter {

  /**
   * Instncia do REST Controller.
   */
  static final private RestController restController = RestController
    .getInstance();

  /**
   * Criador de exceo para lanar caso o usuario nao esteja autorizado
   *
   * @param message a mensagem.
   * @return excepo de autorizao.
   */
  private static WebApplicationException buildUnauthorizedResponse(
    String message) {
    final String realmStr = "Basic realm=\"CSBaseClientRestController\"";
    return new WebApplicationException(Response.status(
      Response.Status.UNAUTHORIZED).header(HttpHeaders.WWW_AUTHENTICATE,
        realmStr).entity(message).build());
  }

  /**
   * Filtro de requisies para verificar o Token de autoirizao
   *
   * @param containerRequestContext O objeto que representa a requisio do
   *        cliente
   * @throws WebApplicationException No caso de falha de autorizao
   */
  @Override
  public void filter(ContainerRequestContext containerRequestContext)
    throws WebApplicationException {
    if (restController.isDeveloperMode()) {
      return;
    }

    String auth = containerRequestContext.getHeaderString("Authorization");
    if (auth == null) {
      String msg = LNG.get("csbase.client.rest.no.authorization.token");
      throw buildUnauthorizedResponse(msg);
    }

    //Validacao do cabealho de autoziacao via Login/Senha
    if (auth.startsWith("Basic")) {

      //Verificar se o metodo de autenticacao do cliente  via Login/Senha
      if (!(restController.getUserLogin() instanceof UserPasswordLogin)) {
        throw buildUnauthorizedResponse("Invalid login type");
      }

      auth = auth.replaceFirst("[Bb]asic ", "");
      String userColonPass = Base64.base64Decode(auth);

      String[] clientAuth = userColonPass.split(":");

      //Erro de sintaxe do cabecalho de autorizacao
      if (clientAuth.length != 2) {
        throw buildUnauthorizedResponse("Invalid Authorization header");
      }

      String login = clientAuth[0];
      String passwd = clientAuth[1];

      //Verificar se usuario  o mesmo que esta logado atualmente
      if (!ClientRemoteMonitor.getInstance().getLogin().equals(login)) {
        throw buildUnauthorizedResponse("Different user is logged in");
      }

      //Autenticar usuario no servidor
      try {
        UserPasswordAuthenticator authenticator = new UserPasswordAuthenticator(
          login, passwd, Locale.getDefault());
        Session authenticate = authenticator.authenticate(ClientServerManager
          .getInstance().getServer());

        //Verificar se usuario  autorizado pelo servidor
        if (authenticate == null) {
          throw buildUnauthorizedResponse("Invalid user/password");
        }

      }
      catch (CSBaseException e) {
        //Houve um erro ao verificar o login
        e.printStackTrace();
        throw buildUnauthorizedResponse(e.getLocalizedMessage());
      }
      //Tudo certo, usuario autenticado
      return;
    }

    //Validacao do cabealho de autoziacao via Token
    if (auth.startsWith("Bearer")) {
      auth = auth.replaceFirst("[Bb]earer ", "");

      //Verificar se o token bate com o token de autorizacao do RestController
      if (!restController.isTokenAuthorized(auth)) {
        String msg = LNG.get("csbase.client.rest.invalid.authorization.token");
        throw buildUnauthorizedResponse(msg);
      }
      return;
    }

    throw buildUnauthorizedResponse("Uknown Authorization method");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void filter(ContainerRequestContext containerRequestContext,
    ContainerResponseContext containerResponseContext)
    throws WebApplicationException {

    //set cors header for documentation
    containerResponseContext.getHeaders().add("Access-Control-Allow-Origin",
      "*");

    //Se est em modo de desenvolvedor, logue as requisicoes no notification panel
    if (restController.isDeveloperMode()) {
      NotificationPanel notificationPanel = DesktopFrame.getInstance()
        .getNotificationPanel();
      String method = containerRequestContext.getMethod();
      String uri = containerRequestContext.getUriInfo().getAbsolutePath()
        .toString();
      int status = containerResponseContext.getStatus();
      String statusStr = containerResponseContext.getStatusInfo().toString();

      final String msg = "[DEV-MODE] Requisio REST: " + method + " " + uri
        + " -> " + status + " " + statusStr;
      notificationPanel.addNotificationLine(notificationPanel
        .makeNotificationLine(new Date(), "ClientRestService", msg, true));
    }
  }
}
