package csbase.client.algorithms;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.text.Document;
import javax.swing.text.html.HTMLEditorKit;

import csbase.logic.algorithms.parameters.ParameterDocumentation;
import csbase.logic.algorithms.parameters.ParameterRegistry;
import csbase.logic.algorithms.parsers.MultipleParameterFactory;
import csbase.logic.algorithms.parsers.ParameterFactory;
import csbase.logic.algorithms.parsers.elements.IElementStructure;
import csbase.logic.algorithms.parsers.elements.attributes.IElementAttribute;
import csbase.remote.ClientRemoteLocator;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;

/**
 * Dilogo para exibio da estrutura dos parmetros disponveis para
 * configurao dos algoritmos.
 */
public class ParametersDescriptionDialog extends JDialog {
  /**
   * Lista de fbricas dos parmetros disponveis.
   */
  private Map<String, ParameterFactory> factories;
  /** rea com o contedo */
  private JEditorPane textArea;
  /** Registro dos parmetros do sistema */
  private ParameterRegistry registry;
  /** Locale a ser utilizado ao buscar a documentao dos parmetros */
  private Locale locale;
  /** Margem padro */
  private static int DEFAULT_MARGIN = 25;
  /** Margem para elementos que geram mais de um parmetro */
  private static int MULTIPLE_VALUES_MARGIN = 15;
  /** Margem dos atributos */
  private static int ATTRIBUTE_MARGIN = 35;
  /** Margem do ttulo dos elementos filhos */
  private static int CHILDREN_TITLE_MARGIN = 35;
  /** Margem do contedo dos elementos filhos */
  private static int CHILDREN_CONTENT_MARGIN = 60;

  /**
   * Construtor.
   * 
   * @param owner Janela me.
   * @param modality Modalidade do dilogo.
   * @param factories lista de fbricas dos parmetros disponveis.
   */
  public ParametersDescriptionDialog(Window owner, ModalityType modality,
    Map<String, ParameterFactory> factories) {
    super(owner, ModalityType.MODELESS);
    setTitle(LNG.get("ParametersDescriptionDialog.title"));

    this.factories = factories;

    try {
      registry = ClientRemoteLocator.algorithmService.getParameterRegistry();
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }

    locale = LNG.getLocale();

    addEscListener();
    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    String content = describeParameters();

    mountPanel();
    updateTextArea(content);

    pack();
    setLocationRelativeTo(owner);
  }

  /**
   * Ajusta a tecla esc para fechar o dilogo.
   */
  private void addEscListener() {
    final AbstractAction cancelAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        dispose();
      }
    };

    final int mode = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    final InputMap inputMap = rootPane.getInputMap(mode);
    final ActionMap actionMap = rootPane.getActionMap();
    final int esc = KeyEvent.VK_ESCAPE;
    final KeyStroke cancelStroke = KeyStroke.getKeyStroke(esc, 0);
    final String actionMapKey = cancelStroke.toString();
    inputMap.put(cancelStroke, actionMapKey);
    actionMap.put(actionMapKey, cancelAction);
  }

  /**
   * Monta a descrio dos parmetros em HTML.
   * 
   * @return descrio dos parmetros.
   */
  private String describeParameters() {
    StringBuilder str = new StringBuilder();
    for (ParameterFactory factory : factories.values()) {
      String description = getDescription(factory);
      str.append(description);
    }
    return str.toString();
  }

  /**
   * Obtm a descrio de um parmetro.
   * 
   * @param factory fbrica do parmetro.
   * @return descrio do parmetro formatada em HTML.
   */
  private String getDescription(ParameterFactory factory) {

    StringBuilder description = new StringBuilder();

    IElementStructure<?> structure = factory.getParameterStructures().get(0);

    description.append("<h1>");
    description.append(LNG.get("ParametersDescriptionDialog.element") + " "
      + structure.getName());
    description.append("</h1>");

    if (MultipleParameterFactory.class.isInstance(factory)) {
      MultipleParameterFactory multipleFactory =
        (MultipleParameterFactory) factory;

      getMultipleParameterDescription(description, multipleFactory, true);

      description.append("<br>");

      getMultipleParameterDescription(description, multipleFactory, false);
    }
    else {

      ParameterDocumentation doc = registry.getParameterDocumentation(factory
        .getParameterStructures().get(0), locale);

      getTitleAndDescription(description, doc);

      List<IElementAttribute<?>> attributes = new ArrayList<>(structure
        .getAttributes());
      attributes.sort(Comparator.<IElementAttribute<?>, Boolean> comparing(
        IElementAttribute::isOptional).thenComparing(
          IElementAttribute::getName));

      if (!attributes.isEmpty()) {
        description.append("<p style='margin-left:" + DEFAULT_MARGIN + "px'>");
        description.append(LNG.get("ParametersDescriptionDialog.attributes"));
        description.append("</p'>");
        description.append(getAttributes(structure, attributes, doc, 1));
      }
    }

    return description.toString();
  }

  /**
   * Gera a descrio de uma tag que gera mltiplos parmetros.
   * 
   * @param description StringBuilder onde o html est sendo gerado.
   * @param multipleFactory fbrica que sabe construir mais de um atributo.
   * @param value valor booleano que determina qual parmetro est sendo
   *        construdo.
   */
  private void getMultipleParameterDescription(StringBuilder description,
    MultipleParameterFactory multipleFactory, Boolean value) {
    ParameterDocumentation doc = registry.getParameterDocumentation(
      multipleFactory.getParameterStructure(value), locale);
    description.append("<h3 style='margin-left:" + MULTIPLE_VALUES_MARGIN
      + "px'>");
    description.append(" " + MessageFormat.format(LNG.get(
      "ParametersDescriptionDialog.attribute.structure"), multipleFactory
        .getAttribute().getName(), value.toString()));
    description.append("</h3>");

    getTitleAndDescription(description, doc);

    description.append("<p style='margin-left:" + DEFAULT_MARGIN + "px'>");
    description.append(LNG.get("ParametersDescriptionDialog.attributes"));
    description.append("</p>");

    description.append(getAttributes(multipleFactory.getParameterStructure(
      true), multipleFactory.getParameterStructure(true).getAttributes(), doc,
      1));
  }

  /**
   * A partir de uma documentao, este mtodo gera o texto html do ttulo e
   * descrio do parmetro.
   * 
   * @param description StringBuilder onde est sendo montado o html.
   * @param doc Documentao do parmetro.
   */
  private void getTitleAndDescription(StringBuilder description,
    ParameterDocumentation doc) {
    if (doc.getTitle() != null || doc.getDescription() != null) {
      description.append("</br>");
      description.append("<p style='margin-left:" + DEFAULT_MARGIN + "px'>");
    }
    if (doc.getTitle() != null) {
      description.append("<b>" + doc.getTitle() + "</b>");
    }

    if (doc.getDescription() != null) {
      description.append("<br>" + doc.getDescription());
    }
    if (doc.getTitle() != null || doc.getDescription() != null) {
      description.append("</p");
    }
  }

  /**
   * Obtm a descrio dos atributos em HTML.
   * 
   * @param structure A estrutura que contm os atributos.
   * @param attributes Os atributos.
   * @param doc Documentao do parmetro.
   * @param marginLevel Nvel de margem.
   * @return String com a descrio dos atributos.
   */
  private String getAttributes(IElementStructure<?> structure,
    List<IElementAttribute<?>> attributes, ParameterDocumentation doc,
    int marginLevel) {

    StringBuilder description = new StringBuilder();
    if (!attributes.isEmpty()) {
      description.append("<ul style='margin-left:" + ATTRIBUTE_MARGIN
        * marginLevel + "px'>");
    }

    for (IElementAttribute<?> attribute : attributes) {
      description.append("<li>");

      description.append("<b>");
      description.append(attribute.getName());
      description.append("</b>");

      if (!attribute.isOptional()) {
        description.append("*");
      }
      description.append(" ");

      description.append(" (");
      if (attribute.getType().isEnum()) {
        Object[] values = attribute.getType().getEnumConstants();
        for (int i = 0; i < values.length; i++) {
          description.append(values[i]);
          if (i < values.length - 1) {
            description.append("|");
          }
        }

      }
      else {
        description.append(attribute.getType().getSimpleName());
      }
      description.append(") ");
      if (attribute.isOptional()) {
        description.append(LNG.get("ParametersDescriptionDialog.default"));
        description.append(" <i>");
        description.append(attribute.getDefaultValue());
        description.append("</i>");
      }
      if (doc != null) {
        String attributeDescription = doc.getAttribute(attribute.getName());
        if (attributeDescription != null) {
          description.append("<br>" + attributeDescription + "<br>");
        }
      }
      description.append("</li>");

    }
    if (!attributes.isEmpty()) {
      description.append("</ul>");
    }

    List<IElementStructure<?>> children = structure.getChildElements();
    if (!children.isEmpty()) {
      description.append("<h2 style='margin-left:" + DEFAULT_MARGIN + "px'>");
      description.append(LNG.get("ParametersDescriptionDialog.child.elements"));
      description.append("</h2>");
      for (IElementStructure<?> child : children) {
        description.append(getChildDescription(child));
      }
    }

    return description.toString();
  }

  /**
   * Obtm a descrio de um elemento filho em HTML.
   * 
   * @param structure estrutura do elemento filho.
   * @return descrio de um elemento filho.
   */
  private String getChildDescription(IElementStructure<?> structure) {
    List<IElementAttribute<?>> attributes = new ArrayList<>(structure
      .getAttributes());
    attributes.sort(Comparator.<IElementAttribute<?>, Boolean> comparing(
      IElementAttribute::isOptional).thenComparing(IElementAttribute::getName));

    StringBuilder description = new StringBuilder();

    description.append("<h3 style='margin-left:" + CHILDREN_TITLE_MARGIN
      + "px'>");
    description.append(structure.getName());
    description.append("</h3>");

    ParameterDocumentation doc = registry.getParameterDocumentation(structure,
      locale);

    if (doc.getTitle() != null) {
      description.append(doc.getTitle() + " - ");
    }

    if (doc.getDescription() != null) {
      description.append(doc.getDescription());
    }

    if (!attributes.isEmpty()) {
      description.append("<p style='margin-left:" + CHILDREN_CONTENT_MARGIN
        + "px'>");
      description.append(LNG.get("ParametersDescriptionDialog.attributes"));
      description.append("</p>");
    }

    description.append(getAttributes(structure, attributes, doc, 2));
    return description.toString();
  }

  /**
   * Atualiza o contedo da rea de texto.
   * 
   * @param content contedo a ser exibido na rea de texto.
   */
  private void updateTextArea(String content) {
    StringBuilder builder = new StringBuilder("<html>");

    builder.append(content);
    builder.append("<br><br>");

    builder.append("</html>");

    textArea.setText(builder.toString());
    textArea.setCaretPosition(0);
  }

  /**
   * Cria o painel principal deste dilogo.
   */
  private void mountPanel() {
    JPanel contentPanel = new JPanel();
    textArea = new JEditorPane();
    textArea.setFont(new Font(Font.MONOSPACED, Font.PLAIN, textArea.getFont()
      .getSize()));
    textArea.setEditable(false);
    //textArea.setContentType("text/html");
    HTMLEditorKit kit = new HTMLEditorKit();
    textArea.setEditorKit(kit);
    Document doc = kit.createDefaultDocument();
    textArea.setDocument(doc);
    //    textArea.setText(
    //      "<html><p style='margin:50px'>Did you know that you can use CSS styles when displaying HTML in a Java Swing application? It's pretty cool, and it can help spice up any simple HTML you may currently be showing in a Java-based editor or viewer. In this tutorial I'll share some source code that shows how this works.</p><html>");

    JScrollPane scrollPane = new JScrollPane(textArea);
    scrollPane.setMinimumSize(new Dimension(800, 600));
    scrollPane.setPreferredSize(new Dimension(800, 600));
    scrollPane.setVerticalScrollBarPolicy(
      JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

    scrollPane.setBorder(BorderFactory.createEtchedBorder());

    contentPanel.setLayout(new GridBagLayout());
    contentPanel.add(scrollPane, new GBC(0, 0).both().insets(6, 6, 6, 6));

    this.setContentPane(contentPanel);
  }
}
