/*
 * $Author$ $Date$ $Revision: 175365 $
 */
package csbase.client.algorithms.parameters;

import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Window;
import java.util.Collections;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.SwingConstants;
import javax.swing.table.DefaultTableCellRenderer;

import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.EnumerationListParameter;
import csbase.logic.algorithms.parameters.EnumerationListParameterListener;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import tecgraf.javautils.gui.selector.ItemContainer;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Viso para o parmetro do tipo enumerao com seleo mltipla.
 * 
 * @author lmoreira
 */
public final class EnumerationListParameterView extends
  SimpleParameterView<List<EnumerationItem>> {

  /**
   * Cria uma viso em modo {@link ParameterView.Mode#CONFIGURATION} para
   * parmetro do tipo enumerao com seleo mltipla.
   * 
   * @param window NO EST SENDO UTILIZADO. Existe somente para manter
   *        compatibilidade com o WebSintesi.
   * @param parameter O parmetro (No aceita {@code null}).
   * 
   * @deprecated para manter compatibilidade com o WebSintesi
   */
  @Deprecated
  public EnumerationListParameterView(Window window,
    EnumerationListParameter parameter) {
    this(parameter, Mode.CONFIGURATION);
  }

  /**
   * Cria uma viso para parmetro do tipo enumerao com seleo mltipla.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. Pode ser
   *        {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}.
   */
  public EnumerationListParameterView(EnumerationListParameter parameter,
    Mode mode) {
    super(parameter, mode);

    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EnumerationListParameter getParameter() {
    return (EnumerationListParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    return new EnumerationListConfigurationParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    return new EnumerationListReportParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateViewContents() {
    ((IEnumerationListParameterComponent) getComponent()).updateViewContents();
  }

  /**
   * Obtm uma mensagem no mecanismo de internacionalizao.
   * 
   * @param key A chave (No aceita {@code null}).
   * 
   * @return A mensagem.
   */
  private String getString(String key) {
    return LNG.get(getClass().getName() + "." + key);
  }

  /**
   * Interface comum s diferentes vises do parmetro.
   */
  private interface IEnumerationListParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#REPORT}.
   */
  private final class EnumerationListReportParameter extends JPanel implements
    IEnumerationListParameterComponent {

    /**
     * rea de texto onde os itens de enumerao so listadas.
     */
    private JTextArea textArea;

    /**
     * Construtor.
     */
    EnumerationListReportParameter() {

      setLayout(new GridLayout());

      textArea = new JTextArea();
      textArea.setToolTipText(getParameter().getDescription());
      ComponentProperties.setProperties(textArea, Mode.REPORT, false);

      JScrollPane scroll = new JScrollPane(textArea);
      scroll.setBorder(ComponentProperties.getInstance(Mode.REPORT)
        .getBorder());
      updateViewContents();
      add(scroll, new GBC(0, 0).both());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      List<EnumerationItem> selectedItems = getParameter().getValue();
      if (selectedItems == null) {
        textArea.setRows(1);
      }
      else {
        textArea.setRows(Math.max(1, Math.min(selectedItems.size(), 6)));

        StringBuffer sb = new StringBuffer();
        for (EnumerationItem item : selectedItems) {
          sb.append(item.getLabel()).append('\n');
        }
        // remove o ltimo breakLine
        String text = sb.substring(0, sb.length() - 1);
        textArea.setText(text);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean enabled) {
      super.setEnabled(enabled);
      textArea.setEnabled(enabled);
    }
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#CONFIGURATION}.
   */
  private final class EnumerationListConfigurationParameter extends JPanel
    implements IEnumerationListParameterComponent {

    /**
     * O container.
     */
    private ContainerSelection<EnumerationItem> containerSelection;

    /**
     * Construtor.
     */
    EnumerationListConfigurationParameter() {
      setLayout(new GridBagLayout());

      this.containerSelection = new ContainerSelection<EnumerationItem>(
        new MyObjectTableProvider(getString("available_items")),
        new MyObjectTableProvider(getString("selected_items")), true, false,
        true, true, SwingConstants.CENTER, false);
      this.containerSelection.loadItems(getParameter().getItems());
      this.containerSelection.addContainerSelectionListener(
        new ContainerSelectionListener() {
          @Override
          public void containerChanged() {
            updateModel();
          }
        });

      ItemContainer<EnumerationItem> availableContainer =
        this.containerSelection.getAvailableContainer();
      SortableTable availableTable = availableContainer.getTable();
      availableTable.setDefaultRenderer(Object.class,
        new DefaultTableCellRenderer() {
          @Override
          public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus, int row,
            int column) {
            EnumerationItem item = containerSelection.getAvailableItems().get(
              row);
            if (item.getDescription() != null) {
              setToolTipText(item.getDescription());
            }
            else {
              setToolTipText(getParameter().getDescription());
            }
            return super.getTableCellRendererComponent(table, value, isSelected,
              hasFocus, row, column);
          }
        });
      availableTable.setToolTipText(getParameter().getDescription());

      ItemContainer<EnumerationItem> selectedContainer = this.containerSelection
        .getSelectedContainer();
      SortableTable selectTable = selectedContainer.getTable();
      selectTable.setDefaultRenderer(Object.class,
        new DefaultTableCellRenderer() {

          @Override
          public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus, int row,
            int column) {
            EnumerationItem item = containerSelection.getSelectedItems().get(
              row);
            if (item.getDescription() != null) {
              setToolTipText(item.getDescription());
            }
            else {
              setToolTipText(getParameter().getDescription());
            }
            return super.getTableCellRendererComponent(table, value, isSelected,
              hasFocus, row, column);
          }
        });
      selectTable.setToolTipText(getParameter().getDescription());

      getParameter().addEnumerationListParameterListener(
        new EnumerationListParameterListener() {
          @Override
          public void itemsWereChanged(
            final EnumerationListParameter parameter) {
            containerSelection.loadItems(parameter.getItems());
          }
        });

      updateViewContents();
      add(containerSelection.getPanel(), new GBC(0, 0).both());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      List<EnumerationItem> selectedItems = getParameter().getValue();
      if (selectedItems == null) {
        selectedItems = Collections.emptyList();
      }
      this.containerSelection.loadItems(getParameter().getItems(),
        selectedItems);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      this.containerSelection.setEnabled(isEnabled);
    }

    /**
     * Reflete as alteraes da viso para o modelo (o parmetro propriamente
     * dito).
     */
    private void updateModel() {
      List<EnumerationItem> selectedItems = this.containerSelection
        .getSelectedItems();
      if (selectedItems.isEmpty()) {
        getParameter().setValue(null);
      }
      else {
        getParameter().setValue(selectedItems);
      }
    }
  }

  /**
   * Utilizada para fornecer dados sobre o tem de enumerao para as tabelas do
   * container de seleo.
   */
  private final class MyObjectTableProvider extends DefaultObjectTableProvider {
    /**
     * O nome da coluna.
     */
    private String columnName;

    /**
     * @param columnName O nome da coluna (No aceita {@code null}).
     */
    MyObjectTableProvider(String columnName) {
      this.columnName = columnName;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getColumnNames() {
      return new String[] { this.columnName };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getColumnClasses() {
      return new Class[] { String.class };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object[] getCellValues(Object row) {
      EnumerationItem item = (EnumerationItem) row;
      return new Object[] { item.getLabel() };
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return true;
  }

}
