package csbase.client.applications.diskmonitor.panels;

import java.awt.BorderLayout;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.labels.StandardPieSectionLabelGenerator;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.title.LegendTitle;
import org.jfree.chart.title.TextTitle;
import org.jfree.chart.title.Title;
import org.jfree.data.general.DefaultPieDataset;
import org.jfree.util.Rotation;

import tecgraf.javautils.gui.BorderUtil;
import csbase.client.applications.diskmonitor.DiskMonitor;
import csbase.client.applications.diskmonitor.panels.core.DiskMonitorPanel;
import csbase.client.remote.srvproxies.DiskUsageProxy;
import csbase.logic.Utilities;
import csbase.logic.diskusageservice.DiskOccupation;

/**
 * Painel de grfico de pizza.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DiskMonitorPieChartPanel extends DiskMonitorPanel {

  /**
   * O grfico JFreeChart
   */
  final private JFreeChart jChart;

  /**
   * Legenda interna
   */
  private LegendTitle internalLegend = null;

  /**
   * Construtor
   * 
   * @param application aplicativo.
   */
  public DiskMonitorPieChartPanel(DiskMonitor application) {
    super(application);

    jChart = ChartFactory.createPieChart("", null, true, false, false);
    internalLegend = jChart.getLegend();
    final ChartPanel chartPanel = new ChartPanel(jChart);
    chartPanel.setPopupMenu(null);
    BorderUtil.setLowBorder(chartPanel);

    setLayout(new BorderLayout());
    add(chartPanel, BorderLayout.CENTER);
  }

  /**
   * Atualizao do grfico
   * 
   * @param id identificador da rea de disco
   * @param occupation ocupao.
   */
  final public void updateChart(final String id, final DiskOccupation occupation) {
    final PiePlot pieplot = (PiePlot) jChart.getPlot();
    double free;
    double used;
    if (occupation == null || !occupation.isValid()) {
      resetChart(id);
      return;
    }

    free = occupation.getFreeSpaceMb();
    used = occupation.getUsedSpaceMb();
    if ((free < 0) || (used < 0)) {
      used = 0;
      free = 0;
    }

    free /= 1024.;
    used /= 1024.;

    updateTitles(id);

    pieplot
      .setNoDataMessage(getString("DiskMonitorPieChartPanel.failed.error"));
    final String freeStr = getString("DiskMonitorPieChartPanel.free.legend");
    final String usedStr = getString("DiskMonitorPieChartPanel.used.legend");

    final DefaultPieDataset dataset = new DefaultPieDataset();
    dataset.setValue(freeStr, free);
    dataset.setValue(usedStr, used);

    final StandardPieSectionLabelGenerator labelGen =
      new StandardPieSectionLabelGenerator("{2}\n{1} gb", Locale.ROOT);
    pieplot.setLabelGenerator(labelGen);

    final StandardPieSectionLabelGenerator legendGen =
      new StandardPieSectionLabelGenerator("{0}");
    pieplot.setLegendLabelGenerator(legendGen);

    pieplot.setSectionPaint(freeStr, new Color(150, 255, 150));
    pieplot.setSectionPaint(usedStr, new Color(255, 150, 150));
    pieplot.setDirection(Rotation.CLOCKWISE);
    pieplot.setBackgroundPaint(null);
    pieplot.setOutlinePaint(null);
    pieplot.setCircular(true);
    pieplot.setDataset(dataset);
  }

  /**
   * @param id
   */
  private void updateTitles(final String id) {
    String title = DiskUsageProxy.translateAreaId(id);
    if (title == null) {
      title = "---";
    }
    jChart.setTitle(title);
    final List<Title> subtitles = new ArrayList<Title>();
    final Date now = new Date();
    final long time = now.getTime();
    final String timeText = Utilities.getFormattedDate(time);
    subtitles.add(new TextTitle(timeText));
    jChart.setSubtitles(subtitles);
    jChart.addLegend(internalLegend);
  }

  /**
   * Reset do grfico
   * 
   * @param id identificador da rea.
   */
  final public void resetChart(final String id) {
    final PiePlot pieplot = (PiePlot) jChart.getPlot();
    pieplot.setDataset(null);
    updateTitles(id);
  }
}
