/*
 * $Id$
 */

package csbase.client.applications.projectsmanager.panels;

import java.awt.GridBagLayout;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.proxy.RetrieveUsersTask;
import csbase.client.util.user.UserDialog;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.UserOutline;

/**
 * Painel que apresenta informaes sobre a seleo de uma nica linha na tabela
 * de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
class SingleSelectionPanel extends AbstractProjectManagerInfoPanel {

  /**
   * Estrutura de dados do projeto.
   */
  private ProjectsManagerData projectsManagerData;

  /**
   * Nome do projeto.
   */
  private JTextField projectNameTextField;

  /**
   * Dono do projeto.
   */
  private JTextField projectOwnerTextField;

  /**
   * Descrio do projeto.
   */
  private JTextField projectDescriptionTextField;

  /**
   * Modo de acesso permitido ao projeto (readonly, read/write).
   */
  private JTextField accessModeTextField;

  /**
   * Boto para exibio de usurios compartilhados.
   */
  private JButton sharedUsersButton;

  /**
   * Construtor.
   * 
   * @param projectsManager A aplicao.
   */
  SingleSelectionPanel(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void initComponents() {
    projectNameTextField = new JTextField(20);
    projectOwnerTextField = new JTextField(20);
    projectDescriptionTextField = new JTextField(80);
    accessModeTextField = new JTextField(20);
    projectNameTextField.setEditable(false);
    projectOwnerTextField.setEditable(false);
    projectDescriptionTextField.setEditable(false);
    accessModeTextField.setEditable(false);

    sharedUsersButton = new JButton();
    sharedUsersButton
      .setToolTipText(getString("SingleSelectionPanel.shared.users.button.tooltip"));

    final ProjectsManager projectsManager = getProjectsManager();
    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    sharedUsersButton.setAction(new ProjectsManagerAction(projectsManager) {

      @Override
      public void actionDone() throws Exception {

        final Set<Object> usersROIds = projectsManagerData.getUsersRO();
        final Set<Object> usersRWIds = projectsManagerData.getUsersRW();

        final Set<Object> users = new TreeSet<Object>();
        users.addAll(usersROIds);
        users.addAll(usersRWIds);

        final RetrieveUsersTask rut =
          new RetrieveUsersTask(projectsManager, users);
        rut.execute(frame, getString("SingleSelectionPanel.title"),
          getString("SingleSelectionPanel.message"));

        if (rut.wasCancelled()) {
          final String err =
            getString("SingleSelectionPanel.cancelled.message");
          rut.showError(err);
          return;
        }

        if (!rut.getStatus()) {
          final Exception exception = rut.getError();
          throw exception;
        }

        final List<UserOutline> userList = rut.getResult();

        if (userList == null) {
          StandardDialogs.showInfoDialog(projectsManager.getApplicationFrame(),
            getString("SingleSelectionPanel.shared.users.read.error.title"),
            getString("SingleSelectionPanel.shared.users.read.error.message"));
          return;
        }

        UserDialog.showProjectUsersDialog(
          projectsManager.getApplicationFrame(), userList, usersROIds,
          usersRWIds);
      }
    });

    sharedUsersButton.setIcon(ApplicationImages.ICON_FIND_USER_16);
    GUIUtils.trimImageButton(sharedUsersButton);
    sharedUsersButton.setEnabled(false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void buildPanel() {
    final JLabel projectNameLabel =
      new JLabel(getString("SingleSelectionPanel.project.name.label"));
    final JLabel projectOwnerLabel =
      new JLabel(getString("SingleSelectionPanel.project.owner.label"));
    final JLabel projectDescriptionLabel =
      new JLabel(getString("SingleSelectionPanel.project.description.label"));
    final JLabel accessModeLabel =
      new JLabel(getString("SingleSelectionPanel.project.access.mode.label"));

    final int T = 12;
    final int TI = 6;
    final int L = 12;
    final int R = 11;
    this.setLayout(new GridBagLayout());

    add(projectNameLabel, new GBC(0, 0).west().insets(T, L, 0, R));
    add(projectNameTextField, new GBC(1, 0).west().insets(T, L, 0, R)
      .horizontal());

    add(projectOwnerLabel, new GBC(2, 0).west().insets(TI, L, 0, R));
    add(projectOwnerTextField, new GBC(3, 0).west().insets(TI, L, 0, R)
      .horizontal());

    add(accessModeLabel, new GBC(0, 1).west().insets(TI, L, 0, R));
    add(accessModeTextField, new GBC(1, 1).west().insets(TI, L, 0, R)
      .horizontal());

    add(sharedUsersButton, new GBC(2, 1).west().insets(TI, L, 0, R).width(1));

    add(projectDescriptionLabel, new GBC(0, 2).west().insets(TI, L, 0, R));
    add(projectDescriptionTextField, new GBC(1, 2).west().insets(TI, L, 0, R)
      .width(3).horizontal());

    BorderUtil.setTitledBorder(this,
      getString("SingleSelectionPanel.project.info.title"));

    validate();
    repaint();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void updateFields() {
    final String projectName =
      String.valueOf(projectsManagerData.getProjectName());
    projectNameTextField.setText(projectName);
    final String description = projectsManagerData.getProjectDescription();
    projectDescriptionTextField.setText(description);
    final Object ownerId = projectsManagerData.getOwnerId();
    final String ownerName = projectsManagerData.getOwnerName();
    projectOwnerTextField.setText(ownerId + " (" + ownerName + ")");
    final String accessModeMessage = getAccessModeMessage();
    accessModeTextField.setText(accessModeMessage);
    if (projectsManagerData.getScope() == ProjectsManagerScope.SHARED) {
      sharedUsersButton.setEnabled(true);
    }
    else {
      sharedUsersButton.setEnabled(false);
    }
  }

  /**
   * Limpa os campos do painel.
   */
  @Override
  public void clearFields() {
    projectNameTextField.setText(null);
    projectDescriptionTextField.setText(null);
    projectOwnerTextField.setText(null);
    accessModeTextField.setText(null);
  }

  /**
   * Gera mensagem a ser colocada no TextField referente ao modo de acesso ao
   * projeto.
   * 
   * @return Mensagem
   */
  private String getAccessModeMessage() {
    final ProjectsManagerScope scope = projectsManagerData.getScope();
    if (scope == null) {
      return getString("SingleSelectionPanel.invalid.access.mode.message");
    }

    switch (scope) {
      case PRIVATE:
      case NOT_APPLICABLE:
      case SHARED:
        return scope.getTitle();

      case PUBLIC:
        final StringBuilder sb = new StringBuilder(scope.getTitle());
        sb.append(" (");
        final SharingType sharingType =
          this.projectsManagerData.getSharingType();
        if (sharingType == SharingType.ALL_RO) {
          sb.append(getString("SingleSelectionPanel.readonly.access.mode.message"));
        }
        else if (sharingType == SharingType.ALL_RW) {
          sb.append(getString("SingleSelectionPanel.read.write.access.mode.message"));
        }
        else {
          throw new RuntimeException("tipo inconsistente de compartilhamento: "
            + sharingType.name());
        }
        sb.append(')');
        return sb.toString();

      default:
        throw new RuntimeException("escopo invlido: " + scope.name());
    }
  }

  /**
   * Ajuste do projeto (dado) a ser detalhado.
   * 
   * @param projectsManagerData dado
   */
  public void setProjectsManagerData(
    final ProjectsManagerData projectsManagerData) {
    this.projectsManagerData = projectsManagerData;
    updateFields();
  }

}
