package csbase.client.csdk.v2.core;

import java.awt.Window;
import java.nio.charset.Charset;
import java.util.Locale;
import java.util.Map;

import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.util.StandardErrorDialogs;
import csdk.v2.api.application.IApplicationContext;
import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.api.core.IContext;
import tecgraf.javautils.core.lng.LNG;

/**
 * Implementao da interface das aplicaes com o ambiente CSBase, incluindo o
 * acesso aos contextos disponveis.
 */
public class CSDKEnvironment implements ICSDKEnvironment {

  /**
   * Contextos da aplicao.
   */
  private final Map<Class<? extends IContext>, IContext> contexts;

  /**
   * Construtor.
   * 
   * @param contexts o mapa de contextos da aplicao.
   */
  public CSDKEnvironment(Map<Class<? extends IContext>, IContext> contexts) {
    this.contexts = contexts;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void finishApplication() {
    final ApplicationManager manager = ApplicationManager.getInstance();
    IApplicationContext appContext =
      (IApplicationContext) contexts.get(IApplicationContext.class);
    if (appContext != null) {
      ApplicationType app =
        manager.getApplicationInstance(appContext.getInstanceId());
      if (app != null) {
        app.closeApplication();
      }
    }
  }

  /**
   * Obtm o contexto da aplicao a partir da classe que o implementa.
   * 
   * @param contextClass a classe do contexto.
   * @return o contexto.
   */
  @Override
  public <T extends IContext> T getContext(Class<T> contextClass) {
    if (contextClass == null) {
      return null;
    }
    IContext context = this.contexts.get(contextClass);
    if (context != null) {
      try {
        T t = contextClass.cast(context);
        if (!context.isActive()) {
          t.onContextActivate();
        }
        return t;
      }
      catch (Exception e) {
        handleException(e, null);
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void handleException(Exception error, Window parent) {
    IApplicationContext appContext = (IApplicationContext) contexts.get(
      IApplicationContext.class);
    String title = null;
    if (appContext != null) {
      title = appContext.getApplicationName();
    }
    StandardErrorDialogs.showExceptionDialog(parent, title, error);
  }

  /**
   * Limpa os contextos.
   */
  public void cleanup() {
    for (IContext context : contexts.values()) {
      if (context.isActive()) {
        context.onContextDeactivate();
      }
    }
    contexts.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Locale getLocale() {
    return LNG.getLocale();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getServerCharset() {
    return Client.getInstance().getSystemDefaultCharset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getCharset() {
    return Client.getInstance().getClientHostCharset();
  }

}
