/**
 * $Id: UsersPermissionsTableModel.java 179638 2017-04-24 20:31:44Z pedrorocha $
 */
package csbase.client.util.user;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.swing.table.DefaultTableModel;

import tecgraf.javautils.core.lng.LNG;
import csbase.logic.UserOutline;

/**
 * Modelo para a tabela de permisses de acesso dos usurios a um determinado
 * projeto.
 * 
 * @author Tecgraf
 */
class UsersPermissionsTableModel extends DefaultTableModel {

  /**
   * Coluna com os logins.
   */
  public static final int COL_LOGIN = 0;
  /**
   * Coluna com os nomes.
   */
  public static final int COL_NAME = 1;
  /**
   * Coluna com os checkboxes RW.
   */
  public static final int COL_RW = 2;
  /**
   * Coluna com os checkboxes RO.
   */
  public static final int COL_RO = 3;

  /**
   * Tipos possveis de acesso para cada usurio.
   */
  enum AccesType {
    /**
     * Usurio no tem acesso ao projeto.
     */
    NONE,
    /**
     * Usurio tem acesso RO ao projeto.
     */
    RO,
    /**
     * Usurio tem acesso RW ao projeto.
     */
    RW,
  }

  /**
   * Lista com as permisses de acesso para cada usurio. Tem estreita relao
   * com a tabela que contm todos os usurios do sistema: cada item desta lista
   * representa a permisso do usurio de mesmo ndice na outra lista.
   * 
   * @see #allUsers
   */
  private List<AccesType> permissions;
  /**
   * Nmero de usurios. Usado em {@link #getRowCount()}.
   */
  private int numUsers;
  /**
   * Lista com todos os usurios do sistema.
   */
  private final List<UserOutline> allUsers;
  /**
   * Nomes das colunas da tabela.
   */
  private static final String[] COLUMN_NAMES = {
      LNG.get("UsersPermissionsTableModel.header.login"),
      LNG.get("UsersPermissionsTableModel.header.name"),
      LNG.get("UsersPermissionsTableModel.header.rw"),
      LNG.get("UsersPermissionsTableModel.header.ro") };
  /**
   * Tipos das colunas da tabela.
   */
  private static final Class<?>[] COLUMN_CLASSES = { String.class,
      String.class, Boolean.class, Boolean.class };

  /** Indica se as colunas de leitura e leitur4a/escrita devem estar habilitadas */
  private boolean enableRORWColumn;

  /**
   * Construtor. Preenche a lista de permisses baseado nas informaes das
   * outras listas.
   * 
   * @param allUsers - lista com todos os usurios do sistema
   * @param usersRO - lista com os usurios que tm acesso RO ao projeto
   * @param usersRW - lista com os usurios que tm acesso RW ao projeto
   */
  UsersPermissionsTableModel(List<UserOutline> allUsers, Set<Object> usersRO,
    Set<Object> usersRW) {

    if (allUsers == null) {
      this.allUsers = null;
      return;
    }
    /*
     * criamos uma cpia para nos proteger de mudanas externas
     */
    this.allUsers = Collections.unmodifiableList(allUsers);
    permissions = new ArrayList<AccesType>();
    /*
     * preenche a lista de permisses a partir das listas de permisses
     */
    for (UserOutline user : allUsers) {
      Object userID = user.getId();
      if (usersRO != null && usersRO.contains(userID)) {
        permissions.add(AccesType.RO);
      }
      else if (usersRW != null && usersRW.contains(userID)) {
        permissions.add(AccesType.RW);
      }
      else {
        permissions.add(AccesType.NONE);
      }
    }
    numUsers = allUsers.size();
    enableRORWColumn = true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int columnIndex) {
    return COLUMN_CLASSES[columnIndex];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int column) {
    return COLUMN_NAMES[column];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return COLUMN_NAMES.length;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    return numUsers;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {
    switch (columnIndex) {
      case COL_LOGIN:
        return allUsers.get(rowIndex).getLogin();

      case COL_NAME:
        return allUsers.get(rowIndex).getName();

      case COL_RW:
        return Boolean.valueOf(permissions.get(rowIndex) == AccesType.RW);

      case COL_RO:
        return Boolean.valueOf(permissions.get(rowIndex) == AccesType.RO);

      default:
        throw new RuntimeException("coluna invlida: " + columnIndex);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex) {
    if (columnIndex < 2) {
      return false;
    }
    else {
      return enableRORWColumn;
    }
  }

  /**
   * Habilita ou desabilita as colunas de RW e RO
   * 
   * @param enable
   */
  public void enableRORWColumn(boolean enable) {
    this.enableRORWColumn = enable;
    fireTableStructureChanged();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAt(Object value, int rowIndex, int columnIndex) {
    AccesType accessType;
    AccesType oldAccessType = permissions.get(rowIndex);
    switch (columnIndex) {
      case COL_RW:
        /*
         * acesso RW. Se o usurio j possua acesso RW, age como toggle (remove
         * a permisso de acesso)
         */
        accessType =
          oldAccessType == AccesType.RW ? AccesType.NONE : AccesType.RW;
        break;

      case COL_RO:
        /*
         * acesso RO. Se o usurio j possua acesso RO, age como toggle (remove
         * a permisso de acesso)
         */
        accessType =
          oldAccessType == AccesType.RO ? AccesType.NONE : AccesType.RO;
        break;

      default:
        throw new RuntimeException("coluna invlida: " + columnIndex);
    }
    /*
     * OBS.: definir um tipo automaticamente far com que o checkbox
     * correspondente ao outro seja desmarcado
     */
    permissions.set(rowIndex, accessType);
    fireTableRowsUpdated(rowIndex, rowIndex);
  }

  /**
   * Atribui read-only true ou false (de acordo com parmetro) para a linha
   * determinada
   * 
   * @param isReadOnly
   * @param rowIndex
   */
  public void setReadOnlyFor(boolean isReadOnly, int rowIndex) {
    permissions.set(rowIndex, isReadOnly ? AccesType.RO : AccesType.NONE);
    fireTableRowsUpdated(rowIndex, rowIndex);
  }

  /**
   * Preenche os conjuntos de usurios com acesso RO e com acesso RW definidos
   * na tabela.
   * 
   * @param usersRO - conjunto de usurios com acesso RO ao projeto, a ser
   *        preenchido pelo mtodo
   * @param usersRW - conjunto de usurios com acesso RW ao projeto, a ser
   *        preenchido pelo mtodo
   */
  public void fillUserSets(Set<Object> usersRO, Set<Object> usersRW) {
    for (int i = 0; i < allUsers.size(); i++) {
      switch (permissions.get(i)) {
        case RO:
          usersRO.add(allUsers.get(i).getId());
          break;

        case RW:
          usersRW.add(allUsers.get(i).getId());
          break;

        default:
          // Nenhuma lista  preenchida.
      }
    }
  }
}
