package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.applications.algorithmsmanager.DataPropertiesListener;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.versiontree.PropertiesPanel;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTreeNode;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.ClientUtilities;
import csbase.logic.algorithms.AlgorithmProperty;
import csbase.logic.algorithms.AlgorithmVersionInfo;

/**
 * Essa classe implementa um painel com as informaes do nmero da verso
 * selecionada e suas propriedades, que podem ser editadas.
 * 
 */
public class VersionPanel extends
  AbstractSimpleApplicationPanel<AlgorithmsManager> {
  /** Nome dos atributos */
  private List<AlgorithmProperty> versionProperties;

  /** Painel de propriedades da verso */
  private PropertiesPanel propertiesPanel;

  /** Nmero da verso exibida */
  private JTextField versionField;

  /** Informaes sobre o algoritmo ao qual pertence a verso */
  protected AlgorithmListItem algoItem;

  /** Listeners de mudanas nas propridades de verso */
  private List<DataPropertiesListener> versionListeners;

  /**
   * Cria um dilogo de verso.
   * 
   * @param application referncia para a aplicao
   * @param algoItem informaes sobre o algoritmo dentro do qual a verso ser
   *        vai ser criada
   */
  public VersionPanel(AlgorithmsManager application, AlgorithmListItem algoItem) {
    super(application);
    this.algoItem = algoItem;
    this.versionListeners = new Vector<>();
    buildPanel();
  }

  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());
    setBorder(BorderFactory.createEmptyBorder());

    JPanel mainInfoPanel = getMainInfoPanel();
    propertiesPanel = getVersionPropertiesPanel();

    add(mainInfoPanel, new GBC(0, 0).horizontal().west().insets(0, 0, 0, 0));
    if (versionProperties.size() > 0) {
      add(propertiesPanel.getPropertiesPanel(), new GBC(0, 1).horizontal()
        .west().insets(0, 0, 0, 0));
    }
  }

  /**
   * Notifica os listeners de que ocorreram mudanas na verso do algoritmo.
   * 
   * @param wasModified se true, alguma propriedade foi alterada, caso
   *        contrrio, todos os valores das propriedades esto iguais os valores
   *        originais
   */
  protected void notifyListeners(boolean wasModified) {
    for (DataPropertiesListener versionListener : versionListeners) {
      versionListener.propertiesChanged(wasModified);
    }
  }

  /**
   * Adiciona um listener de mudanas em uma verso do algoritmo
   * 
   * @param listener listener de mudanas em uma verso do algoritmo
   */
  public void addVersionListener(DataPropertiesListener listener) {
    versionListeners.add(listener);
  }

  /**
   * Cria o painel com as propriedades da verso.
   * 
   * @return o painel com as propriedades da verso
   */
  private PropertiesPanel getVersionPropertiesPanel() {
    versionProperties = AlgorithmManagementProxy
      .getAlgorithmVersionProperties(getWindow());

    if (propertiesPanel == null) {
      propertiesPanel = new PropertiesPanel(versionProperties,
              new Hashtable<>(), getApplication().getString(
          "VersionPanel.label.properties"), getWindow(), true);
      propertiesPanel.addPropertiesListener(wasModified -> notifyListeners(wasModified));
    }
    return propertiesPanel;
  }

  /**
   * Obtm o painel principal com as informaes da verso.
   * 
   * @return o painel criado
   */
  private JPanel getMainInfoPanel() {
    JPanel mainInfoPanel = new JPanel(new GridBagLayout());

    JLabel versionLabel = new JLabel(getApplication().getString(
      "VersionPanel.label.version_id"));

    versionField = new JTextField(11);
    versionField.setEditable(false);
    versionField.setBorder(null);
    mainInfoPanel.add(versionLabel);

    mainInfoPanel.add(versionLabel, new GBC(0, 0).none().west().insets(5, 5, 5,
      5));
    mainInfoPanel.add(versionField, new GBC(1, 0).none().west().insets(5, 5, 5,
      5));
    mainInfoPanel.add(buildEmptyPanel(), new GBC(2, 0).both()
      .insets(5, 5, 5, 5));

    return mainInfoPanel;
  }

  /**
   * Cosntri um painel vazio.
   * 
   * @return o painel vazio
   */
  private Component buildEmptyPanel() {
    return new JPanel();
  }

  /**
   * Obtm a janela da aplicao Gerenciador de Algoritmos.
   * 
   * @return a janela da aplcicao
   */
  private Window getWindow() {
    return getApplication().getApplicationFrame();
  }

  /**
   * Valida um campo de texto numrico para identificao de verso (major,
   * minor, patch).
   * 
   * @param field referncia para o campo de texto
   * @param errMsg mensagem de erro para o campo correspondente
   * 
   * @return valor numrico correspondente ou -1 em caso de erro
   */
  public int validateVersionNumberField(JTextField field, String errMsg) {
    try {
      int value = Integer.parseInt(field.getText());
      if ((value >= 0) && (value <= 999)) {
        return value;
      }
    }
    catch (NumberFormatException e) {
    }
    ClientUtilities.showErrorByTextField(
      getApplication().getApplicationFrame(), errMsg, null, field);
    return -1;
  }

  /**
   * Obtm as propriedades da verso.
   * 
   * @return um mapa com as propriedades da verso
   */
  public Hashtable<String, String> getProperties() {
    return propertiesPanel.getProperties();
  }

  /**
   * Valida as propriedades da verso.
   * 
   * @return retorna true se so vlidas, caso contrrio, retorna false
   */
  public boolean validateProperties() {
    return propertiesPanel.validateProperties();
  }

  /**
   * Inicializa um n de verso selecionado.
   * 
   * @param algoItem item de algoritmo modificado
   * @param versionNode n de verso
   */
  public void initialize(AlgorithmListItem algoItem, VersionTreeNode versionNode) {
    this.algoItem = algoItem;
    AlgorithmVersionInfo version = null;
    if (versionNode != null) {
      // um n de verso est selecionado, seno  null
      version = (AlgorithmVersionInfo) versionNode.getUserObject();
      enablePropertiesEdition(true);
    }
    else {
      enableEdition(false);
    }
    String versionValue = (versionNode != null) ? String.valueOf(version
      .toString()) : "";
    Hashtable<String, String> properties = (versionNode != null) ? (Hashtable<String, String>) version
      .getPropertyValues()
      : new Hashtable<>();
    versionField.setText(versionValue);
    propertiesPanel.initializeFields(properties);
  }

  /**
   * Habilita ou desabilita a edio da verso.
   * 
   * @param state se true, habilita a edio, caso contrrio, desabilita a
   *        edio
   */
  public void enableEdition(boolean state) {
    enablePropertiesEdition(state);
  }

  /**
   * Habilita ou desabilita das propriedades da verso.
   * 
   * @param state se true, habilita a edio, caso contrrio, desabilita a
   *        edio
   */
  public void enablePropertiesEdition(boolean state) {
    propertiesPanel.enableEdition(state);
  }

  /**
   * Indica se houve alterao nas informaes de uma verso do algoritmo.
   * 
   * @return true se houver alterao nas informaes de uma verso do
   *         algoritmo, ou false caso as informaes sejam as originais
   */
  public boolean wasModified() {
    if (propertiesPanel.wasModified()) {
      return true;
    }
    return false;
  }

}
