/*
 * $Id: ExecutorFrame.java 175310 2016-08-03 14:38:03Z fpina $
 */
package csbase.client.applications.executor;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.swing.JButton;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.MenuButton;
import tecgraf.javautils.gui.MenuButton.PopupPosition;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.algorithms.AlgorithmChooserEvent;
import csbase.client.algorithms.AlgorithmChooserListener;
import csbase.client.algorithms.AlgorithmChooserPanel;
import csbase.client.algorithms.AlgorithmConfiguratorFactory;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.flowapplication.FlowApplication;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.ProjectTreePath;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.sga.CommandExecutionDialog;
import csbase.client.util.sga.CommandRequestedListener;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.serializer.AlgorithmConfigurationSerializerFactory;
import csbase.logic.algorithms.serializer.IAlgorithmConfigurationSerializer;
import csbase.logic.algorithms.serializer.exception.AlgorithmConfigurationSerializerParameterNotFoundException;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * A classe <code>ExecutorFrame</code> implementa o dilogo de execuo de
 * algoritmos.
 */
public final class ExecutorFrame extends ApplicationProject {
  /**
   * Painel seletor de verses de algoritmos
   */
  private AlgorithmChooserPanel algorithmChooserPanel;

  /**
   * Viso do configurador de algoritmos
   */
  private AlgorithmConfiguratorView algorithmConfiguratorView;

  /**
   * Configurador de algoritmos
   */
  private AlgorithmConfigurator algorithmConfigurator;

  /**
   * Algoritmos carregados na inicializao do executor
   */
  private SortedSet<AlgorithmInfo> algorithms = new TreeSet<AlgorithmInfo>();

  /**
   * Indica se nessa execuo deve-se realizar a carga inicial de algoritmos
   */
  private boolean mustLoadAlgorithms;

  /**
   * Painel de botes
   */
  private JPanel buttonPanel;

  /**
   * Boto Executar
   */
  private JButton executeButton;

  /**
   * Boto Informar Parmetros
   */
  private JButton paramButton;

  /**
   * Boto Fechar
   */
  private JButton closeButton;

  /**
   * Listeners interessados nos comandos solicitados com o dilogo de execuo
   * de comandos
   */
  private final List<CommandRequestedListener> listeners;

  /**
   * Indica se deve informar ao usurio que o comando foi submetido ao servidor,
   * e se deseja monitor-lo
   */
  private boolean mustShowCommandRequested;

  /**
   * Cria a janela executora de algoritmos.
   *
   * @param id O identificador da aplicao.
   */
  public ExecutorFrame(final String id) {
    super(id);
    buildFrame();
    listeners = new LinkedList<CommandRequestedListener>();
    mustShowCommandRequested = true;
    mustLoadAlgorithms = true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    // No faz nada.
  }

  /**
   * (non-Javadoc)
   *
   * @see csbase.client.applicationmanager.ApplicationType#sendMessage(java.lang.String,
   *      java.lang.Object, java.lang.String)
   */
  @Override
  public void sendMessage(final String name, final Object value, String senderId) {
    if (name == null) {
      throw new IllegalArgumentException("O parmetro name no pode ser nulo.");
    }
    if (value == null) {
      throw new IllegalArgumentException("O parmetro value no pode ser nulo.");
    }
    if (name.equals(PROJECT_FILE_MESSAGE)) {
      final ClientProjectFile clientProjectFile = (ClientProjectFile) value;
      if (clientProjectFile.getType().equals("PARAMETERS")) {

        final AlgorithmConfigurator configurator =
          readParameterFile(clientProjectFile);
        if (configurator == null) {
          return;
        }
        final String algorithmName = configurator.getAlgorithmName();
        final AlgorithmVersionId versionId =
          configurator.getAlgorithmVersionId();
        if ((algorithmName != null) && (versionId != null)) {
          if (!createConfigurator(algorithmName, versionId)) {
            closeApplication();
            return;
          }
        }
        else {
          final String message =
            MessageFormat.format(getString("error.missing_information"),
              FileUtils.joinPath('/', clientProjectFile.getPath()));
          showError(message);
          closeApplication();
          return;
        }
        this.algorithmConfiguratorView.setCurrentPath(new ProjectTreePath(
          clientProjectFile));
        this.algorithmConfiguratorView.launch();
        return;
      }
      final String path = FileUtils.joinPath('/', clientProjectFile.getPath());
      final String type = clientProjectFile.getType();
      final FileURLValue inputFile = new FileURLValue(path, type);
      this.algorithmConfigurator.setDefaultInputFile(inputFile);
      return;
    }
  }

  /**
   * Exibe um configurador para um algoritmo na sua verso mais recente.
   *
   * @param algorithmName O nome do algoritmo.
   * @param algorithmVersionId A verso do algoritmo.
   * @param algorithmParameters Os parmetros do algoritmo.
   *
   * @throws ApplicationException Caso ocorra erro durante o procedimento.
   */
  public void showConfigurator(final String algorithmName,
    final AlgorithmVersionId algorithmVersionId,
    final Map<String, String> algorithmParameters) throws ApplicationException {
    try {
      if (this.algorithmChooserPanel.selectAlgorithm(algorithmName)) {
        if (this.algorithmChooserPanel.selectVersion(algorithmVersionId)) {
          if (algorithmParameters != null) {
            this.algorithmConfigurator
            .setParameterValuesByName(algorithmParameters);
          }
          if (this.algorithmConfigurator != null) {
            this.algorithmConfiguratorView.launch();
          }
        }
      }
    }
    catch (final ParseException e) {
      throw new ApplicationException(e);
    }
    catch (ParameterNotFoundException e) {
      throw new ApplicationException(e);
    }
  }

  /**
   * Exibe um configurador para um algoritmo na sua verso mais recente.
   *
   * @param algorithmName O nome do algoritmo.
   * @param algorithmParameters Os parmetros do algoritmo.
   *
   * @throws ApplicationException Caso ocorra erro durante o procedimento.
   */
  public void showConfigurator(final String algorithmName,
    final Map<String, String> algorithmParameters) throws ApplicationException {
    if (this.algorithmChooserPanel.selectAlgorithm(algorithmName)) {
      final AlgorithmInfo info =
        this.algorithmChooserPanel.getSelectedAlgorithmInfo();
      final AlgorithmVersionId algorithmVersionId =
        info.getLastVersion().getId();

      showConfigurator(algorithmName, algorithmVersionId, algorithmParameters);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startApplication() throws ApplicationException {
    if (this.algorithmChooserPanel != null) {
      this.algorithmChooserPanel.setAlgorithmSet(this.getAlgorithmInfoSet());
      if (!this.algorithmChooserPanel.hasAlgorithms() && mustLoadAlgorithms) {
        final Window window = DesktopFrame.getInstance().getDesktopFrame();
        StandardDialogs.showWarningDialog(window,
          getString("application.title"), getString("error.no_algorithms"));
        return;
      }
    }
    super.startApplication();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userCanKillApplication() {
    return true;
  }

  /**
   * Cria a janela de execuo de algoritmos
   */
  private void buildFrame() {
    createAlgorithmChooserPanel();
    createButtonsPanel();
    populateFrame();
  }

  /**
   * Preenche a janela da aplicao com os widgets.
   */
  private void populateFrame() {
    final Container cp = getApplicationFrame().getContentPane();
    cp.removeAll();
    cp.add(this.algorithmChooserPanel, BorderLayout.CENTER);
    cp.add(this.buttonPanel, BorderLayout.SOUTH);
    cp.validate();
    getApplicationFrame().pack();
  }

  /**
   * Remove o configurador de algoritmos atual e atualiza os widgets
   * relacionados.
   */
  private void clearConfigurator() {
    this.algorithmConfigurator = null;
    this.algorithmConfiguratorView = null;
  }

  /**
   * Cria o painel de seleo de algoritmos e verses.
   */
  private void createAlgorithmChooserPanel() {
    this.algorithmChooserPanel =
      new AlgorithmChooserPanel(getApplicationFrame(), mustShowHistory());
    this.algorithmChooserPanel
    .addAlgorithmChooserListener(new AlgorithmChooserListener() {
      @Override
      public void wasChangedVersion(final AlgorithmChooserEvent event) {
        updateVersionInfo();
        updateButtons();
      }

      @Override
      public void wasChangedAlgorithm(final AlgorithmChooserEvent event) {
        clearConfigurator();
        updateButtons();
      }
    });

    BorderUtil.setEtchedBorder(algorithmChooserPanel, null);
  }

  /**
   * Cria o painel de botes.
   */
  private void createButtonsPanel() {
    this.closeButton = new JButton(getString("button.close"));
    this.closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        closeApplication();
      }
    });
    this.executeButton = new JButton(getString("button.execute"));
    this.executeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        showExecuteDialog();
      }
    });
    this.paramButton = new JButton(getString("button.parameter"));
    this.paramButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        showParamsDialog();
      }
    });

    MenuButton extrasMenu =
      new MenuButton(ApplicationImages.ICON_BLANK_16, PopupPosition.BOTTOM);
    final List<JMenuItem> extraMenuItems = getExtraOperations();
    for (JMenuItem extraMenuItem : extraMenuItems) {
      extrasMenu.add(extraMenuItem);
    }
    extrasMenu.setVisible(!extraMenuItems.isEmpty());

    ClientUtilities.adjustEqualSizes(this.paramButton, this.closeButton,
      this.executeButton);
    this.buttonPanel = new JPanel(new FlowLayout());
    this.buttonPanel.add(this.paramButton);
    this.buttonPanel.add(this.executeButton);
    this.buttonPanel.add(this.closeButton);
    this.buttonPanel.add(extrasMenu);
    updateButtons();
  }

  /**
   * Monta a lista de ietns de menu com operaes extraordinrias, eventualmente
   * configuradas por sistema.
   *
   * @return lista de itens de menu.
   */
  private List<JMenuItem> getExtraOperations() {
    final List<JMenuItem> ops = new ArrayList<JMenuItem>();

    if (getBooleanSpecificProperty("flow.extra.operation", true)) {
      final String openInFlowText = getString("button.flowapp");
      final JMenuItem openInFlowAppItem =
        new JMenuItem(openInFlowText, ApplicationImages.ICON_FLOW_16);
      openInFlowAppItem.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent arg0) {
          openInFlowApplication();
        }
      });
      ops.add(openInFlowAppItem);
    }

    return ops;
  }

  /**
   * Abre a aplicao de edio de fluxos com o algoritmo selecionado.
   */
  private void openInFlowApplication() {
    if (algorithmConfigurator != null) {
      try {
        ApplicationManager applicationManager =
          ApplicationManager.getInstance();
        FlowApplication flowApp =
          applicationManager.runApplication(FlowApplication.class);
        flowApp.sendMessage(FlowApplication.CONFIGURATOR_MESSAGE,
          this.algorithmConfigurator, getInstanceId());
      }
      catch (ApplicationException e) {
        showError(getString("error.flowapp_not_available"));
      }
    }
  }

  /**
   * Cria o configurador de uma determinada verso de algoritmo.
   *
   * @param algorithmName O nome do algoritmo.
   * @param versionId O identificador da verso.
   *
   * @return O configurador.
   */
  private boolean createConfigurator(final String algorithmName,
    final AlgorithmVersionId versionId) {
    if (!this.algorithmChooserPanel.selectAlgorithm(algorithmName)) {
      showError(MessageFormat.format(getString("error.algorithm_not_found"),
        algorithmName));
      return false;
    }
    if (!this.algorithmChooserPanel.selectVersion(versionId)) {
      showError(MessageFormat.format(getString("error.version_not_found"),
        algorithmName, versionId));
      return false;
    }
    return true;
  }

  /**
   * Verifica se  necessrio permitir que o histrico seja exibido.
   *
   * @return <code>true</code> se o histrico precisa ser exibido ou
   *         <code>false</code> caso contrrio.
   */
  private boolean mustShowHistory() {
    return getBooleanSpecificProperty("showHistory", true);
  }

  /**
   * L o arquivo de parmetros do arquivo de projeto informado.
   *
   * @param clientProjectFile O arquivo de projeto.
   *
   * @return O arquivo de parmetros.
   */
  private AlgorithmConfigurator readParameterFile(
    final ClientProjectFile clientProjectFile) {
    final RemoteTask<AlgorithmConfigurator> task =
      new RemoteTask<AlgorithmConfigurator>() {

      @Override
      protected void performTask() throws Exception {
        InputStream inputStream = null;
        try {
          inputStream = clientProjectFile.getInputStream();

          final IAlgorithmConfigurationSerializer serializer =
            AlgorithmConfigurationSerializerFactory
            .getSerializer(AlgorithmConfigurator.class);

          final AlgorithmConfigurator configurator =
            serializer.read(inputStream);

          setResult(configurator);
        }
        finally {
          if (inputStream != null) {
            inputStream.close();
          }
        }
      }

      @Override
      protected void handleError(final Exception error) {
        if (error instanceof ParseException) {
          showError(error.getLocalizedMessage());
        }
        else if (error instanceof AlgorithmConfigurationSerializerParameterNotFoundException) {
          showError(error.getLocalizedMessage());
        }
        else {
          super.handleError(error);
        }
      }
    };
    if (task.execute(getApplicationFrame(), getName(),
      getString("msg.reading.file"))) {
      return task.getResult();
    }
    return null;
  }

  /**
   * Atribui a viso do configurador a esta aplicao.
   *
   * @param configuratorView A viso do configurador.
   */
  private void setConfiguratorView(
    final AlgorithmConfiguratorView configuratorView) {
    this.algorithmConfiguratorView = configuratorView;
    this.algorithmConfigurator = configuratorView.getConfigurator();
  }

  /**
   * Exibe o dilogo de obteno de parmetros do algoritmo
   */
  private void showParamsDialog() {
    this.algorithmConfiguratorView.launch();
  }

  /**
   * Exibe o dilogo de seleo de servidores.
   */
  private void showExecuteDialog() {
    final CommandExecutionDialog commandExecutionDialog =
      new CommandExecutionDialog(getApplicationFrame(),
        this.algorithmConfiguratorView.getConfigurator(),
        getApplicationProject());
    commandExecutionDialog
    .addCommandRequestedListener(new CommandRequestedListener() {
      @Override
      public void commandsWereRequested(
        final Set<CommandInfo> submittedCommands) {
        if (!mustShowCommandRequested) {
          return;
        }
        final int numberOfSubmittedCommands = submittedCommands.size();
        if (numberOfSubmittedCommands == 1) {
          final CommandInfo command = submittedCommands.iterator().next();
          if (command.isAutomatic()) {
            showCommandRequest(MessageFormat.format(
              getString("msg.command_auto_requested"), command.getTip()));
          }
          else {
            showCommandRequest(MessageFormat.format(
              getString("msg.command_requested_on_server"), command.getTip(),
              command.getSGAName()));
          }
        }
        else {
          // Quando mais de um comando foi submetido, basta testar se o
          // primeiro  automtico
          final CommandInfo firstCommand =
            submittedCommands.iterator().next();
          if (firstCommand.isAutomatic()) {
            showCommandRequest(MessageFormat.format(
              getString("msg.command_auto_requested_on_servers"),
              firstCommand.getTip(), numberOfSubmittedCommands));
          }
          else {
            String requestedServers = "";
            final Set<String> sgaNames = new HashSet<String>();
            for (final CommandInfo command : submittedCommands) {
              if (sgaNames.add(command.getSGAName())) {
                requestedServers =
                  requestedServers + "\n - " + command.getSGAName();
              }
            }
            int nExecutions =
              firstCommand.getExecutionCountPerSGAForMultipleExecution();
            showCommandRequest(MessageFormat.format(
              getString("msg.command_requested_on_servers"), firstCommand
                  .getTip(), requestedServers, nExecutions));
          }
        }
      }
    });
    addExternalListeners(commandExecutionDialog);
    commandExecutionDialog.setVisible(true);
  }

  /**
   * Exibe uma mensagem de comando submetido.
   *
   * @param message A mensagem (No aceita {@code null}).
   */
  private void showCommandRequest(final String message) {
    if (message == null) {
      throw new IllegalArgumentException("O parmetro message est nulo.");
    }
    final String[] buttons =
      new String[] { getString("button.monitor"), getString("button.close") };
    final ApplicationFrame applicationFrame = getApplicationFrame();
    final String applicationTitle = applicationFrame.getTitle();
    JTextArea area = new JTextArea(message);
    area.setEditable(false);
    final int min_rows = 3;
    final int max_rows = 10;
    String s[] = message.split("\\n", max_rows);
    if (s.length > min_rows) {
      area.setRows(max_rows);
    }
    JScrollPane scrollPane = new JScrollPane(area);
    final int option =
      JOptionPane.showOptionDialog(applicationFrame, scrollPane,
        applicationTitle, JOptionPane.DEFAULT_OPTION,
        JOptionPane.INFORMATION_MESSAGE, null, buttons, buttons[1]);
    if (option == 0) {
      try {
        ApplicationManager.getInstance().runApplication("commandsmonitor");
      }
      catch (final ApplicationException ex) {
        showException(ex.getMessage(), ex);
      }
    }
  }

  /**
   * Adiciona listeners externos para o comando solicitado pela janela de
   * execuo de comandos. Esses listeners so cadastrados por classes que
   * executem essa aplicao.
   *
   * @param commandExecutionDialog janela de execuo de comandos
   */
  private void addExternalListeners(
    final CommandExecutionDialog commandExecutionDialog) {
    for (final CommandRequestedListener listener : listeners) {
      commandExecutionDialog.addCommandRequestedListener(listener);
    }
  }

  /**
   * Adiciona um listener para um comando solicitado com o dilogo de execuo
   * de comandos.
   *
   * @param listener listener de um comando solicitado com o dilogo de execuo
   *        de comandos
   */
  public void addCommandRequestedListener(
    final CommandRequestedListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Atualiza o botes. Caso o algoritmo no possa ser executado, exibe uma
   * mensagem na barra de status explicando o porqu.
   */
  private void updateButtons() {
    boolean enableButton = true;
    if (this.algorithmConfigurator == null) {
      enableButton = false;
      getApplicationFrame().getStatusBar().setError(
        getString("status.error.null_configurator"));
    }
    else if (algorithmConfigurator.hasParameterThatRequiresPipe()) {
      enableButton = false;
      getApplicationFrame().getStatusBar().setError(
        getString("status.error.flow_only_algorithm"));
    }
    else {
      getApplicationFrame().getStatusBar().setStatus(null);
    }
    this.executeButton.setEnabled(enableButton);
    this.paramButton.setEnabled(enableButton);
  }

  /**
   * Atualiza o painel de seleo de verses de algoritmos.
   */
  private void updateChooserPanel() {
    this.algorithmChooserPanel.setDescription(this.algorithmConfigurator
      .getDescription());
  }

  /**
   * Obtm o conjunto de todos os algoritmos. Esse conjunto  obtido usando-se o
   * proxy do servio (<code>AlgorithManagementProxy</code>), que retorna um
   * array de <code>AlgorithmInfo</code>. O array  convertido para um
   * <code>Set</code>. o prprio proxy poderia j retornar o <code>Set</code>.
   *
   * @return o conjunto de todos os algoritmos obtidos do servio
   */
  private SortedSet<AlgorithmInfo> getAlgorithmInfoSet() {
    algorithms = new TreeSet<AlgorithmInfo>();
    final List<String> algoIds = getStringListSpecificProperty("algo.id");

    // Verifica se deve ser feita a carga inicial dos algoritmos do servidor
    if (mustLoadAlgorithms) {
      final ApplicationFrame frame = this.getApplicationFrame();
      if (algoIds.size() > 0) {
        // Algoritmos especficos da aplicao
        for (final String algoId : algoIds) {
          final AlgorithmInfo info =
            AlgorithmManagementProxy.getAlgorithmInfoById(algoId, frame,
              AlgorithmManagementProxy.AlgorithmOperation.EXECUTE_ALGORITHM);
          if (info != null) {
            algorithms.add(info);
          }
        }
      }
      else {
        // Todos os algoritmos
        final AlgorithmInfo[] infos =
          AlgorithmManagementProxy.getAllAlgorithmInfos(frame,
            AlgorithmManagementProxy.AlgorithmOperation.EXECUTE_ALGORITHM);
        if (infos != null) {
          for (final AlgorithmInfo info : infos) {
            final String hideProp = AlgorithmInfo.HIDE_ALGORITHM_PROPERTY;
            if (info.getPropertyValue(hideProp) == null) {
              algorithms.add(info);
            }
          }
        }
      }
    }
    return algorithms;
  }

  /**
   * Atualiza as informaes da verso selecionada.
   */
  private void updateVersionInfo() {
    final AlgorithmVersionInfo selectedVersionInfo =
      this.algorithmChooserPanel.getSelectedVersionInfo();
    clearConfigurator();
    if (selectedVersionInfo != null) {
      final AlgorithmConfiguratorView configuratorView =
        AlgorithmConfiguratorFactory.getInstance().createConfigurationView(
          getApplicationFrame(), selectedVersionInfo, ValidationMode.FULL);
      if (configuratorView == null) {
        return;
      }
      setConfiguratorView(configuratorView);
      updateChooserPanel();
    }
    getApplicationFrame().pack();
  }

  /**
   * Obtm remotamente as informaes de um determinado algoritmo, dado o seu
   * nome.
   *
   * @param algorithmName nome do algoritmo procurado
   * @return uma estrutura com as informaes do algoritmo procurado, ou null,
   *         caso ele no tenha sido encontrado no servidor
   * @throws ApplicationException no caso do algoritmo no ser encontrado.
   */
  private AlgorithmInfo getAlgorithmInfo(final String algorithmName)
    throws ApplicationException {
    final AlgorithmInfo info =
      AlgorithmManagementProxy.getAlgorithmInfoByName(algorithmName,
        getApplicationFrame(),
        AlgorithmManagementProxy.AlgorithmOperation.EXECUTE_ALGORITHM);
    if (info == null) {
      throw new ApplicationException(getString("error.inexistent_algorithm"));
    }
    return info;
  }

  /**
   * Obtm um conjunto ordenado com as verses dos algoritmos, cujos
   * identificadores da verso especificados.
   *
   * @param algorithmInfo informaes do algoritmo
   * @param versionIdList array contendo os identificadores das verses
   *        procuradas do algoritmo
   * @return retorna um conjunto ordenado com as verses dos algoritmos
   */
  private SortedSet<AlgorithmVersionInfo> getAlgorithmVersionSet(
    final AlgorithmInfo algorithmInfo, final AlgorithmVersionId[] versionIdList) {
    final SortedSet<AlgorithmVersionInfo> sortedVersionSet =
      new TreeSet<AlgorithmVersionInfo>();
    for (final AlgorithmVersionId versionId : versionIdList) {
      final AlgorithmVersionInfo versionInfo =
        algorithmInfo.getVersionInfo(versionId);
      if (versionInfo != null) {
        sortedVersionSet.add(versionInfo);
      }
    }
    return sortedVersionSet;
  }

  /**
   * Estabelece um nico algoritmo, com seu nome e verses especificados, para
   * ser exibido na lista de algoritmos que o usurio pode editar e executar.
   *
   * @param algorithmName nome do algoritmo
   * @param versionIdList array contendo os identificadores das verses
   *        procuradas do algoritmo
   * @throws ApplicationException exceo lanada quando o algoritmo requerido
   *         no est cadastrado no servidor
   */
  public void setSingleAlgorithm(final String algorithmName,
    final AlgorithmVersionId[] versionIdList) throws ApplicationException {
    // Algoritmo a ser exibido na lista
    final AlgorithmInfo algorithmInfo = getAlgorithmInfo(algorithmName);
    algorithms.add(algorithmInfo);
    this.algorithmChooserPanel.setAlgorithmSet(algorithms);

    if (versionIdList != null) {
      // Verses do algoritmo a serem exibidas na lista
      final SortedSet<AlgorithmVersionInfo> algorithmVersionSet =
        getAlgorithmVersionSet(algorithmInfo, versionIdList);
      this.algorithmChooserPanel.setAlgorithmVersion(algorithmVersionSet);
    }
  }

  /**
   * Estabelece um nico algoritmo para ser exibido na lista de algoritmos que o
   * usurio pode editar e executar, de acordo com o nome especificado. Todas as
   * verses do algoritmo so exibidas na lista de seleo de verses.
   *
   * @param algorithmName nome do algoritmo
   * @throws ApplicationException exceo lanada quando o algoritmo requerido
   *         no est cadastrado no servidor
   */
  public void setSingleAlgorithm(final String algorithmName)
    throws ApplicationException {
    setSingleAlgorithm(algorithmName, null);
  }

  /**
   * Verifica se a informao de que o comando foi solicitado deve ser exibida.
   *
   * @return retorna true, se deve ser exibida a informao de que o comando foi
   *         solicitado, caso contrrio, retorna false
   */
  public boolean isMustShowCommandRequested() {
    return mustShowCommandRequested;
  }

  /**
   * Determina se deve ou no exibir uma janela com a informao de que o
   * comando foi solicitado, e se deseja monitor-lo.
   *
   * @param state se true, exibe a informao quando o comando for solicitado,
   *        caso contrrio, a janela no  exibida
   */
  public void showCommandRequestedInfo(final boolean state) {
    mustShowCommandRequested = state;
  }

  /**
   * Determina se deve ou no carregar os algoritmos do servidor inicialmente,
   * ou se isso ser feito sob demanda.
   *
   * @param state se true, carrega os algoritmos inicialmente, caso contrrio,
   *        cada algoritmo ser carregado sob demanda
   */
  public void enableLoadAlgorithms(final boolean state) {
    mustLoadAlgorithms = state;
  }

  /**
   * Estabelece a descrio default do comando a ser gerado pelo algoritmo.
   *
   * @param commandDescription descrio do comando
   */
  public void setCommandDescription(final String commandDescription) {
    algorithmConfigurator.setCommandDescription(commandDescription);
  }
}
