package csbase.client.applications.fileexchanger.logic;

import java.awt.geom.Point2D;
import java.io.File;
import java.util.ArrayList;

import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * Item de transferncia
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class Exchange {

  /**
   * Arquivo local
   */
  private final File localFile;

  /**
   * Arquivo remoto
   */
  private ClientProjectFile remoteFile;

  /**
   * Diretrio remoto
   */
  private final ClientProjectFile remoteDir;

  /**
   * Projeto que vai conter o arquivo.
   */
  private final CommonClientProject project;

  /**
   * Tipo
   */
  final private ExchangeType type;

  /**
   * Hora de incio
   */
  private long startTime;

  /**
   * Hora atual (para efeito de contabilizao de taxa de transferncia)
   */
  private long currentTime;

  /**
   * Hora de criao
   */
  private final long creationTime;

  /**
   * Exceo capturada, se houver (pode ser <code>null</code>)
   */
  private Exception exception;

  /**
   * Thread (viva) se estiver em endamento.
   */
  private ExchangeThread thread;

  /**
   * Tamanho (em bytes) total da transferncia
   */
  private long totalTransferSize;

  /**
   * Tamanho (em bytes) corrente da transferncia
   */
  private long currentTransferSize;

  /**
   * Estado
   */
  private ExchangeState state;

  /**
   * Lista de pontos
   */
  final private ArrayList<Point2D.Double> points =
    new ArrayList<Point2D.Double>();

  /**
   * Flag para garantir que um ping no pegue dois estados iguais.
   */
  private boolean isAlreadyPinged;

  /**
   * Modo.
   */
  final private ExchangeMode transferMode;

  /**
   * Tamanho de bloco.
   */
  final private BlockSize blockSize;

  /**
   * Consulta o valor de blockSize
   * 
   * @return o valor
   */
  public final BlockSize getBlockSize() {
    return blockSize;
  }

  /**
   * Consulta o valor de totalTransferSize
   * 
   * @return o valor
   */
  public final long getTotalTransferSize() {
    return totalTransferSize;
  }

  /**
   * Ajusta o valor de totalTransferSize.
   * 
   * @param totalTransferSize o valor a ser ajustado.
   */
  public void setTotalTransferSize(final long totalTransferSize) {
    this.totalTransferSize = totalTransferSize;
  }

  /**
   * Consulta o valor de currentTransferSize
   * 
   * @return o valor
   */
  public synchronized final long getCurrentTransferSize() {
    return currentTransferSize;
  }

  /**
   * Ajusta o valor de currentTransferSize
   * 
   * @param size o valor a ser ajustado.
   */
  public synchronized final void addCurrentTransferSize(final long size) {
    this.currentTransferSize += size;
    this.currentTime = System.currentTimeMillis();
    this.isAlreadyPinged = false;
  }

  /**
   * Consulta o valor de exception
   * 
   * @return o valor
   */
  public final Exception getException() {
    return exception;
  }

  /**
   * Consulta o valor de localFile
   * 
   * @return o valor
   */
  public final File getLocalFile() {
    return localFile;
  }

  /**
   * Consulta o valor de remoteFile
   * 
   * @return o valor
   */
  public final ClientProjectFile getRemoteFile() {
    return remoteFile;
  }

  /**
   * Define o arquivo remoto
   * 
   * @param remoteFile o arquivo remoto a ser definido
   */
  public void setRemoteFile(final ClientProjectFile remoteFile) {
    this.remoteFile = remoteFile;
  }

  /**
   * Obtm o diretrio remoto
   * 
   * @return o diretrio remoto
   */
  public final ClientProjectFile getRemoteDir() {
    return remoteDir;
  }

  /**
   * Obtm o projeto remoto
   * 
   * @return o projeto remoto
   */
  public final CommonClientProject getProject() {
    return project;
  }

  /**
   * Consulta o valor de type
   * 
   * @return o valor
   */
  public final ExchangeType getType() {
    return type;
  }

  /**
   * Consulta o valor de startTime
   * 
   * @return o valor
   */
  public final long getStartTime() {
    return startTime;
  }

  /**
   * Consulta o valor de creationTime
   * 
   * @return o valor
   */
  public final long getCreationTime() {
    return creationTime;
  }

  /**
   * Indica se a operao est em andamento.
   * 
   * @return indicativo
   */
  final synchronized public ExchangeState getState() {
    return state;
  }

  /**
   * Incio da operao.
   */
  final synchronized public void start() {
    if (state == ExchangeState.PAUSED) {
      this.state = ExchangeState.RUNNING;
      return;
    }
    if (state == ExchangeState.QUEUED) {
      this.startTime = System.currentTimeMillis();
      this.currentTime = startTime;
      this.thread = createThread();
      this.thread.start();
      this.state = ExchangeState.RUNNING;
    }
  }

  /**
   * Pausa na operao.
   */
  final synchronized public void pause() {
    if (state != ExchangeState.RUNNING) {
      return;
    }
    state = ExchangeState.PAUSED;
  }

  /**
   * Reincio de operao.
   */
  final synchronized public void resume() {
    if (state != ExchangeState.PAUSED) {
      return;
    }
    state = ExchangeState.RUNNING;
  }

  /**
   * Interrupo da operao.
   */
  final synchronized public void interrupt() {
    if (state == ExchangeState.QUEUED || state == ExchangeState.CREATED) {
      state = ExchangeState.INTERRUPTED;
      return;
    }

    if (state != ExchangeState.RUNNING && state != ExchangeState.PAUSED) {
      return;
    }

    // Faz ltimo ping.
    ping();

    // A sinalizaao do estado vem antes para que se possa sinalizar
    // corretamente
    // na IHM. Seno, haver exceo e o programa acusar erro.
    state = ExchangeState.INTERRUPTED;

    thread.interrupt();
    thread = null;
    exception = null;
    currentTime = System.currentTimeMillis();
  }

  /**
   * Sinalizao de trmino da operao.
   * 
   * @param ex a exceo de erro (se for o caso) ou <code>null</code>
   */
  public synchronized void signalEnded(final Exception ex) {
    this.exception = ex;
    this.thread = null;
    currentTime = System.currentTimeMillis();
    if (state == ExchangeState.INTERRUPTED) {
      this.exception = null;
      return;
    }

    // Faz ltimo ping.
    ping();

    if (ex == null) {
      state = ExchangeState.FINISHED;
    }
    else {
      state = ExchangeState.ERROR;
    }
  }

  /**
   * @return o progresso.
   */
  final synchronized public double getProgress() {
    if (state == ExchangeState.FINISHED) {
      return 1.0;
    }
    return ((double) currentTransferSize) / ((double) totalTransferSize);
  }

  /**
   * Consulta o tempo de durao da operao.
   * 
   * @return o tempo
   */
  final public synchronized long getElapsedTime() {
    if (state == ExchangeState.QUEUED) {
      return -1;
    }
    final long dt = currentTime - startTime;
    return dt;
  }

  /**
   * Consulta o tempo total estimado de transferncia
   * 
   * @return o tempo em segundos
   */
  final public synchronized long getTotalTime() {
    if (state != ExchangeState.RUNNING) {
      return -1;
    }
    final double ratio =
      ((double) currentTransferSize) / ((double) totalTransferSize);
    final long dt = getElapsedTime();
    return (long) (dt / ratio);
  }

  /**
   * Consulta o tempo total estimado de transferncia
   * 
   * @return o tempo em segundos
   */
  final public synchronized long getRemainingTime() {
    if (state != ExchangeState.RUNNING) {
      return -1;
    }
    final long diff = getTotalTime() - getElapsedTime();
    return diff;
  }

  /**
   * Consulta a taxa global de trasferncia de dados
   * 
   * @return a taxa
   */
  final public synchronized double getInstantTransferRateMbSec() {
    if (state != ExchangeState.RUNNING) {
      return -1.0;
    }

    if (points == null) {
      return -2.0;
    }
    final int size = points.size();
    if (size < 2) {
      return -3.0;
    }

    final Point2D.Double pt1 = points.get(size - 1);
    final Point2D.Double pt2 = points.get(size - 2);
    final double dt = pt2.x - pt1.x;
    final double ds = pt2.y - pt1.y;
    final double rate = ds / dt;
    return rate;
  }

  /**
   * Consulta a taxa global de trasferncia de dados
   * 
   * @return a taxa
   */
  final public synchronized double getGlobalTransferRateMbSec() {
    final long dt = getElapsedTime();
    final double tSec = dt / 1000.;
    final double sizeMb = currentTransferSize / 1024. / 1024.;
    if (Math.abs(tSec) <= 1e-7) {
      return 0.0;
    }
    return sizeMb / tSec;
  }

  /**
   * Sinalizao de operao adicionada no programa
   */
  public synchronized void signalAdded() {
    state = ExchangeState.QUEUED;
  }

  /**
   * Sinalizao de operao adicionada com erro no programa
   */
  public synchronized void signalError() {
    state = ExchangeState.ERROR;
  }

  /**
   * Sinalizao de operao removida do programa.
   */
  public synchronized void signalRemoved() {
    state = ExchangeState.REMOVED;
  }

  /**
   * Consulta o nome.
   * 
   * @return o nome
   */
  final public String getLabel() {
    if (getRemoteFile() != null) {
      return getRemoteFile().getName();
    }
    return "";
  }

  /**
   * Marca o ponto.
   */
  final synchronized public void ping() {
    if (isAlreadyPinged) {
      return;
    }
    final double time = getElapsedTime() / 1000.;
    final double sz = currentTransferSize / 1024. / 1024.;
    final Point2D.Double pt = new Point2D.Double(time, sz);
    points.add(pt);
    isAlreadyPinged = true;
  }

  /**
   * Consulta os pontos.
   * 
   * @return os pontos
   */
  final synchronized public ArrayList<Point2D.Double> getPoints() {
    return points;
  }

  /**
   * Consulta o modo de transferncia
   * 
   * @return o modo
   */
  final public ExchangeMode getTransferMode() {
    return transferMode;
  }

  /**
   * Construtor
   * 
   * @param type tipo de operao.
   * @param transferMode modo de transferncia
   * @param blockSize tamannho de bloco
   * @param localFile arquivo local
   * @param remoteFile arquivo remoto.
   * @param remoteDir diretrio onde ser colocado o arquivo remoto
   * @param totalTransferSize tamanho total da transferncia
   * @param project projeto onde ser colocado o arquivo
   */
  public Exchange(final ExchangeType type, final ExchangeMode transferMode,
    final BlockSize blockSize, final File localFile,
    final ClientProjectFile remoteFile, final ClientProjectFile remoteDir,
    final long totalTransferSize, final CommonClientProject project) {

    this.blockSize = blockSize;
    this.transferMode = transferMode;
    this.type = type;
    this.localFile = localFile;
    this.remoteDir = remoteDir;
    this.remoteFile = remoteFile;
    this.project = project;
    this.creationTime = System.currentTimeMillis();
    this.startTime = 0;
    this.currentTime = 0;
    this.totalTransferSize = totalTransferSize;
    this.currentTransferSize = 0;
    this.state = ExchangeState.CREATED;
  }

  /**
   * Criao de uma thread para execuo.
   * 
   * @return a thread
   */
  abstract protected ExchangeThread createThread();

  /**
   * Obtm o caminho do arquivo remoto como String.
   * 
   * @return o caminho
   */
  public String getRemoteFilePath() {
    if (remoteFile != null) {
      return remoteFile.getStringPath();
    }
    return remoteDir.getStringPath() + File.separator + localFile.getName();
  }
}
