package csbase.client.applications.flowapplication.graph;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.vix.TypeMessage;
import csbase.client.applications.flowapplication.messages.ErrorMessage;
import csbase.client.applications.flowapplication.messages.PickLinkMessage;
import csbase.client.applications.flowapplication.messages.PickNodeMessage;
import csbase.logic.algorithms.parameters.AbstractFileParameter;
import csbase.logic.algorithms.parameters.FileParameterListener;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.SimpleParameterListener;

/**
 * <p>
 * A aresta que representa a ligao de 2 parmetros de arquivo de 2 ns que
 * representam configuradores de algoritmo.<br>
 * No momento, pode ser uma ligao entre URLs ou entre Arquivos
 *
 * </p>
 *
 * @author lmoreira
 */
public final class GraphLink extends GraphElement implements
FileParameterListener, SimpleParameterListener {

  /**
   * Cor padro da ligao.
   */
  private static final Color DEFAULT_COLOR = Color.BLACK;
  /**
   * Cor da ligao com erro.
   */
  private static final Color ERROR_COLOR = Color.RED;
  /**
   * Cor da ligao desviada.
   */
  private static final Color BYPASSED_COLOR = new Color(150, 150, 150);
  /**
   * Cor da linha a ser removida.
   */
  private static final Color REMOVE_LINE_COLOR = new Color(255, 255, 100);
  /**
   * Estilo da linha que desenha a ligao.
   */
  private static final Stroke STROKE = new BasicStroke(1f,
    BasicStroke.CAP_BUTT, BasicStroke.JOIN_ROUND);
  /**
   * Cor atual da ligao.
   */
  private Color color;
  /**
   * Representao geomtrica da ligao.
   */
  private Path2D path;

  /**
   * Descritor do arquivo de entrada da ligao.
   */
  private GraphFileDescriptor inputFileDescriptor;

  /**
   * Indica se a ligao est realada.
   */
  private boolean isHighlighted;

  /**
   * Descritor do arquivo de sada da ligao.
   */
  private GraphFileDescriptor outputFileDescriptor;

  /**
   * Lista de pontos pertencentes  ligao.
   */
  private final List<GraphPoint> pointList;
  /**
   * Ponto temporrio utilizado nas operaes com a ligao.
   */
  private GraphPoint temporaryPoint;

  /**
   * Construtor
   *
   * @param graph grafo
   * @param id id do link.
   */
  GraphLink(final Graph graph, final int id) {
    super(graph, id);
    this.pointList = new ArrayList<GraphPoint>();
    createShape();
    updateColor();
  }

  /**
   * Adiciona um ponto  ligao, "quebrando" a linha que a representa.
   *
   * @param point o ponto a ser adicionado.
   * @return verdadeiro se o novo ponto foi adicionado com sucesso.
   */
  public boolean breakPoint(final Point2D point) {
    final GraphPoint graphPoint = new GraphPoint(point);
    for (int i = 1; i < this.pointList.size(); i++) {
      final GraphPoint priorPoint = this.pointList.get(i - 1);
      final GraphPoint nextPoint = this.pointList.get(i);
      if (contains(graphPoint, priorPoint, nextPoint)) {
        this.pointList.add(i, new GraphPoint(point));
        return true;
      }
    }
    return false;
  }

  /**
   * Adiciona pontos  ligao de forma que some use linhas ortogonais. O ponto
   * de quebra ser calculado automaticamente.
   */
  public void breakAsOrthogonalLink() {
    breakAsOrthogonalLinkAt(null);
  }

  /**
   * Adiciona pontos  ligao de forma que somente use linhas ortogonais.
   *
   * @param y altura do ponto de quebra.
   */
  private void breakAsOrthogonalLinkAt(Double y) {
    GraphPoint firstPoint = pointList.get(0);
    Point2D outputPoint = firstPoint.getLocation();
    GraphPoint lastPoint = pointList.get(pointList.size() - 1);
    Point2D inputPoint = lastPoint.getLocation();
    double deltaX = inputPoint.getX() - outputPoint.getX();
    double deltaY = inputPoint.getY() - outputPoint.getY();
    removeIntermediatePoints();
    if (deltaX != 0 && deltaY != 0) {
      double newY;
      if (y == null) {
        newY = outputPoint.getY() + deltaY / 2;
      }
      else {
        newY = y;
      }
      Point2D outputBreakPoint = new Point2D.Double();
      Point2D inputBreakPoint = new Point2D.Double();
      outputBreakPoint.setLocation(outputPoint.getX(), newY);
      inputBreakPoint.setLocation(inputPoint.getX(), newY);
      GraphPoint inputGraphPoint =
        new GraphPoint(Grid.adjustPoint(inputBreakPoint));
      GraphPoint outputGraphPoint =
        new GraphPoint(Grid.adjustPoint(outputBreakPoint));
      pointList.add(1, outputGraphPoint);
      pointList.add(2, inputGraphPoint);
      createShape();
      repaint();
    }
  }

  /**
   * Remove todos os pontos intermediriso pertencentes  esta ligao.
   */
  private void removeIntermediatePoints() {
    if (pointList.size() > 2) {
      pointList.subList(1, pointList.size() - 1).clear();
      createShape();
      repaint();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    super.callbackRepaint(g);
    if (this.path != null) {
      g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
        RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
      g.setPaint(this.color);
      g.setStroke(STROKE);
      g.draw(this.path);
      if (isSelected()) {
        for (int i = 0; i < this.pointList.size(); i++) {
          final GraphPoint point = this.pointList.get(i);
          point.paint(g);
        }
        if (this.temporaryPoint != null) {
          this.temporaryPoint.paint(g);
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deattach() {
    if (this.inputFileDescriptor != null) {
      this.inputFileDescriptor.setLinkTo(null);
      setInputFileDescriptor(null);
    }
    if (this.outputFileDescriptor != null) {
      this.outputFileDescriptor.removeLinkFrom(this);
      setOutputFileDescriptor(null);
    }
    super.deattach();
  }

  /**
   * Finaliza a ligao, "fechando" a conexo com o descritor do parmetro de
   * entrada.
   *
   * @param inputDescriptor o descritor do parmetro de entrada.
   * @return verdadeiro se a ligao foi finalizada com sucesso.
   */
  public boolean finish(final GraphFileDescriptor inputDescriptor) {
    if (inputDescriptor == null) {
      throw new IllegalArgumentException(
        "O parmetro inputFileDescriptor est nulo.");
    }
    AbstractFileParameter inputParameter = inputDescriptor.getFileParameter();
    if (inputParameter == null) {
      return false;
    }
    if (this.outputFileDescriptor != null) {
      AbstractFileParameter outputParameter =
        this.outputFileDescriptor.getFileParameter();
      if (outputParameter == null) {
        return false;
      }

      String[] outputFileType = outputParameter.getFileTypes();
      String[] inputFileType = inputParameter.getFileTypes();
      if ((inputFileType == null && outputFileType != null)
        || (inputFileType != null && !Arrays
        .equals(inputFileType, outputFileType))) {
        if (outputFileType == null) {
          outputFileType =
            new String[] { LNG.get(GraphLink.class.getName() + ".unknownFileType") };
        }
        if (inputFileType == null) {
          inputFileType = new String[] {
            LNG.get(GraphLink.class.getName() + ".unknownFileType")};
        }
        new ErrorMessage(GraphLink.class.getName()
          + ".error_different_file_types", outputFileType, inputFileType)
        .sendVS(getVS());
        return false;
      }
    }
    final GraphNode node = inputDescriptor.getNode();
    if (inputDescriptor.getLinkTo() != null) {
      removeLastPoint();
      new ErrorMessage(GraphLink.class.getName() + ".error_only_one_link")
      .sendVS(getVS());
      return false;
    }
    if (node.canReach(getStartNode())) {
      removeLastPoint();
      new ErrorMessage(GraphLink.class.getName() + ".error_create_cicle")
      .sendVS(getVS());
      return false;
    }
    setEnd(inputDescriptor);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Rectangle2D getBounds2D() {
    if (this.path == null) {
      return new Rectangle2D.Double();
    }
    /*
     * Cria uma forma a partir da linha para poder retornar um retngulo com
     * largura > 0, que permita testar a interese
     */
    BasicStroke stroke = new BasicStroke();
    Shape shape = stroke.createStrokedShape(path);
    Rectangle2D bounds2d = shape.getBounds2D();
    return bounds2d;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getHint(final Point2D point) {
    if (!contains(point)) {
      return null;
    }
    return toString();
  }

  /**
   * Obtm o descritor do arquivo de entrada desta ligao.
   *
   * @return o descritor.
   */
  public GraphFileDescriptor getInputFileDescriptor() {
    return this.inputFileDescriptor;
  }

  /**
   * Obtm a lista de pontos pertencentes  esta ligao.
   *
   * @return a lista de pontos.
   */
  public List<Point2D> getPointList() {
    final List<Point2D> newPointList =
      new ArrayList<Point2D>(this.pointList.size());
    for (int i = 0; i < this.pointList.size(); i++) {
      final GraphPoint graphPoint = this.pointList.get(i);
      newPointList.add(graphPoint.getLocation());
    }
    return Collections.unmodifiableList(newPointList);
  }

  /**
   * Obtm a lista de pontos intermediriso pertencentes  esta ligao.
   * Corresponde  lista de todos os pontos menos os pontos inicial e final.
   *
   * @return a lista de pontos.
   */
  public List<Point2D> getIntermediatePointList() {
    final List<Point2D> newPointList =
      new ArrayList<Point2D>(this.pointList.size());
    for (int i = 1; i < this.pointList.size() - 1; i++) {
      final GraphPoint graphPoint = this.pointList.get(i);
      newPointList.add(graphPoint.getLocation());
    }
    return Collections.unmodifiableList(newPointList);
  }

  /**
   * Obtm o descritor do arquivo de sada desta ligao.
   *
   * @return o descritor.
   */
  public GraphFileDescriptor getOutputFileDescriptor() {
    return this.outputFileDescriptor;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void hasLinkWasChanged(final AbstractFileParameter parameter) {
    updateColor();
    repaint();
  }

  /**
   * Aumenta o tamanho da linha que representa a ligao.
   *
   * @param point novo ponto at onde aumentar a ligao.
   */
  public void increase(Point2D point) {
    Point2D pt = Grid.adjustPoint(point);
    final GraphPoint graphPoint = new GraphPoint(pt);
    this.pointList.add(graphPoint);
    if (this.pointList.size() != 1) {
      final GraphPoint previousPoint =
        this.pointList.get(this.pointList.size() - 2);
      if (canBeRemoved(previousPoint)) {
        this.pointList.remove(previousPoint);
      }
    }
    createShape();
    for (GraphElementListener listener : getListenerList()) {
      listener.wasIncreased(this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean msgHandlerVO(final TypeMessage msg) {
    if (msg instanceof PickLinkMessage) {
      return handlePickLinkMessage((PickLinkMessage) msg);
    }
    return super.msgHandlerVO(msg);
  }

  /**
   * Simula o aumento da linha que representa a ligao.
   *
   * @param pt ponto at onde aumentar a linha.
   */
  public void pretentToIncrease(final Point2D pt) {
    this.temporaryPoint = new GraphPoint(pt);
    createShape();
  }

  /**
   * Remove o ltimo ponto da linha.
   */
  public void removeLastPoint() {
    this.pointList.remove(this.pointList.size() - 1);
  }

  /**
   * Atribui o "final" da ligao, que  o descritor do arquivo de entrada.
   *
   * @param inputFileDescriptor o descritor do arquivo de entrada da ligao.
   */
  public void setEnd(final GraphFileDescriptor inputFileDescriptor) {
    if (inputFileDescriptor == null) {
      throw new IllegalArgumentException(
        "O parmetro inputFileDescriptor est nulo.");
    }
    inputFileDescriptor.setLinkTo(this);
    setInputFileDescriptor(inputFileDescriptor);
    createShape();
  }

  /**
   * Atribui o "incio" da ligao, que  o descritor do arquivo de sada.
   *
   * @param outputFileDescriptor o descritor do arquivo de sada da ligao.
   */
  public void setStart(final GraphFileDescriptor outputFileDescriptor) {
    if (outputFileDescriptor == null) {
      throw new IllegalArgumentException(
        "O parmetro outputFileDescriptor est nulo.");
    }
    outputFileDescriptor.addLinkFrom(this);
    setOutputFileDescriptor(outputFileDescriptor);
    createShape();
  }

  /**
   * Inicia uma nova ligao a partir de um descritor de um arquivo de sada.
   *
   * @param outputDescriptor o descritor do arquivo de sada da ligao.
   * @return verdadeiro se a ligao foi iniciada com sucesso.
   */
  public boolean start(final GraphFileDescriptor outputDescriptor) {
    if (outputDescriptor.getFileParameter() == null) {
      return false;
    }
    final GraphNode node = outputDescriptor.getNode();
    if (this.inputFileDescriptor != null) {
      String[] inputFileType =
        this.inputFileDescriptor.getFileParameter().getFileTypes();
      String[] outputFileType = outputDescriptor.getFileParameter().getFileTypes();
      if ((inputFileType == null && outputFileType != null)
        || (inputFileType != null && !Arrays
        .equals(inputFileType, outputFileType))) {
        if (outputFileType == null) {
          outputFileType =
            new String[] { LNG.get(GraphLink.class.getName() + ".unknownFileType") };
        }
        if (inputFileType == null) {
          inputFileType = new String[] {
            LNG.get(GraphLink.class.getName() + ".unknownFileType")};
        }
        new ErrorMessage(GraphLink.class.getName()
          + ".error_different_file_types", outputFileType, inputFileType)
        .sendVS(getVS());
        return false;
      }
    }
    if (getEndNode() != null && getEndNode().canReach(node)) {
      removeLastPoint();
      new ErrorMessage(GraphLink.class.getName() + ".error_create_cicle")
      .sendVS(getVS());
      return false;
    }
    setStart(outputDescriptor);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    final String start =
      (this.outputFileDescriptor == null ? "X" : this.outputFileDescriptor
        .toString());
    final String end =
      (this.inputFileDescriptor == null ? "X" : this.inputFileDescriptor
        .toString());
    return start + " -> " + end;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean turnOnHighlight(final Point2D pt) {
    if (!contains(pt)) {
      return false;
    }
    this.isHighlighted = true;
    updateColor();
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void turnOffHighlight() {
    this.isHighlighted = false;
    updateColor();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void typeWasChanged(final AbstractFileParameter parameter) {
    updateColor();
    repaint();
    for (GraphElementListener listener : getListenerList()) {
      listener.wasLinkStatusChanged(this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void capabilityWasChanged(final SimpleParameter parameter) {
    updateColor();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void defaultValueWasChanged(final SimpleParameter parameter) {
    // Ignora o evento.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void labelWasChanged(final SimpleParameter parameter) {
    // Ignora o evento.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void valueWasChanged(final SimpleParameter parameter) {
    // Ignora o evento.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void visibilityWasChanged(final SimpleParameter parameter) {
    updateColor();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean contains(final Point2D pt) {
    final GraphPoint point = new GraphPoint(pt);
    for (int i = 1; i < this.pointList.size(); i++) {
      final GraphPoint startPoint = this.pointList.get(i - 1);
      final GraphPoint endPoint = this.pointList.get(i);
      if (contains(point, startPoint, endPoint)) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean doDrag(final Point2D startPoint, Point2D endPoint) {
    Point2D adjustedPoint = adjustPoint(endPoint);
    final int indexOf = this.pointList.indexOf(new GraphPoint(startPoint));
    if (indexOf < 0) {
      return false;
    }
    this.pointList.set(indexOf, new GraphPoint(adjustedPoint));
    if (indexOf == 0) {
      moveFirstPoint(adjustedPoint);
    }
    else if (indexOf == this.pointList.size() - 1) {
      moveLastPoint(adjustedPoint);
    }
    createShape();
    updateColor();
    if (canBeRemoved(new GraphPoint(Grid.adjustPoint(adjustedPoint)))) {
      this.color = REMOVE_LINE_COLOR;
      repaint();
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void doDrag(final double tx, final double ty) {
    for (int i = 1; i < this.pointList.size() - 1; i++) {
      final GraphPoint graphPoint = this.pointList.get(i);
      final Point2D point = graphPoint.getLocation();
      final double x = point.getX() + tx;
      final double y = point.getY() + ty;
      Point2D newPoint = new Point2D.Double(x, y);
      newPoint = adjustPoint(newPoint);
      graphPoint.setLocation(newPoint.getX(), newPoint.getY());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void doDrop(final Point2D pt) {
    final GraphPoint graphPoint = new GraphPoint(pt);
    if (canBeRemoved(graphPoint)) {
      this.pointList.remove(graphPoint);
    }
    else if (this.pointList.indexOf(graphPoint) == 0) {
      final PickNodeMessage message = new PickNodeMessage(pt);
      message.sendVO(getGraph());
      final GraphNode node = message.getNode();
      if (node != null) {
        node.startLink(this, pt);
      }
      else {
        deattach();
      }
    }
    else if (this.pointList.indexOf(graphPoint) == this.pointList.size() - 1) {
      final PickNodeMessage message = new PickNodeMessage(pt);
      message.sendVO(getGraph());
      final GraphNode node = message.getNode();
      if (node != null) {
        node.finishLink(this, pt);
      }
      else {
        deattach();
      }
    }
    for (int i = 1; i < this.pointList.size() - 1; i++) {
      final GraphPoint currentGraphPoint = this.pointList.get(i);
      this.pointList.set(i, new GraphPoint(Grid.adjustPoint(currentGraphPoint
        .getLocation())));
    }
    createShape();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void reset() {
    super.reset();
    if (this.temporaryPoint != null) {
      deattach();
    }
  }

  /**
   * Obtm o n de entrada (ou final) da ligao.
   *
   * @return o n de entrada da ligao.
   */
  public GraphNode getEndNode() {
    if (this.inputFileDescriptor != null) {
      return this.inputFileDescriptor.getNode();
    }
    return null;
  }

  /**
   * Obtm o n de sada (ou incio) da ligao.
   *
   * @return o n de sada da ligao.
   */
  public GraphNode getStartNode() {
    if (this.outputFileDescriptor != null) {
      return this.outputFileDescriptor.getNode();
    }
    return null;
  }

  /**
   * Determina se a ligao  bem formada, ou seja, se  uma ligao com todas
   * as suas propriedades vlidas.
   *
   * @return verdadeiro se a ligao  bem formada ou falso, caso contrrio.
   */
  public boolean isWellFormed() {
    if (this.outputFileDescriptor == null) {
      return false;
    }
    if (!this.outputFileDescriptor.isWellFormed()) {
      return false;
    }
    if (this.inputFileDescriptor == null) {
      return false;
    }
    if (!this.inputFileDescriptor.isWellFormed()) {
      return false;
    }

    AbstractFileParameter outputParameter =
      this.outputFileDescriptor.getFileParameter();
    AbstractFileParameter inputParameter =
      this.inputFileDescriptor.getFileParameter();

    final String[] outputFileType = outputParameter.getFileTypes();
    final String[] inputFileType = inputParameter.getFileTypes();
    if (outputFileType == null) {
      if (inputFileType != null) {
        return false;
      }
      return true;
    }
    return Arrays.equals(outputFileType, inputFileType);
  }

  /**
   * Atualiza o ponto de entrada (ou final) da ligao.
   */
  void updateEndPoint() {
    this.pointList.set(this.pointList.size() - 1, new GraphPoint(
      this.inputFileDescriptor.getLinkablePoint()));
    this.temporaryPoint = null;
    bringToFront();
    createShape();
    updateColor();
  }

  /**
   * Atualiza o ponto de sada (ou incio) da ligao.
   */
  void updateStartPoint() {
    this.pointList.set(0, new GraphPoint(this.outputFileDescriptor
      .getLinkablePoint()));
    this.temporaryPoint = null;
    bringToFront();
    createShape();
    updateColor();
  }

  /**
   * Ajusta a posio do ponto, evitando valores invlidos.
   *
   * @param point o ponto original.
   * @return o ponto ajustado.
   */
  private Point2D adjustPoint(final Point2D point) {
    final double x = Math.max(point.getX(), 0);
    final double y = Math.max(point.getY(), 0);
    final Point2D newEndPoint = new Point2D.Double(x, y);
    return newEndPoint;
  }

  /**
   * Indica se o ponto pode ser removido.
   *
   * @param point o ponto a ser removido.
   * @return verdadeiro se o ponto deve ser removido ou falso, caso contrrio.
   */
  private boolean canBeRemoved(final GraphPoint point) {
    final int indexOf = this.pointList.indexOf(point);
    if (indexOf < 0) {
      return false;
    }
    if (indexOf == 0) {
      return false;
    }
    if (indexOf == this.pointList.size() - 1) {
      return false;
    }
    final GraphPoint priorPoint = this.pointList.get(indexOf - 1);
    final GraphPoint nextPoint = this.pointList.get(indexOf + 1);
    return contains(point, priorPoint, nextPoint);
  }

  /**
   * Determina se o ponto est contido numa determinada linha.
   *
   * @param point o ponto.
   * @param startPoint o ponto de incio da linha.
   * @param endPoint o ponto de fim da linha.
   * @return verdadeiro se o ponto est contido na linha ou falso, caso
   *         contrrio.
   */
  private boolean contains(final GraphPoint point, final GraphPoint startPoint,
    final GraphPoint endPoint) {
    if (point.equals(startPoint)) {
      return true;
    }
    if (point.equals(endPoint)) {
      return true;
    }
    final Line2D line =
      new Line2D.Double(startPoint.getLocation(), endPoint.getLocation());
    return line.intersects(point.getBounds());
  }

  /**
   * Cria a representao geomtrica da linha que representa a ligao.
   */
  private void createShape() {
    if (!this.pointList.isEmpty()) {
      GraphPoint graphPoint = this.pointList.get(0);
      Point2D point = graphPoint.getLocation();
      this.path = new Path2D.Float();
      this.path.moveTo((float) point.getX(), (float) point.getY());
      for (int i = 1; i < this.pointList.size(); i++) {
        graphPoint = this.pointList.get(i);
        point = graphPoint.getLocation();
        this.path.lineTo((float) point.getX(), (float) point.getY());
      }
      if (this.temporaryPoint != null) {
        point = this.temporaryPoint.getLocation();
        this.path.lineTo((float) point.getX(), (float) point.getY());
      }
    }
  }

  /**
   * Trata a mensagem do tipo {@link PickLinkMessage}.
   *
   * @param message a mensagem.
   * @return verdadeiro se a mensagem foi tratada.
   */
  private boolean handlePickLinkMessage(final PickLinkMessage message) {
    if (contains(message.getPoint())) {
      message.setLink(this);
      return true;
    }
    return false;
  }

  /**
   * Move o ponto inicial da ligao.
   *
   * @param pt o novo ponto.
   */
  private void moveFirstPoint(final Point2D pt) {
    if (this.outputFileDescriptor != null) {
      if (!this.outputFileDescriptor.getBounds2D().contains(pt)) {
        this.outputFileDescriptor.removeLinkFrom(this);
        setOutputFileDescriptor(null);
      }
    }
  }

  /**
   * Move o ponto final da ligao.
   *
   * @param pt o novo ponto.
   */
  private void moveLastPoint(final Point2D pt) {
    if (this.inputFileDescriptor != null) {
      if (!this.inputFileDescriptor.getBounds2D().contains(pt)) {
        this.inputFileDescriptor.setLinkTo(null);
        setInputFileDescriptor(null);
      }
    }
  }

  /**
   * Notifica os observadores de que a ligao foi associada a um descritor de
   * parmetro.
   *
   * @param descriptor o descritor de parmetro.
   */
  private void notifyAnchored(final GraphFileDescriptor descriptor) {
    for (GraphElementListener listener : getListenerList()) {
      listener.wasAnchored(this, descriptor);
    }
  }

  /**
   * Notifica os observadores de que a ligao foi desassociada a um descritor
   * de parmetro.
   *
   * @param descriptor o descritor de parmetro.
   */
  private void notifyUnanchored(final GraphFileDescriptor descriptor) {
    for (GraphElementListener listener : getListenerList()) {
      listener.wasUnanchored(this, descriptor);
    }
  }

  /**
   * Atribui o descritor do arquivo de entrada.
   *
   * @param descriptor o descritor do arquivo de entrada.
   */
  private void setInputFileDescriptor(final GraphFileDescriptor descriptor) {
    if (descriptor == null) {
      if (this.inputFileDescriptor != null) {
        final GraphFileDescriptor oldDescriptor = this.inputFileDescriptor;
        final AbstractFileParameter fileParameter =
          this.inputFileDescriptor.getFileParameter();
        if (fileParameter != null) {
          fileParameter.removeFileParameterListener(this);
          fileParameter.removeSimpleParameterListener(this);
        }
        updateColor();
        notifyUnanchored(oldDescriptor);
        this.inputFileDescriptor = null;
      }
    }
    else {
      this.inputFileDescriptor = descriptor;
      updateEndPoint();
      updateColor();
      notifyAnchored(this.inputFileDescriptor);
      final AbstractFileParameter fileParameter =
        this.inputFileDescriptor.getFileParameter();
      if (fileParameter != null) {
        fileParameter.addParameterListener(this);
        fileParameter.addSimpleParameterListener(this);
      }
    }
  }

  /**
   * Atribui o descritor do arquivo de sada.
   *
   * @param descriptor o descritor do arquivo de sada.
   */
  private void setOutputFileDescriptor(final GraphFileDescriptor descriptor) {
    if (descriptor == null) {
      if (this.outputFileDescriptor != null) {
        final GraphFileDescriptor oldDescriptor = this.outputFileDescriptor;
        final AbstractFileParameter fileParameter =
          this.outputFileDescriptor.getFileParameter();
        if (fileParameter != null) {
          fileParameter.removeFileParameterListener(this);
          fileParameter.removeSimpleParameterListener(this);
        }
        updateColor();
        notifyUnanchored(oldDescriptor);
        this.outputFileDescriptor = null;
      }
    }
    else {
      this.outputFileDescriptor = descriptor;
      updateStartPoint();
      updateColor();
      final AbstractFileParameter fileParameter =
        this.outputFileDescriptor.getFileParameter();
      notifyAnchored(this.outputFileDescriptor);
      if (fileParameter != null) {
        fileParameter.addParameterListener(this);
        fileParameter.addSimpleParameterListener(this);
      }
    }
  }

  /**
   * Atualiza a cor da linha que representa a ligao.
   */
  private void updateColor() {
    if (!isWellFormed()) {
      this.color = ERROR_COLOR;
    }
    else if (this.isHighlighted) {
      this.color = HIGHT_LIGHT_COLOR;
    }
    else if (isBypassed()) {
      this.color = BYPASSED_COLOR;
    }
    else {
      this.color = DEFAULT_COLOR;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void translate(Point2D pt) {
    for (int i = 1; i < this.pointList.size() - 1; i++) {
      final GraphPoint graphPoint = this.pointList.get(i);
      Point2D location = graphPoint.getLocation();
      double newX = pt.getX() + location.getX();
      double newY = pt.getY() + location.getY();
      graphPoint.setLocation(newX, newY);
    }
  }

  /**
   * Determina se a ligao est desativada, de acordo com o estado dos ns
   * envolvidos.
   *
   * @return verdadeiro se a ligao est desativada ou falso, caso contrrio.
   */
  public boolean isBypassed() {
    GraphNode inputNode = inputFileDescriptor.getNode();
    GraphNode outputNode = outputFileDescriptor.getNode();
    if (inputNode != null && outputNode != null) {
      return inputNode.isBypassed() || outputNode.isBypassed();
    }
    else {
      return false;
    }
  }

}
